"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsLoadBalancerController = void 0;
const cdk8s = require("cdk8s");
const constructs_1 = require("constructs");
const k8s = require("./imports/k8s");
/**
 * Generate aws-load-balancer-controller config yaml.
 * see https://github.com/kubernetes-sigs/aws-aws-load-balancer-controller/blob/master/docs/install/v2_0_0_full.yaml
*/
class AwsLoadBalancerController extends constructs_1.Construct {
    constructor(scope, id, options) {
        var _a, _b, _c, _d;
        super(scope, id);
        this.serviceAccountName = (_a = options.serviceAccountName) !== null && _a !== void 0 ? _a : 'aws-load-balancer-controller';
        this.deploymentName = 'aws-load-balancer-controller';
        this.clusterName = options.clusterName;
        this.namespace = (_b = options === null || options === void 0 ? void 0 : options.namespace) !== null && _b !== void 0 ? _b : 'kube-system';
        new cdk8s.ApiObject(this, 'aws-load-balancer-controller-crd', {
            apiVersion: 'apiextensions.k8s.io/v1beta1',
            kind: 'CustomResourceDefinition',
            metadata: {
                annotations: {
                    'controller-gen.kubebuilder.io/version': 'v0.4.0',
                },
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                    ...options.labels,
                },
                name: 'targetgroupbindings.elbv2.k8s.aws',
            },
            spec: {
                additionalPrinterColumns: [
                    {
                        JSONPath: '.spec.serviceRef.name',
                        description: "The Kubernetes Service's name",
                        name: 'SERVICE-NAME',
                        type: 'string',
                    },
                    {
                        JSONPath: '.spec.serviceRef.port',
                        description: "The Kubernetes Service's port",
                        name: 'SERVICE-PORT',
                        type: 'string',
                    },
                    {
                        JSONPath: '.spec.targetType',
                        description: "The AWS TargetGroup's TargetType",
                        name: 'TARGET-TYPE',
                        type: 'string',
                    },
                    {
                        JSONPath: '.spec.targetGroupARN',
                        description: "The AWS TargetGroup's Amazon Resource Name",
                        name: 'ARN',
                        priority: 1,
                        type: 'string',
                    },
                    {
                        JSONPath: '.metadata.creationTimestamp',
                        name: 'AGE',
                        type: 'date',
                    },
                ],
                group: 'elbv2.k8s.aws',
                names: {
                    categories: [
                        'all',
                    ],
                    kind: 'TargetGroupBinding',
                    listKind: 'TargetGroupBindingList',
                    plural: 'targetgroupbindings',
                    singular: 'targetgroupbinding',
                },
                scope: 'Namespaced',
                subresources: {
                    status: {},
                },
                validation: {
                    openAPIV3Schema: {
                        description: 'TargetGroupBinding is the Schema for the TargetGroupBinding API',
                        properties: {
                            apiVersion: {
                                description: 'APIVersion defines the versioned schema of this representation of an object. Servers should convert recognized schemas to the latest internal value, and may reject unrecognized values. More info: https://git.k8s.io/community/contributors/devel/sig-architecture/api-conventions.md#resources',
                                type: 'string',
                            },
                            kind: {
                                description: 'Kind is a string value representing the REST resource this object represents. Servers may infer this from the endpoint the client submits requests to. Cannot be updated. In CamelCase. More info: https://git.k8s.io/community/contributors/devel/sig-architecture/api-conventions.md#types-kinds',
                                type: 'string',
                            },
                            metadata: {
                                type: 'object',
                            },
                            spec: {
                                description: 'TargetGroupBindingSpec defines the desired state of TargetGroupBinding',
                                properties: {
                                    networking: {
                                        description: 'networking provides the networking setup for ELBV2 LoadBalancer to access targets in TargetGroup.',
                                        properties: {
                                            ingress: {
                                                description: 'List of ingress rules to allow ELBV2 LoadBalancer to access targets in TargetGroup.',
                                                items: {
                                                    properties: {
                                                        from: {
                                                            description: 'List of peers which should be able to access the targets in TargetGroup. At least one NetworkingPeer should be specified.',
                                                            items: {
                                                                description: 'NetworkingPeer defines the source/destination peer for networking rules.',
                                                                properties: {
                                                                    ipBlock: {
                                                                        description: 'IPBlock defines an IPBlock peer. If specified, none of the other fields can be set.',
                                                                        properties: {
                                                                            cidr: {
                                                                                description: 'CIDR is the network CIDR. Both IPV4 or IPV6 CIDR are accepted.',
                                                                                type: 'string',
                                                                            },
                                                                        },
                                                                        required: [
                                                                            'cidr',
                                                                        ],
                                                                        type: 'object',
                                                                    },
                                                                    securityGroup: {
                                                                        description: 'SecurityGroup defines a SecurityGroup peer. If specified, none of the other fields can be set.',
                                                                        properties: {
                                                                            groupID: {
                                                                                description: 'GroupID is the EC2 SecurityGroupID.',
                                                                                type: 'string',
                                                                            },
                                                                        },
                                                                        required: [
                                                                            'groupID',
                                                                        ],
                                                                        type: 'object',
                                                                    },
                                                                },
                                                                type: 'object',
                                                            },
                                                            type: 'array',
                                                        },
                                                        ports: {
                                                            description: 'List of ports which should be made accessible on the targets in TargetGroup. If ports is empty or unspecified, it defaults to all ports with TCP.',
                                                            items: {
                                                                properties: {
                                                                    port: {
                                                                        'anyOf': [
                                                                            {
                                                                                type: 'integer',
                                                                            },
                                                                            {
                                                                                type: 'string',
                                                                            },
                                                                        ],
                                                                        'description': 'The port which traffic must match. When NodePort endpoints(instance TargetType) is used, this must be a numerical port. When Port endpoints(ip TargetType) is used, this can be either numerical or named port on pods. if port is unspecified, it defaults to all ports.',
                                                                        'x-kubernetes-int-or-string': true,
                                                                    },
                                                                    protocol: {
                                                                        description: 'The protocol which traffic must match. If protocol is unspecified, it defaults to TCP.',
                                                                        enum: [
                                                                            'TCP',
                                                                            'UDP',
                                                                        ],
                                                                        type: 'string',
                                                                    },
                                                                },
                                                                type: 'object',
                                                            },
                                                            type: 'array',
                                                        },
                                                    },
                                                    required: [
                                                        'from',
                                                        'ports',
                                                    ],
                                                    type: 'object',
                                                },
                                                type: 'array',
                                            },
                                        },
                                        type: 'object',
                                    },
                                    serviceRef: {
                                        description: 'serviceRef is a reference to a Kubernetes Service and ServicePort.',
                                        properties: {
                                            name: {
                                                description: 'Name is the name of the Service.',
                                                type: 'string',
                                            },
                                            port: {
                                                'anyOf': [
                                                    {
                                                        type: 'integer',
                                                    },
                                                    {
                                                        type: 'string',
                                                    },
                                                ],
                                                'description': 'Port is the port of the ServicePort.',
                                                'x-kubernetes-int-or-string': true,
                                            },
                                        },
                                        required: [
                                            'name',
                                            'port',
                                        ],
                                        type: 'object',
                                    },
                                    targetGroupARN: {
                                        description: 'targetGroupARN is the Amazon Resource Name (ARN) for the TargetGroup.',
                                        type: 'string',
                                    },
                                    targetType: {
                                        description: 'targetType is the TargetType of TargetGroup. If unspecified, it will be automatically inferred.',
                                        enum: [
                                            'instance',
                                            'ip',
                                        ],
                                        type: 'string',
                                    },
                                },
                                required: [
                                    'serviceRef',
                                    'targetGroupARN',
                                ],
                                type: 'object',
                            },
                            status: {
                                description: 'TargetGroupBindingStatus defines the observed state of TargetGroupBinding',
                                properties: {
                                    observedGeneration: {
                                        description: 'The generation observed by the TargetGroupBinding controller.',
                                        format: 'int64',
                                        type: 'integer',
                                    },
                                },
                                type: 'object',
                            },
                        },
                        type: 'object',
                    },
                },
                version: 'v1alpha1',
                versions: [
                    {
                        name: 'v1alpha1',
                        served: true,
                        storage: false,
                    },
                    {
                        name: 'v1beta1',
                        served: true,
                        storage: true,
                    },
                ],
            },
        });
        new k8s.MutatingWebhookConfiguration(this, 'aws-load-balancer-webhook', {
            metadata: {
                annotations: {
                    'cert-manager.io/inject-ca-from': 'kube-system/aws-load-balancer-serving-cert',
                },
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                    ...options.labels,
                },
                name: 'aws-load-balancer-webhook',
            },
            webhooks: [
                {
                    clientConfig: {
                        caBundle: 'Cg==',
                        service: {
                            name: 'aws-load-balancer-webhook-service',
                            namespace: this.namespace,
                            path: '/mutate-v1-pod',
                        },
                    },
                    failurePolicy: 'Fail',
                    name: 'mpod.elbv2.k8s.aws',
                    namespaceSelector: {
                        matchExpressions: [
                            {
                                key: 'elbv2.k8s.aws/pod-readiness-gate-inject',
                                operator: 'In',
                                values: ['enabled'],
                            },
                        ],
                    },
                    rules: [{
                            apiGroups: [''],
                            apiVersions: ['v1'],
                            operations: [
                                'CREATE',
                            ],
                            resources: ['pods'],
                        }],
                    sideEffects: 'None',
                },
                {
                    clientConfig: {
                        caBundle: 'Cg==',
                        service: {
                            name: 'aws-load-balancer-webhook-service',
                            namespace: this.namespace,
                            path: '/mutate-elbv2-k8s-aws-v1beta1-targetgroupbinding',
                        },
                    },
                    failurePolicy: 'Fail',
                    name: 'mtargetgroupbinding.elbv2.k8s.aws',
                    rules: [{
                            apiGroups: ['elbv2.k8s.aws'],
                            apiVersions: ['v1beta1'],
                            operations: [
                                'CREATE',
                                'UPDATE',
                            ],
                            resources: ['targetgroupbindings'],
                        }],
                    sideEffects: 'None',
                },
            ],
        });
        new k8s.ServiceAccount(this, 'aws-load-balancer-controller-sa', {
            metadata: {
                labels: {
                    'app.kubernetes.io/component': 'controller',
                    'app.kubernetes.io/name': this.serviceAccountName,
                    ...options.labels,
                },
                name: this.serviceAccountName,
                namespace: this.namespace,
            },
        });
        new k8s.Role(this, 'aws-load-balancer-controller-leader-election-role', {
            metadata: {
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                    ...options.labels,
                },
                name: 'aws-load-balancer-controller-leader-election-role',
                namespace: this.namespace,
            },
            rules: [
                {
                    apiGroups: [''],
                    resources: ['configmaps'],
                    verbs: ['create'],
                }, {
                    apiGroups: [''],
                    resources: ['configmaps'],
                    resourceNames: ['aws-load-balancer-controller-leader'],
                    verbs: ['get', 'update', 'patch'],
                },
            ],
        });
        new k8s.ClusterRole(this, 'aws-load-balancer-controller-cluster-role', {
            metadata: {
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                    ...options.labels,
                },
                name: `${this.serviceAccountName}-role`,
            },
            rules: [
                {
                    apiGroups: [''],
                    resources: ['endpoints'],
                    verbs: ['get', 'list', 'watch'],
                },
                {
                    apiGroups: [''],
                    resources: ['events'],
                    verbs: ['create', 'patch'],
                },
                {
                    apiGroups: [''],
                    resources: ['namespaces'],
                    verbs: ['get', 'list', 'watch'],
                },
                {
                    apiGroups: [''],
                    resources: ['nodes'],
                    verbs: ['get', 'list', 'watch'],
                },
                {
                    apiGroups: [''],
                    resources: ['pods'],
                    verbs: ['get', 'list', 'watch'],
                },
                {
                    apiGroups: [''],
                    resources: ['pods/status'],
                    verbs: ['patch', 'update'],
                },
                {
                    apiGroups: [''],
                    resources: ['secrets'],
                    verbs: ['get', 'list', 'watch'],
                },
                {
                    apiGroups: [''],
                    resources: ['services'],
                    verbs: ['get', 'list', 'patch', 'update', 'watch'],
                },
                {
                    apiGroups: [''],
                    resources: ['services/status'],
                    verbs: ['patch', 'update'],
                },
                {
                    apiGroups: ['elbv2.k8s.aws'],
                    resources: ['targetgroupbindings'],
                    verbs: ['create', 'delete', 'get', 'list', 'patch', 'update', 'watch'],
                },
                {
                    apiGroups: ['elbv2.k8s.aws'],
                    resources: ['targetgroupbindings/status'],
                    verbs: ['patch', 'update'],
                },
                {
                    apiGroups: ['extensions'],
                    resources: ['ingresses'],
                    verbs: ['get', 'list', 'patch', 'update', 'watch'],
                },
                {
                    apiGroups: ['extensions'],
                    resources: ['ingresses/status'],
                    verbs: ['patch', 'update'],
                },
                {
                    apiGroups: ['networking.k8s.io'],
                    resources: ['ingresses'],
                    verbs: ['get', 'list', 'patch', 'update', 'watch'],
                },
                {
                    apiGroups: ['networking.k8s.io'],
                    resources: ['ingresses/status'],
                    verbs: ['patch', 'update'],
                },
            ],
        });
        new k8s.RoleBinding(this, 'aws-load-balancer-controller-leader-election-rolebinding', {
            metadata: {
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                    ...options.labels,
                },
                name: 'aws-load-balancer-controller-leader-election-rolebinding',
                namespace: this.namespace,
            },
            roleRef: {
                apiGroup: 'rbac.authorization.k8s.io',
                kind: 'Role',
                name: 'aws-load-balancer-controller-leader-election-role',
            },
            subjects: [
                {
                    kind: 'ServiceAccount',
                    name: this.serviceAccountName,
                    namespace: this.namespace,
                },
            ],
        });
        new k8s.ClusterRoleBinding(this, 'aws-load-balancer-controller-rolebinding', {
            metadata: {
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                    ...options.labels,
                },
                name: 'aws-load-balancer-controller-rolebinding',
            },
            roleRef: {
                apiGroup: 'rbac.authorization.k8s.io',
                kind: 'ClusterRole',
                name: `${this.serviceAccountName}-role`,
            },
            subjects: [
                {
                    kind: 'ServiceAccount',
                    namespace: this.namespace,
                    name: this.serviceAccountName,
                },
            ],
        });
        new k8s.Service(this, 'aws-load-balancer-webhook-service', {
            metadata: {
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                    ...options.labels,
                },
                name: 'aws-load-balancer-webhook-service',
                namespace: this.namespace,
            },
            spec: {
                ports: [
                    {
                        port: 443,
                        targetPort: 9443,
                    },
                ],
                selector: {
                    'app.kubernetes.io/component': 'controller',
                    'app.kubernetes.io/name': this.serviceAccountName,
                },
            },
        });
        new k8s.Deployment(this, 'aws-load-balancer-controller-deployment', {
            metadata: {
                labels: {
                    'app.kubernetes.io/component': 'controller',
                    'app.kubernetes.io/name': this.serviceAccountName,
                    ...options.labels,
                },
                namespace: this.namespace,
                name: this.deploymentName,
            },
            spec: {
                replicas: (_c = options === null || options === void 0 ? void 0 : options.replicas) !== null && _c !== void 0 ? _c : 1,
                selector: {
                    matchLabels: {
                        'app.kubernetes.io/component': 'controller',
                        'app.kubernetes.io/name': this.serviceAccountName,
                        ...options.labels,
                    },
                },
                template: {
                    metadata: {
                        labels: {
                            'app.kubernetes.io/component': 'controller',
                            'app.kubernetes.io/name': this.serviceAccountName,
                            ...options.labels,
                        },
                    },
                    spec: {
                        containers: [{
                                name: 'controller',
                                image: (_d = options === null || options === void 0 ? void 0 : options.image) !== null && _d !== void 0 ? _d : 'docker.io/amazon/aws-alb-ingress-controller:v2.0.0',
                                args: this.argsFunc(options.args),
                                env: this.envFunc(options.env),
                                livenessProbe: {
                                    failureThreshold: 2,
                                    httpGet: {
                                        path: '/healthz',
                                        port: 61779,
                                        scheme: 'HTTP',
                                    },
                                    initialDelaySeconds: 30,
                                    timeoutSeconds: 10,
                                },
                                ports: [
                                    {
                                        containerPort: 9443,
                                        name: 'webhook-server',
                                        protocol: 'TCP',
                                    },
                                ],
                                resources: {
                                    limits: {
                                        cpu: '200m',
                                        memory: '500Mi',
                                    },
                                    requests: {
                                        cpu: '100m',
                                        memory: '200Mi',
                                    },
                                },
                                securityContext: {
                                    allowPrivilegeEscalation: false,
                                    readOnlyRootFilesystem: true,
                                    runAsNonRoot: true,
                                },
                                volumeMounts: [
                                    {
                                        mountPath: '/tmp/k8s-webhook-server/serving-certs',
                                        name: 'cert',
                                        readOnly: true,
                                    },
                                ],
                            }],
                        securityContext: {
                            fsGroup: 1337,
                        },
                        serviceAccountName: `${this.serviceAccountName}`,
                        terminationGracePeriodSeconds: 10,
                        volumes: [{
                                name: 'cert',
                                secret: {
                                    defaultMode: 420,
                                    secretName: 'aws-load-balancer-webhook-tls',
                                },
                            }],
                    },
                },
            },
        });
        new cdk8s.ApiObject(this, 'aws-load-balancer-serving-cert', {
            apiVersion: 'cert-manager.io/v1alpha2',
            kind: 'Certificate',
            metadata: {
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                    ...options.labels,
                },
                name: 'aws-load-balancer-serving-cert',
                namespace: this.namespace,
            },
            spec: {
                dnsNames: [
                    'aws-load-balancer-webhook-service.kube-system.svc',
                    'aws-load-balancer-webhook-service.kube-system.svc.cluster.local',
                ],
                issuerRef: {
                    kind: 'Issuer',
                    name: 'aws-load-balancer-selfsigned-issuer',
                },
                secretName: 'aws-load-balancer-webhook-tls',
            },
        });
        new cdk8s.ApiObject(this, 'aws-load-balancer-selfsigned-issuer', {
            apiVersion: 'cert-manager.io/v1alpha2',
            kind: 'Issuer',
            metadata: {
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                    ...options.labels,
                },
                name: 'aws-load-balancer-selfsigned-issuer',
                namespace: this.namespace,
            },
            spec: {
                selfSigned: {},
            },
        });
        new k8s.ValidatingWebhookConfiguration(this, 'aws-load-balancer-selfsigned-issuer-valid', {
            metadata: {
                annotations: {
                    'cert-manager.io/inject-ca-from': 'kube-system/aws-load-balancer-serving-cert',
                },
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                    ...options.labels,
                },
                name: 'aws-load-balancer-webhook',
            },
            webhooks: [
                {
                    clientConfig: {
                        caBundle: 'Cg==',
                        service: {
                            name: 'aws-load-balancer-webhook-service',
                            namespace: this.namespace,
                            path: '/validate-elbv2-k8s-aws-v1beta1-targetgroupbinding',
                        },
                    },
                    failurePolicy: 'Fail',
                    name: 'vtargetgroupbinding.elbv2.k8s.aws',
                    rules: [{
                            apiGroups: ['elbv2.k8s.aws'],
                            apiVersions: ['v1beta1'],
                            operations: [
                                'CREATE',
                                'UPDATE',
                            ],
                            resources: ['targetgroupbindings'],
                        }],
                    sideEffects: 'None',
                },
            ],
        });
    }
    argsFunc(args) {
        const defaultArgs = ['--ingress-class=alb', `--cluster-name=${this.clusterName}`];
        if (args) {
            args.forEach(e => defaultArgs.push(e));
        }
        return defaultArgs;
    }
    envFunc(envSet) {
        return envSet;
    }
}
exports.AwsLoadBalancerController = AwsLoadBalancerController;
//# sourceMappingURL=data:application/json;base64,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