"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerlessClamscan = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const path = require("path");
const aws_ec2_1 = require("monocdk/aws-ec2");
const aws_efs_1 = require("monocdk/aws-efs");
const aws_events_1 = require("monocdk/aws-events");
const aws_events_targets_1 = require("monocdk/aws-events-targets");
const aws_iam_1 = require("monocdk/aws-iam");
const aws_lambda_1 = require("monocdk/aws-lambda");
const aws_lambda_destinations_1 = require("monocdk/aws-lambda-destinations");
const aws_lambda_event_sources_1 = require("monocdk/aws-lambda-event-sources");
const aws_s3_1 = require("monocdk/aws-s3");
const aws_sqs_1 = require("monocdk/aws-sqs");
const monocdk_1 = require("monocdk");
/**
 * An [aws-cdk](https://github.com/aws/aws-cdk) construct that uses [ClamAV®](https://www.clamav.net/). to scan objects in Amazon S3 for viruses. The construct provides a flexible interface for a system to act based on the results of a ClamAV virus scan.
 *
 * The construct creates a Lambda function with EFS integration to support larger files.
 * A VPC with isolated subnets, a S3 Gateway endpoint will also be created.
 *
 * Additionally creates an twice-daily job to download the latest ClamAV definition files to the
 * Virus Definitions S3 Bucket by utilizing an EventBridge rule and a Lambda function and
 * publishes CloudWatch Metrics to the 'serverless-clamscan' namespace.
 *
 * __Important O&M__:
 * When ClamAV publishes updates to the scanner you will see “Your ClamAV installation is OUTDATED” in your scan results.
 * While the construct creates a system to keep the database definitions up to date, you must update the scanner to
 * detect all the latest Viruses.
 *
 * Update the docker images of the Lambda functions with the latest version of ClamAV by re-running `cdk deploy`.
 *
 * Successful Scan Event format
 * ```json
 * {
 *     "source": "serverless-clamscan",
 *     "input_bucket": <input_bucket_name>,
 *     "input_key": <object_key>,
 *     "status": <"CLEAN"|"INFECTED"|"N/A">,
 *     "message": <scan_summary>,
 *   }
 * ```
 *
 * Note: The Virus Definitions bucket policy will likely cause a deletion error if you choose to delete
 * the stack associated in the construct. However since the bucket itself gets deleted, you can delete
 * the stack again to resolve the error.
 *
 * @stability stable
 */
class ServerlessClamscan extends monocdk_1.Construct {
    /**
     * Creates a ServerlessClamscan construct.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param props A `ServerlessClamscanProps` interface.
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this._efsRootPath = '/lambda';
        this._efsMountPath = `/mnt${this._efsRootPath}`;
        this._efsDefsPath = 'virus_database/';
        if (!props.onResult) {
            this.resultBus = new aws_events_1.EventBus(this, 'ScanResultBus');
            this.resultDest = new aws_lambda_destinations_1.EventBridgeDestination(this.resultBus);
            this.infectedRule = new aws_events_1.Rule(this, 'InfectedRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked INFECTED',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['INFECTED'],
                        },
                    },
                },
            });
            this.cleanRule = new aws_events_1.Rule(this, 'CleanRule', {
                eventBus: this.resultBus,
                description: 'Event for when a file is marked CLEAN',
                eventPattern: {
                    detail: {
                        responsePayload: {
                            source: ['serverless-clamscan'],
                            status: ['CLEAN'],
                        },
                    },
                },
            });
        }
        else {
            this.resultDest = props.onResult;
        }
        if (!props.onError) {
            this.errorDeadLetterQueue = new aws_sqs_1.Queue(this, 'ScanErrorDeadLetterQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            });
            this.errorQueue = new aws_sqs_1.Queue(this, 'ScanErrorQueue', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
                deadLetterQueue: {
                    maxReceiveCount: 3,
                    queue: this.errorDeadLetterQueue,
                },
            });
            this.errorDest = new aws_lambda_destinations_1.SqsDestination(this.errorQueue);
            const cfnDlq = this.errorDeadLetterQueue.node.defaultChild;
            cfnDlq.addMetadata('cdk_nag', {
                rules_to_suppress: [
                    { id: 'AwsSolutions-SQS3', reason: 'This queue is a DLQ.' },
                ],
            });
        }
        else {
            this.errorDest = props.onError;
        }
        const vpc = new aws_ec2_1.Vpc(this, 'ScanVPC', {
            subnetConfiguration: [
                {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE_ISOLATED,
                    name: 'Isolated',
                },
            ],
        });
        vpc.addFlowLog('FlowLogs');
        this._s3Gw = vpc.addGatewayEndpoint('S3Endpoint', {
            service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
        });
        const fileSystem = new aws_efs_1.FileSystem(this, 'ScanFileSystem', {
            vpc: vpc,
            encrypted: props.efsEncryption === false ? false : true,
            lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_7_DAYS,
            performanceMode: aws_efs_1.PerformanceMode.GENERAL_PURPOSE,
            removalPolicy: monocdk_1.RemovalPolicy.DESTROY,
            securityGroup: new aws_ec2_1.SecurityGroup(this, 'ScanFileSystemSecurityGroup', {
                vpc: vpc,
                allowAllOutbound: false,
            }),
        });
        const lambda_ap = fileSystem.addAccessPoint('ScanLambdaAP', {
            createAcl: {
                ownerGid: '1000',
                ownerUid: '1000',
                permissions: '755',
            },
            posixUser: {
                gid: '1000',
                uid: '1000',
            },
            path: this._efsRootPath,
        });
        const logs_bucket = (_b = props.defsBucketAccessLogsConfig) === null || _b === void 0 ? void 0 : _b.logsBucket;
        const logs_bucket_prefix = (_c = props.defsBucketAccessLogsConfig) === null || _c === void 0 ? void 0 : _c.logsPrefix;
        if (logs_bucket === true || logs_bucket === undefined) {
            this.defsAccessLogsBucket = new aws_s3_1.Bucket(this, 'VirusDefsAccessLogsBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                removalPolicy: monocdk_1.RemovalPolicy.RETAIN,
                serverAccessLogsPrefix: 'access-logs-bucket-logs',
                blockPublicAccess: {
                    blockPublicAcls: true,
                    blockPublicPolicy: true,
                    ignorePublicAcls: true,
                    restrictPublicBuckets: true,
                },
            });
            this.defsAccessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:*'],
                resources: [
                    this.defsAccessLogsBucket.arnForObjects('*'),
                    this.defsAccessLogsBucket.bucketArn,
                ],
                principals: [new aws_iam_1.AnyPrincipal()],
                conditions: {
                    Bool: {
                        'aws:SecureTransport': false,
                    },
                },
            }));
        }
        else if (logs_bucket != false) {
            this.defsAccessLogsBucket = logs_bucket;
        }
        const defs_bucket = new aws_s3_1.Bucket(this, 'VirusDefsBucket', {
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            removalPolicy: monocdk_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            serverAccessLogsBucket: this.defsAccessLogsBucket,
            serverAccessLogsPrefix: logs_bucket === false ? undefined : logs_bucket_prefix,
            blockPublicAccess: {
                blockPublicAcls: true,
                blockPublicPolicy: true,
                ignorePublicAcls: true,
                restrictPublicBuckets: true,
            },
        });
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:*'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                Bool: {
                    'aws:SecureTransport': false,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
            conditions: {
                StringEquals: {
                    'aws:SourceVpce': this._s3Gw.vpcEndpointId,
                },
            },
        }));
        defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.DENY,
            actions: ['s3:PutBucketPolicy', 's3:DeleteBucketPolicy'],
            resources: [defs_bucket.bucketArn],
            notPrincipals: [new aws_iam_1.AccountRootPrincipal()],
        }));
        this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            resources: [defs_bucket.arnForObjects('*'), defs_bucket.bucketArn],
            principals: [new aws_iam_1.AnyPrincipal()],
        }));
        this._scanFunction = new aws_lambda_1.DockerImageFunction(this, 'ServerlessClamscan', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/scan'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            onSuccess: this.resultDest,
            onFailure: this.errorDest,
            filesystem: aws_lambda_1.FileSystem.fromEfsAccessPoint(lambda_ap, this._efsMountPath),
            vpc: vpc,
            vpcSubnets: { subnets: vpc.isolatedSubnets },
            allowAllOutbound: false,
            timeout: monocdk_1.Duration.minutes(15),
            memorySize: 10240,
            environment: {
                EFS_MOUNT_PATH: this._efsMountPath,
                EFS_DEF_PATH: this._efsDefsPath,
                DEFS_URL: defs_bucket.virtualHostedUrlForObject(),
                POWERTOOLS_METRICS_NAMESPACE: 'serverless-clamscan',
                POWERTOOLS_SERVICE_NAME: 'virus-scan',
            },
        });
        if (this._scanFunction.role) {
            const cfnScanRole = this._scanFunction.role.node.defaultChild;
            cfnScanRole.addMetadata('cdk_nag', {
                rules_to_suppress: [
                    {
                        id: 'AwsSolutions-IAM4',
                        reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch. The AWSLambdaVPCAccessExecutionRole is required for functions with VPC access to manage elastic network interfaces.',
                    },
                ],
            });
            const cfnScanRoleChildren = this._scanFunction.role.node.children;
            for (const child of cfnScanRoleChildren) {
                const resource = child.node.defaultChild;
                if (resource != undefined && resource.cfnResourceType == 'AWS::IAM::Policy') {
                    resource.addMetadata('cdk_nag', {
                        rules_to_suppress: [
                            {
                                id: 'AwsSolutions-IAM5',
                                reason: 'The EFS mount point permissions are controlled through a condition which limit the scope of the * resources.',
                            },
                        ],
                    });
                }
            }
        }
        this._scanFunction.connections.allowToAnyIpv4(aws_ec2_1.Port.tcp(443), 'Allow outbound HTTPS traffic for S3 access.');
        defs_bucket.grantRead(this._scanFunction);
        const download_defs = new aws_lambda_1.DockerImageFunction(this, 'DownloadDefs', {
            code: aws_lambda_1.DockerImageCode.fromImageAsset(path.join(__dirname, '../assets/lambda/code/download_defs'), {
                buildArgs: {
                    // Only force update the docker layer cache once a day
                    CACHE_DATE: new Date().toDateString(),
                },
                extraHash: Date.now().toString(),
            }),
            timeout: monocdk_1.Duration.minutes(5),
            memorySize: 1024,
            environment: {
                DEFS_BUCKET: defs_bucket.bucketName,
                POWERTOOLS_SERVICE_NAME: 'freshclam-update',
            },
        });
        const stack = monocdk_1.Stack.of(this);
        if (download_defs.role) {
            const download_defs_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${download_defs.role.roleName}/${download_defs.functionName}`;
            const download_defs_assumed_principal = new aws_iam_1.ArnPrincipal(download_defs_role);
            defs_bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:PutObject*'],
                resources: [defs_bucket.arnForObjects('*')],
                notPrincipals: [download_defs.role, download_defs_assumed_principal],
            }));
            defs_bucket.grantReadWrite(download_defs);
            const cfnDownloadRole = download_defs.role.node.defaultChild;
            cfnDownloadRole.addMetadata('cdk_nag', {
                rules_to_suppress: [
                    {
                        id: 'AwsSolutions-IAM4',
                        reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch.',
                    },
                ],
            });
            const cfnDownloadRoleChildren = download_defs.role.node.children;
            for (const child of cfnDownloadRoleChildren) {
                const resource = child.node.defaultChild;
                if (resource != undefined && resource.cfnResourceType == 'AWS::IAM::Policy') {
                    resource.addMetadata('cdk_nag', {
                        rules_to_suppress: [
                            {
                                id: 'AwsSolutions-IAM5',
                                reason: 'The function is allowed to perform operations on all prefixes in the specified bucket.',
                            },
                        ],
                    });
                }
            }
        }
        new aws_events_1.Rule(this, 'VirusDefsUpdateRule', {
            schedule: aws_events_1.Schedule.rate(monocdk_1.Duration.hours(12)),
            targets: [new aws_events_targets_1.LambdaFunction(download_defs)],
        });
        const init_defs_cr = new aws_lambda_1.Function(this, 'InitDefs', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_8,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../assets/lambda/code/initialize_defs_cr')),
            handler: 'lambda.lambda_handler',
            timeout: monocdk_1.Duration.minutes(5),
        });
        download_defs.grantInvoke(init_defs_cr);
        if (init_defs_cr.role) {
            const cfnScanRole = init_defs_cr.role.node.defaultChild;
            cfnScanRole.addMetadata('cdk_nag', {
                rules_to_suppress: [
                    {
                        id: 'AwsSolutions-IAM4',
                        reason: 'The AWSLambdaBasicExecutionRole does not provide permissions beyond uploading logs to CloudWatch.',
                    },
                ],
            });
        }
        new monocdk_1.CustomResource(this, 'InitDefsCr', {
            serviceToken: init_defs_cr.functionArn,
            properties: {
                FnName: download_defs.functionName,
            },
        });
        if (props.buckets) {
            props.buckets.forEach((bucket) => {
                this.addSourceBucket(bucket);
            });
        }
    }
    /**
     * Sets the specified S3 Bucket as a s3:ObjectCreate* for the ClamAV function.
     *
     * Grants the ClamAV function permissions to get and tag objects.
     * Adds a bucket policy to disallow GetObject operations on files that are tagged 'IN PROGRESS', 'INFECTED', or 'ERROR'.
     *
     * @param bucket The bucket to add the scanning bucket policy and s3:ObjectCreate* trigger to.
     * @stability stable
     */
    addSourceBucket(bucket) {
        this._scanFunction.addEventSource(new aws_lambda_event_sources_1.S3EventSource(bucket, { events: [aws_s3_1.EventType.OBJECT_CREATED] }));
        bucket.grantRead(this._scanFunction);
        this._scanFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
            resources: [bucket.arnForObjects('*')],
        }));
        if (this._scanFunction.role) {
            const stack = monocdk_1.Stack.of(this);
            const scan_assumed_role = `arn:${stack.partition}:sts::${stack.account}:assumed-role/${this._scanFunction.role.roleName}/${this._scanFunction.functionName}`;
            const scan_assumed_principal = new aws_iam_1.ArnPrincipal(scan_assumed_role);
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                resources: [bucket.bucketArn, bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            this._s3Gw.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:PutObjectTagging', 's3:PutObjectVersionTagging'],
                resources: [bucket.arnForObjects('*')],
                principals: [this._scanFunction.role, scan_assumed_principal],
            }));
            // Need the assumed role for the not Principal Action with Lambda
            bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.DENY,
                actions: ['s3:GetObject'],
                resources: [bucket.arnForObjects('*')],
                notPrincipals: [this._scanFunction.role, scan_assumed_principal],
                conditions: {
                    StringEquals: {
                        's3:ExistingObjectTag/scan-status': [
                            'IN PROGRESS',
                            'INFECTED',
                            'ERROR',
                        ],
                    },
                },
            }));
        }
    }
}
exports.ServerlessClamscan = ServerlessClamscan;
_a = JSII_RTTI_SYMBOL_1;
ServerlessClamscan[_a] = { fqn: "monocdk-serverless-clamscan.ServerlessClamscan", version: "1.0.0" };
//# sourceMappingURL=data:application/json;base64,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