#!/usr/bin/env python3
# Copyright 2004-present Facebook. All Rights Reserved.

from abc import abstractmethod
from time import time  # TODO: Replace with LabGraph clock
from typing import Any, Dict, List, Tuple

from ..graphs.method import AsyncPublisher, get_method_metadata
from ..graphs.node import Node, NodeMeta
from ..graphs.topic import Topic
from .event_generator import BaseEventGenerator, EventPublishingHeap
from .event_messages import WaitEndMessage


CHECK_FOR_WAIT_COMPLETION_DELAY = 0.1
ACCEPTABLE_PUBLISH_TIME_DIFF = 0.01


class BaseEventGeneratorNodeMeta(NodeMeta):
    """
    Metaclass for EventGeneratorNodes.  This metaclass is responsible
    for dynamically populating the `publish_events` function on the
    derived event generator with @publisher decorators for all topics
    defined on the class.
    """

    _PUBLISH_FUNCTION_KEY = "publish_events"

    def __init__(
        cls, name: str, bases: Tuple[type, ...], fields: Dict[str, Any]
    ) -> None:
        # Pre-process topics before NodeMeta
        topics: List[Topic] = []
        for field_value in fields.values():
            if isinstance(field_value, Topic):
                # Only subscribe to wait end topic
                if field_value.message_type is not WaitEndMessage:
                    topics.append(field_value)

        publishing_func = fields[cls._PUBLISH_FUNCTION_KEY]
        if len(topics) > 0:
            metadata = get_method_metadata(publishing_func)
            metadata.published_topics = topics

        super(BaseEventGeneratorNodeMeta, cls).__init__(name, bases, fields)


class BaseEventGeneratorNode(Node, metaclass=BaseEventGeneratorNodeMeta):
    """
    An abstract base class for an EventGeneratorNode, which publishes
    messages from its event generator based on times specified for
    each message.
    """

    def __init__(self) -> None:
        super(BaseEventGeneratorNode, self).__init__()
        self._start_time: float = time()

    def _time_elapsed_since_start(self) -> float:
        return time() - self._start_time

    def setup_generator(self, generator: BaseEventGenerator) -> None:
        """
        Saves a generator to the node.  Should be overridden to
        perform any necessary setup for the generator.
        """

        self._generator = generator

    def generate_events(self) -> EventPublishingHeap:
        """
        Returns the heap of events generated by the generator associated
        with this node.
        """

        return self._generator.generate_events()

    @abstractmethod
    async def publish_events(self) -> AsyncPublisher:
        """
        Publishes the events returned from `generate_events` based on the time
        specified for each event in the graph.
        """

        raise NotImplementedError()
