#!/usr/bin/env python3
# Copyright 2004-present Facebook. All Rights Reserved.

import time
from abc import ABC, abstractmethod
from dataclasses import dataclass
from typing import Any, Dict, Optional, Tuple, Type

from ..graphs.topic import Topic
from ..messages.message import Message, TimestampedMessage
from ..util.error import LabGraphError
from ..util.min_heap import MinHeap


class DeferredMessage:
    """
    Certain members of a Message might be dependent on when the message
    is generated - Example: `timestamp`.
    Since `Message` is immutable, we pass `DeferredMessage` to an event,
    and the actual message is constructed on access.
    """

    def __init__(self, message_type: Type[Message], *args: Any, **kwargs: Any) -> None:
        self._message_type = message_type
        self._args = args
        self._kwargs = kwargs
        self._message: Optional[Message] = None

    def build_message(self) -> Message:
        if not self._message:
            if issubclass(self._message_type, TimestampedMessage):
                # TimestampedMessage has timestamp as the first field/argument
                # - Add timestamp at the front of args to satisfy this
                self._args = (time.time(),) + self._args
            self._message = self._message_type(*self._args, **self._kwargs)
            del self._args
            del self._kwargs
        return self._message


@dataclass
class Event:
    """
    Dataclass representing an event generated by an event generator,
    where `message` should be published to `topic` at `delay` seconds
    after its predecessor.

    Args:
      _message: The DeferredMessage which should be published for this event.
                The underlying message is built when the property is accessed.
      topic: The topic this event's message should be published on.
      delay: The delay between the start of this event and the end of
             the previous event.
      duration: The amount of time this event will last for.
    """

    _message: DeferredMessage
    topic: Topic
    delay: float
    duration: float = 0.0

    def __post_init__(self) -> None:
        if self.duration < 0.0:
            raise LabGraphError("event cannot have a negative duration.")

    def __hash__(self) -> int:  # Needed for usage as dictionary key
        return hash(id(self))

    @property
    def message(self) -> Message:
        return self._message.build_message()


EventPublishingHeapEntry = Tuple[float, int, Event]
"""
An entry of an EventGeneratorNode's publishing heap. Each entry is composed
as (timestamp, id, event).
"""

EventPublishingHeap = MinHeap[EventPublishingHeapEntry]
"""
The heap used by an EventGeneratorNode to determine the order that its events
should be published in.
"""


class EventGraph:
    """
    Helper class wrapping functionality for joining events.

    Args:
        last_event_added: The most recently added event to the event heap.
        heap: The underlying MinHeap used to store events.
        _accumulated_times: Accumulated times for events inserted so far.
    """

    last_event_added: Event
    heap: EventPublishingHeap
    _accumulated_times: Dict[Event, float]

    def __init__(self, start_event: Event) -> None:
        self.heap = EventPublishingHeap()
        self._accumulated_times = {}
        self._add_start_event(start_event)

    def add_event_at_start(self, event: Event, previous_event: Event) -> None:
        """
        Adds `event` to the heap based on accumulated time of `previous_event`, where
        `event` occurs at the beginning of the latter event.
        """

        accumulated_time = self._get_accumulated_time(event, previous_event, False)
        self._push_heap_entry(event, accumulated_time)

    def add_event_at_end(self, event: Event, previous_event: Event) -> None:
        """
        Adds `event` to the heap based on accumulated time of `previous_event`, where
        `event` occurs after the duration specified for the latter.
        """

        accumulated_time = self._get_accumulated_time(event, previous_event, True)
        self._push_heap_entry(event, accumulated_time)

    def _add_start_event(self, event: Event) -> None:
        """
        Adds `event` to the heap as the first event of the graph.
        """
        if event.delay != 0.0:
            raise LabGraphError("start_event cannot have a non-zero delay.")
        self._push_heap_entry(event, 0.0)

    def _get_accumulated_time(
        self, event: Event, previous_event: Event, add_duration: bool
    ) -> float:
        accumulated_time = self._accumulated_times.get(previous_event)
        if accumulated_time is None:
            raise LabGraphError("previous_event has not been inserted yet.")
        if add_duration:
            accumulated_time += previous_event.duration
        accumulated_time += event.delay
        if accumulated_time < 0.0:
            raise LabGraphError("event occurs before start time.")
        return accumulated_time

    def _push_heap_entry(self, event: Event, accumulated_time: float) -> None:
        self._accumulated_times[event] = accumulated_time
        heap_entry: EventPublishingHeapEntry = (
            accumulated_time,
            # Use insert order for selection when same delay
            self.heap.count,
            event,
        )
        self.heap.push(heap_entry)
        self.last_event_added = event


class BaseEventGenerator(ABC):
    """
    The base event generator which should act as a parent class for future
    generators.  The generate_events function should be overwritten to create
    and return the event heap.
    """

    def __init__(self) -> None:
        self._next_publish_time: float = 0.0

    @abstractmethod
    def generate_events(self) -> EventPublishingHeap:
        """
        Generates a graph of events and returns the first event in the graph.
        """

        raise NotImplementedError()

    @abstractmethod
    def set_topics(self) -> None:
        """
        Derived classes should set its topics as instance
        attributes for access in this method.
        """

        raise NotImplementedError()
