import { JupyterFrontEnd } from '@jupyterlab/application';
import { IChangedArgs, ISettingRegistry } from '@jupyterlab/coreutils';
import { CommandRegistry } from '@phosphor/commands';
import { JSONObject } from '@phosphor/coreutils';
import { ISignal } from '@phosphor/signaling';
import { IGitExtension, Git } from './tokens';
/** Main extension class */
export declare class GitExtension implements IGitExtension {
    constructor(app?: JupyterFrontEnd, settings?: ISettingRegistry.ISettings);
    /**
     * The list of branch in the current repo
     */
    readonly branches: Git.IBranch[];
    readonly commands: CommandRegistry | null;
    /**
     * The current branch
     */
    readonly currentBranch: Git.IBranch;
    /**
     * A signal emitted when the HEAD of the git repository changes.
     */
    readonly headChanged: ISignal<IGitExtension, void>;
    /**
     * Get whether the model is disposed.
     */
    readonly isDisposed: boolean;
    /**
     * Test whether the model is ready.
     */
    readonly isReady: boolean;
    /**
     * A promise that fulfills when the model is ready.
     */
    readonly ready: Promise<void>;
    /**
     * A signal emitted when the current marking of the git repository changes.
     */
    readonly markChanged: ISignal<IGitExtension, void>;
    /**
     * Git Repository path
     *
     * This is the top-level folder fullpath.
     * null if not defined.
     */
    pathRepository: string | null;
    /**
     * A signal emitted when the current git repository changes.
     */
    readonly repositoryChanged: ISignal<IGitExtension, IChangedArgs<string | null>>;
    readonly shell: JupyterFrontEnd.IShell | null;
    /**
     * Files list resulting of a git status call.
     */
    readonly status: Git.IStatusFile[];
    /**
     * A signal emitted when the current status of the git repository changes.
     */
    readonly statusChanged: ISignal<IGitExtension, Git.IStatusFile[]>;
    /**
     * Make request to add one or all files into
     * the staging area in repository
     *
     * If filename is not provided, all files will be added.
     *
     * @param filename Optional name of the files to add
     */
    add(...filename: string[]): Promise<Response>;
    /**
     * Make request to add all unstaged files into
     * the staging area in repository 'path'
     */
    addAllUnstaged(): Promise<Response>;
    /**
     * Make request to add all untracked files into
     * the staging area in the repository
     */
    addAllUntracked(): Promise<Response>;
    /**
     * Add file named fname to current marker obj
     *
     * @param fname Filename
     * @param mark Mark to set
     */
    addMark(fname: string, mark: boolean): void;
    /**
     * get current mark of fname
     *
     * @param fname Filename
     * @returns Mark of the file
     */
    getMark(fname: string): boolean;
    /**
     * Toggle mark for file named fname in current marker obj
     *
     * @param fname Filename
     */
    toggleMark(fname: string): void;
    /**
     * Add a remote Git repository to the current repository
     *
     * @param url Remote repository URL
     * @param name Remote name
     */
    addRemote(url: string, name?: string): Promise<void>;
    /**
     * Make request for all git info of the repository
     * (This API is also implicitly used to check if the current repo is a Git repo)
     *
     * @param historyCount: Optional number of commits to get from git log
     * @returns Repository history
     */
    allHistory(historyCount?: number): Promise<Git.IAllHistory>;
    /**
     * Make request to switch current working branch,
     * create new branch if needed,
     * or discard a specific file change or all changes
     * TODO: Refactor into seperate endpoints for each kind of checkout request
     *
     * If a branch name is provided, check it out (with or without creating it)
     * If a filename is provided, check the file out
     * If nothing is provided, check all files out
     *
     * @param options Checkout options
     */
    checkout(options?: Git.ICheckoutOptions): Promise<Git.ICheckoutResult>;
    /**
     * Make request for the Git Clone API.
     *
     * @param path Local path in which the repository will be cloned
     * @param url Distant Git repository URL
     * @param auth Optional authentication information for the remote repository
     */
    clone(path: string, url: string, auth?: Git.IAuth): Promise<Git.ICloneResult>;
    /**
     * Make request to commit all staged files in repository
     *
     * @param message Commit message
     */
    commit(message: string): Promise<Response>;
    /**
     * Get or set Git configuration options
     *
     * @param options Configuration options to set (undefined to get)
     */
    config(options?: JSONObject): Promise<Response>;
    /**
     * Make request to revert changes from selected commit
     *
     * @param message Commit message to use for the new repository state
     * @param commitId Selected commit ID
     */
    revertCommit(message: string, commitId: string): Promise<Response>;
    /**
     * Make request for detailed git commit info of
     * commit 'hash'
     *
     * @param hash Commit hash
     */
    detailedLog(hash: string): Promise<Git.ISingleCommitFilePathInfo>;
    /**
     * Dispose of the resources held by the model.
     */
    dispose(): void;
    /**
     * Gets the path of the file relative to the Jupyter server root.
     *
     * If no path is provided, returns the Git repository top folder relative path.
     * If no Git repository selected, return null
     *
     * @param path the file path relative to Git repository top folder
     */
    getRelativeFilePath(path?: string): string | null;
    /**
     * Make request to initialize a  new git repository at path 'path'
     *
     * @param path Folder path to initialize as a git repository.
     */
    init(path: string): Promise<Response>;
    /**
     * Make request for git commit logs
     *
     * @param historyCount: Optional number of commits to get from git log
     */
    log(historyCount?: number): Promise<Git.ILogResult>;
    /**
     * Register a new diff provider for specified file types
     *
     * @param filetypes File type list
     * @param callback Callback to use for the provided file types
     */
    registerDiffProvider(filetypes: string[], callback: Git.IDiffCallback): void;
    /** Make request for the Git Pull API. */
    pull(auth?: Git.IAuth): Promise<Git.IPushPullResult>;
    /** Make request for the Git Push API. */
    push(auth?: Git.IAuth): Promise<Git.IPushPullResult>;
    /**
     * General Git refresh
     */
    refresh(): Promise<void>;
    /**
     * Make request for a list of all Git branches
     */
    refreshBranch(): Promise<void>;
    /**
     * Request Git status refresh
     */
    refreshStatus(): Promise<void>;
    /**
     * Make request to move one or all files from the staged to the unstaged area
     *
     * @param filename - Path to a file to be reset. Leave blank to reset all
     *
     * @returns a promise that resolves when the request is complete.
     */
    reset(filename?: string): Promise<Response>;
    /**
     * Make request to reset to selected commit
     *
     * @param commitId - Git commit specification. Leave blank to reset to HEAD
     *
     * @returns a promise that resolves when the request is complete.
     */
    resetToCommit(commitId?: string): Promise<Response>;
    /** Make request for the prefix path of a directory 'path',
     * with respect to the root directory of repository
     */
    showPrefix(path: string): Promise<Git.IShowPrefixResult>;
    /** Make request for top level path of repository 'path' */
    showTopLevel(path: string): Promise<Git.IShowTopLevelResult>;
    /**
     * Make request for a list of all git branches in the repository
     *
     * @returns The repository branches
     */
    protected _branch(): Promise<Git.IBranchResult>;
    /**
     * Set repository status
     *
     * @param v Repository status
     */
    protected _setStatus(v: Git.IStatusFile[]): void;
    private _getServerRoot;
    /**
     * set marker obj for repo path/branch combination
     */
    private _setMarker;
    private _status;
    private _pathRepository;
    private _branches;
    private _currentBranch;
    private _serverRoot;
    private _app;
    private _diffProviders;
    private _isDisposed;
    private _markerCache;
    private _currentMarker;
    private _readyPromise;
    private _pendingReadyPromise;
    private _poll;
    private _settings;
    private _headChanged;
    private _markChanged;
    private _repositoryChanged;
    private _statusChanged;
}
export declare class BranchMarker implements Git.IBranchMarker {
    private _refresh;
    constructor(_refresh: () => void);
    add(fname: string, mark?: boolean): void;
    get(fname: string): boolean;
    set(fname: string, mark: boolean): void;
    toggle(fname: string): void;
    private _marks;
}
export declare class Markers {
    private _refresh;
    constructor(_refresh: () => void);
    get(path: string, branch: string): BranchMarker;
    static markerKey(path: string, branch: string): string;
    private _branchMarkers;
}
