import * as React from 'react';
import Tabs from '@material-ui/core/Tabs';
import Tab from '@material-ui/core/Tab';
import { showErrorMessage, showDialog } from '@jupyterlab/apputils';
import { panelWrapperClass, repoButtonClass, selectedTabClass, tabClass, tabsClass, tabIndicatorClass, warningWrapperClass } from '../style/GitPanel';
import { GitAuthorForm } from '../widgets/AuthorBox';
import { FileList } from './FileList';
import { HistorySideBar } from './HistorySideBar';
import { Toolbar } from './Toolbar';
import { CommitBox } from './CommitBox';
/** A React component for the git extension's main display */
export class GitPanel extends React.Component {
    constructor(props) {
        super(props);
        this.refreshBranch = async () => {
            const { currentBranch } = this.props.model;
            this.setState({
                branches: this.props.model.branches,
                currentBranch: currentBranch ? currentBranch.name : 'master'
            });
        };
        this.refreshHistory = async () => {
            if (this.props.model.pathRepository !== null) {
                // Get git log for current branch
                let logData = await this.props.model.log(this.props.settings.composite['historyCount']);
                let pastCommits = new Array();
                if (logData.code === 0) {
                    pastCommits = logData.commits;
                }
                this.setState({
                    pastCommits: pastCommits
                });
            }
        };
        this.refreshStatus = async () => {
            await this.props.model.refreshStatus();
        };
        /**
         * Refresh widget, update all content
         */
        this.refresh = async () => {
            if (this.props.model.pathRepository !== null) {
                await this.refreshBranch();
                await this.refreshHistory();
                await this.refreshStatus();
            }
        };
        /**
         * Commits all marked files.
         *
         * @param message - commit message
         * @returns a promise which commits the files
         */
        this.commitMarkedFiles = async (message) => {
            await this.props.model.reset();
            await this.props.model.add(...this._markedFiles.map(file => file.to));
            await this.commitStagedFiles(message);
        };
        /**
         * Commits all staged files.
         *
         * @param message - commit message
         * @returns a promise which commits the files
         */
        this.commitStagedFiles = async (message) => {
            try {
                if (message &&
                    message !== '' &&
                    (await this._hasIdentity(this.props.model.pathRepository))) {
                    await this.props.model.commit(message);
                }
            }
            catch (error) {
                console.error(error);
                showErrorMessage('Fail to commit', error);
            }
        };
        /**
         * Callback invoked upon changing the active panel tab.
         *
         * @param event - event object
         * @param tab - tab number
         */
        this._onTabChange = (event, tab) => {
            if (tab === 1) {
                this.refreshHistory();
            }
            this.setState({
                tab: tab
            });
        };
        /**
         * Callback invoked upon refreshing a repository.
         *
         * @returns promise which refreshes a repository
         */
        this._onRefresh = async () => {
            await this.refreshBranch();
            if (this.state.tab === 1) {
                this.refreshHistory();
            }
            else {
                this.refreshStatus();
            }
        };
        this._previousRepoPath = null;
        this.state = {
            branches: [],
            currentBranch: '',
            files: [],
            inGitRepository: false,
            pastCommits: [],
            tab: 0
        };
    }
    componentDidMount() {
        const { model, settings } = this.props;
        model.repositoryChanged.connect((_, args) => {
            this.setState({
                inGitRepository: args.newValue !== null
            });
            this.refresh();
        }, this);
        model.statusChanged.connect(() => {
            this.setState({ files: model.status });
        }, this);
        model.headChanged.connect(async () => {
            await this.refreshBranch();
            if (this.state.tab === 1) {
                this.refreshHistory();
            }
            else {
                this.refreshStatus();
            }
        }, this);
        model.markChanged.connect(() => this.forceUpdate());
        settings.changed.connect(this.refresh, this);
    }
    /**
     * Renders the component.
     *
     * @returns React element
     */
    render() {
        return (React.createElement("div", { className: panelWrapperClass },
            this._renderToolbar(),
            this._renderMain()));
    }
    /**
     * Renders a toolbar.
     *
     * @returns React element
     */
    _renderToolbar() {
        const disableBranching = Boolean(this.props.settings.composite['disableBranchWithChanges'] &&
            (this._hasUnStagedFile() || this._hasStagedFile()));
        return (React.createElement(Toolbar, { model: this.props.model, branching: !disableBranching, refresh: this._onRefresh }));
    }
    /**
     * Renders the main panel.
     *
     * @returns React element
     */
    _renderMain() {
        if (this.state.inGitRepository) {
            return (React.createElement(React.Fragment, null,
                this._renderTabs(),
                this.state.tab === 1 ? this._renderHistory() : this._renderChanges()));
        }
        return this._renderWarning();
    }
    /**
     * Renders panel tabs.
     *
     * @returns React element
     */
    _renderTabs() {
        return (React.createElement(Tabs, { classes: {
                root: tabsClass,
                indicator: tabIndicatorClass
            }, value: this.state.tab, onChange: this._onTabChange },
            React.createElement(Tab, { classes: {
                    root: tabClass,
                    selected: selectedTabClass
                }, title: "View changed files", label: "Changes", disableFocusRipple: true, disableRipple: true }),
            React.createElement(Tab, { classes: {
                    root: tabClass,
                    selected: selectedTabClass
                }, title: "View commit history", label: "History", disableFocusRipple: true, disableRipple: true })));
    }
    /**
     * Renders a panel for viewing and committing file changes.
     *
     * @returns React element
     */
    _renderChanges() {
        return (React.createElement(React.Fragment, null,
            React.createElement(FileList, { files: this._sortedFiles, model: this.props.model, renderMime: this.props.renderMime, settings: this.props.settings }),
            this.props.settings.composite['simpleStaging'] ? (React.createElement(CommitBox, { hasFiles: this._markedFiles.length > 0, onCommit: this.commitMarkedFiles })) : (React.createElement(CommitBox, { hasFiles: this._hasStagedFile(), onCommit: this.commitStagedFiles }))));
    }
    /**
     * Renders a panel for viewing commit history.
     *
     * @returns React element
     */
    _renderHistory() {
        return (React.createElement(HistorySideBar, { branches: this.state.branches, commits: this.state.pastCommits, model: this.props.model, renderMime: this.props.renderMime }));
    }
    /**
     * Renders a panel for prompting a user to find a Git repository.
     *
     * @returns React element
     */
    _renderWarning() {
        return (React.createElement("div", { className: warningWrapperClass },
            React.createElement("div", null, "Unable to detect a Git repository."),
            React.createElement("button", { className: repoButtonClass, onClick: () => this.props.model.commands.execute('filebrowser:toggle-main') }, "Find a repository")));
    }
    /**
     * Determines whether a user has a known Git identity.
     *
     * @param path - repository path
     * @returns a promise which returns a success status
     */
    async _hasIdentity(path) {
        // If the repository path changes, check the user identity
        if (path !== this._previousRepoPath) {
            try {
                let res = await this.props.model.config();
                if (res.ok) {
                    const options = (await res.json()).options;
                    const keys = Object.keys(options);
                    // If the user name or e-mail is unknown, ask the user to set it
                    if (keys.indexOf('user.name') < 0 || keys.indexOf('user.email') < 0) {
                        const result = await showDialog({
                            title: 'Who is committing?',
                            body: new GitAuthorForm()
                        });
                        if (!result.button.accept) {
                            console.log('User refuses to set identity.');
                            return false;
                        }
                        const identity = result.value;
                        res = await this.props.model.config({
                            'user.name': identity.name,
                            'user.email': identity.email
                        });
                        if (!res.ok) {
                            console.log(await res.text());
                            return false;
                        }
                    }
                    this._previousRepoPath = path;
                }
            }
            catch (error) {
                throw new Error('Failed to set your identity. ' + error.message);
            }
        }
        return Promise.resolve(true);
    }
    _hasStagedFile() {
        return this.state.files.some(file => file.status === 'staged');
    }
    _hasUnStagedFile() {
        return this.state.files.some(file => file.status === 'unstaged');
    }
    /**
     * List of marked files.
     */
    get _markedFiles() {
        return this._sortedFiles.filter(file => this.props.model.getMark(file.to));
    }
    /**
     * List of sorted modified files.
     */
    get _sortedFiles() {
        let { files } = this.state;
        files.sort((a, b) => a.to.localeCompare(b.to));
        return files;
    }
}
//# sourceMappingURL=GitPanel.js.map