"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseAwsUrl = exports.AWS_URL_PATTERN = exports.normalizeComparisonString = exports.normalizeIdentifier = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
/**
 * Normalize string value to standard identifier form.
 *
 * Removes extraneous prefixes (Amazon, AWS), replaces all non-alphanumerics with underscore (_),
 * and converts to lower-case.
 * @param value - Value to normalize
 * @returns Returns normalized identifier string.
 */
function normalizeIdentifier(value) {
    value = value.replace(/^(Amazon|AWS)-?/i, ""); // remove prefix
    value = value.replace(/[^A-Za-z0-9]/g, "_"); // only alphanum + _
    value = value.replace(/_+/g, "_"); // de-dupe _
    return value.toLowerCase();
}
exports.normalizeIdentifier = normalizeIdentifier;
const ComparisonDictionary = {
    acl: "access_control_list",
    eni: "elastic_network_interface",
    eip: "elastic_ip_address",
};
/**
 * Normalizes string value for consistent comparison with variants between systems.
 * @param value - The value to normalize for comparison
 */
function normalizeComparisonString(value) {
    value = value.replace(/\.(png|svg)$/, "");
    value = normalizeIdentifier(value);
    Object.entries(ComparisonDictionary).forEach(([_key, _value]) => {
        _value = _value.replace(new RegExp(_key, "ig"), _value);
    });
    value = value.replace(/[_-]+(amazon|aws)[_-]+/gi, "");
    value = value.replace(/_(for|on|and|the|of|&)_/gi, "");
    value = value.replace(/v\d+/g, "");
    value = value.replace(/_/g, "");
    return value;
}
exports.normalizeComparisonString = normalizeComparisonString;
/**
 * RegEx pattern for https://aws.amazon.com based urls.
 */
exports.AWS_URL_PATTERN = /^https:\/\/(?<domain>(?:(?:docs)\.)?aws\.amazon\.com)\/(?<service>[^\/]+)(?:\/(?<resource>[^#]+))?\/?(?:#(?<dataType>.+))?$/i;
/** Parses https://aws.amazon.com based url into common definition. */
function parseAwsUrl(url) {
    url = url.replace(/\/(index\.html.*)?$/, "");
    const groups = url.match(exports.AWS_URL_PATTERN)
        ?.groups;
    if (groups) {
        const code = normalizeIdentifier(`${groups.service}_${groups.resource || ""}_${groups.dataType || ""}`);
        return {
            ...groups,
            code,
        };
    }
    throw new Error(`Failed to parse aws url: ${url}`);
}
exports.parseAwsUrl = parseAwsUrl;
//# sourceMappingURL=data:application/json;base64,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