# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aim_build']

package_data = \
{'': ['*']}

install_requires = \
['cerberus>=1.3,<2.0',
 'ninja-syntax>=1.7,<2.0',
 'tabulate>=0.8.7,<0.9.0',
 'toml>=0.10.0,<0.11.0']

entry_points = \
{'console_scripts': ['aim = aim_build.main:entry']}

setup_kwargs = {
    'name': 'aim-build',
    'version': '0.1.37',
    'description': 'A build tool for C++.',
    'long_description': '<p align="center">\n<img src="https://github.com/diwalkerdev/Assets/blob/master/Aim/aim.png" width="300" height="300">\n</p>\n\n![GitHub release (latest SemVer including pre-releases)](https://img.shields.io/github/v/release/diwalkerdev/aim?include_prereleases)\n![GitHub commits since latest release (by SemVer including pre-releases)](https://img.shields.io/github/commits-since/diwalkerdev/aim/latest/dev?include_prereleases)\n![Python package](https://github.com/diwalkerdev/Aim/workflows/Python%20package/badge.svg?branch=dev)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/aim-build)\n![PyPI - Downloads](https://img.shields.io/pypi/dm/aim-build)\n![GitHub contributors](https://img.shields.io/github/contributors/diwalkerdev/aim)\n![GitHub](https://img.shields.io/github/license/diwalkerdev/aim)\n\n\n# Aim\nA command line tool for building C++ projects. \n\n## Introduction\nAim is an attempt to make building C++ projects from source as simple as possible while encouraging a modular approach to software development.\n\nAim only requires a `target.py` file which is used to specify the builds of your project. Each build specifies a\ncomponent of your project, like a static library, dynamic library, or an executable.\n\nAim supports:\n * Windows with `msvc` frontend.\n * Linux with `gcc` frontend.\n * It should also be possible to use the `gcc` frontend on Windows when using GCC-like compilers but this hasn\'t been\ntested.\n\n## Updates\n* (23/12/2021) CLI has changed. `list`, `build`, `run` and `clobber` are now `target` commands and are executed like so: \n`aim target <path> build <name>` instead of `aim build --target=<path> <name>`. This is to make switching between \ncommands easier.\n* Aim no longer uses the `toml` for the `target` file format. `target` files are now written in Python. The motivation\nfor this change is that it can be useful to access environment variables and to store properties, such as compiler flags,\nas variables. To support this change, there is the `util/convert_toml.py` script. To convert a `toml` file, execute from \nthe aim root directory:`poetry run python util\\convert_toml.py <relative/path/to/target.toml>`. The Python file will be\nwritten to the same directory as the `target.toml` file.\n\n## Getting Started\n### Prerequisites\nAim requires the following dependencies:\n* [python](https://www.python.org/) - version 3.7 or above.\n* [ninja](https://ninja-build.org/)\n* [poetry](https://python-poetry.org/) - for development only\n\n### Installation\nAim is a `python` project and is installed using `pip`.\n\n```\npip install --user aim-build\n```\n\n### Using\n\nBasic usage:\n```\naim init --demo-files  # creates src, include, lib directory and adds demo files.\naim target builds/clang++-linux-debug list  # lists the builds in the target file.\naim target builds/clang++-linux-debug build <build-name>  # runs the build as specified by <name>.\naim target builds/clang++-linux-debug clobber  # deletes all build artifacts.\n```\nYou can run executables directly or using the `run` command:\n```\n./builds/clang++-linux-debug/<build-name>/<output-name>\naim target builds/clang++-linux-debug run <build-name> \n```\n\n<img src="https://github.com/diwalkerdev/Assets/blob/master/Aim/aim-init-demo.gif?raw=true" width="600px">\n\n## Target files\nA `target.py` file describes a project and its build components.\n\nBegin by specifying `projectRoot` which is the path from the target file to your source files. All relative paths \nwill be relative to this directory.\n\nThe compiler frontend informs Aim how to construct the arguments for the compiler. Use `gcc`\nfor GCC-like compilers and `msvc` for Microsoft cl-like compilers. Next specify the `compiler`, `archiver`, `flags` and any `defines`. \n```\nprojectRoot = "../.."\n\ncompilerFrontend="gcc"\ncompiler = "clang++"\narchiver = "ar"\n\nflags = [\n    "-std=c++17",\n    "-O3",\n    "-g",\n    "-Wall",\n]\n\n# defines = [...] # Defines do not need the -D prefix.\n```\n\nNext specify your builds. For each build you must specify the `name` and `buildRule`. Valid build rules are \n`staticLibrary`, `dynamicLibrary`, `executable`, `headerOnly` or `libraryReference`. A `target.py` that consists of a\ndynamic or shared library, an application and a test executable looks like:\n\n```\nbuilds = [\n    {\n        "name": "calculatorstatic",\n        "buildRule": "staticLibrary",\n        "outputName": "CalculatorStatic",\n        "sourceFiles": ["lib/*.cpp"],\n        "includePaths": [\n            "include"\n        ]\n    },\n    {\n        "name": "calculatordynamic",\n        "buildRule": "dynamicLibrary",\n        "outputName": "CalculatorShared",\n        "sourceFiles": ["lib/*.cpp"],\n        "includePaths": [\n            "include"\n        ]\n    },\n    {\n        "name": "calculatortests",\n        "buildRule": "executable",\n        "requires": ["calculatorstatic"],\n        "outputName": "CalculatorTests",\n        "sourceFiles": ["tests/*.cpp"],\n        "includePaths": ["include"]\n    },\n    {\n        "name": "calculatorapp",\n        "buildRule": "executable",\n        "requires": ["calculatordynamic"],\n        "outputName": "CalculatorApp",\n        "sourceFiles": ["src/*.cpp"],\n        "includePaths": ["include"]\n    }\n]\n```\n\nOther notes:\n\n* The `requires` field is important as it is how you specify the dependencies for a build. For example, if you create a static library named "myAwesomeLibrary", this can be used in other builds simply by specifying  `requires=["myAwesomeLibrary"]`. \n\n* A `headerOnly` build does not have an `outputName` or `sourceFiles` as it is not built. The `headerOnly` rule is not essential and is mostly for convenience. If you have a header only library, repeating the include paths across several builds can be become repetitive. Instead, create a `headerOnly` build to capture the include paths and use it in other builds by adding the rule to the builds `requires` field. \n\n* A `libraryReference` does not have `sourceFiles` as it is not built. Like the `headerOnly` rule it is mostly for convience to reduce duplication. The primary use case is for capturing the `includePaths`, `libraryPaths` and `libraries` of a third party library that you need to use in a build. A `libraryReference` can then be used by other builds by adding it to a builds `requires` field.\n\n* The fields `compiler`, `flags` and `defines` are normally written at the top of the target file before the builds section. By default, all builds will use these fields i.e. they are global, but they can also be overridden by specifying them again in a build. Note that when these fields are specified specifically for a build, they completely replace the global definition; any `flags` or `defines` that you specify must be written out in full as they will not share any values with the global definition.\n\n* Since target files are just python, you can have variables. However, since target files are validated with a schema, variables must be escaped with a leading underscore. For example `_custom_defines = [...]` is okay, but `custom_defines = [...]` will cause a schema error.\n\n## Supporting Multiple Targets\nAim treats any build variation as its own unique build target with its own unique `target.py`. \n\nA build target is some combination of _things_ that affects the output binary such as:\n * operating system (Windows, OSX, Gnu Linux)\n * compiler (MSVC, GCC, Clang)\n * build type (Release, Debug, Sanitized)\n * etc. \n \nEach build target and corresponding `target.py` file must have its own directory ideally named using a unique \nidentifier that comprises the \'parts\' that make up the build. For example, `builds/linux-clang++-release/target.py` indicates that the target file describes a project that is a `release` build, uses the `clang++` compiler and is for the `linux` operating system. \n\nAs an example, if you were developing an application for both Windows and Linux, you may end up with a build directory structure like the following:\n * `builds/linux-clang++-release/target.py`\n * `builds/linux-clang++-debug/target.py`\n * `builds/windows-clangcl-release/target.py`\n * `builds/windows-clangcl-debug/target.py`\n\nNote: each `target.py` file must be written out in full for each target that you need to support. There is no way for\ntarget files to share information or to depend on another. While this leads to duplication between target files, it \nmakes them very explicit and makes debugging builds much easier.\n\n## Advice Structuring Projects\nIf you structure your project/libraries as individual repositories then it may seem logical to nest dependencies inside \none another. For example, if library B depends on library A, then B needs a copy of A in order for it to be built.\nSo you may choose to nest the source of A inside B, perhaps using a git submodule.\n\nThe problem comes when your dependency hierarchy becomes more complex. If library C also depends on A, and an application D\ndepends on B and C, you\'ll end up with multiple copies of library A which can become difficult to manage.\n\nYou may need to use this approach, as it can be useful to build a library in isolation, however you should do so in such \na way where pulling the source for the dependencies is optional.\n\nThe approach the author uses is to use a non-project-specific directory that includes all your projects directly below it\ni.e. a "flat" structure. So rather than nesting dependencies you have:\n\n```\n + MyProjects\n + - LibA\n + - LibB\n + - LibC\n + - Application_1\n + - Application_2\n + - builds\n + - - App1\n + - - - linux-clang++-debug\n + - - - - target.py\n```\n\nThe flat structure has a single build directory and a single target file for each build target you need to support. This eliminates any \nduplication and is easy to manage. `Aim` is flexible enough that you can add additional levels to the project structure \nshould you need to. For example, you may want to group all libraries under a libraries sub-directory. But the take-away message \nis that you should not _enforce_ nested dependencies as this leads to duplication.\n\n\n## Developing Aim\n\nAim is a Python project and uses the [poetry](https://python-poetry.org/) dependency manager. See [poetry installation](https://python-poetry.org/docs/#installation) for instructions.\n\nOnce you have cloned the project, the virtual environment and dependencies can be installed by executing:\n\n```\npoetry install\n```\n\n### Dev Install\nUnfortunately, unlike `setuptools`, there is no means to do a \'dev install\' using poetry. A dev install effectively generates\nan application that internally references the active source files under development. This allows developers to test the application\nwithout having to re-install the application after each change.\n\nIn order to use a development version of Aim on the command line, is it recommended creating an alias. The alias needs to:\n* add the Aim directory to `PYTHONPATH` to resolve import/module paths \n* execute the main Aim script using the virtualenv created by poetry\n\nThere are `dev-env.bash` and `dev-env.fish` scripts that configure this for you in the root of the Aim project directory. \nNote, these files must be sourced in order for them to work. \n\n',
    'author': 'David Walker',
    'author_email': 'diwalkerdev@twitter.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/diwalkerdev/Aim',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
