import os
import io
import yaml
from collections import OrderedDict

from rich.console import Console

from malwarebazaar.platform import get_config_path, get_config_dir


class Config(OrderedDict):
    instance = None

    def __init__(self):
        ec = Console(stderr=True, style="bold red")
        Config.ensure_path(ec)
        config_file = get_config_path()
        if not os.path.exists(config_file) or os.path.getsize(config_file) == 0:
            ec.print("Config does not exist, please run the init command.")
            exit(-1)

        with io.open(config_file, "r") as handle:
            config_data = yaml.load(handle.read(), Loader=yaml.Loader)

        super().__init__(**config_data)

    @staticmethod
    def get_instance():
        if not Config.instance:
            return Config()
        return Config.instance

    @staticmethod
    def ensure_path(ec: Console = Console(stderr=True, style="bold red")):
        config_dir = get_config_dir()

        if not os.path.exists(config_dir):
            os.mkdir(config_dir)

        if not os.path.isdir(config_dir):
            ec.print(f"{config_dir} should be a dir, but is a file.")
            exit(-1)

    @staticmethod
    def init_config(key: str):
        Config.ensure_path()
        with io.open(get_config_path(), "w") as handle:
            bytes = handle.write(yaml.dump(
                {
                    "api_key": key,
                    "csv_columns": {
                        "md5": "md5_hash",
                        "sha1": "sha1_hash",
                        "sha256": "sha256_hash",
                        "imphash": "imphash",
                        "signature": "signature",
                        "tags": "tags"
                    }
                },
                Dumper=yaml.Dumper
            ))

        if bytes <= 0:
            raise IOError(f"Writing to config file failed.")
        return True
