import io
import shutil

import click
import requests
from rich.console import Console
from rich.progress import Progress


@click.command(help="Downloads daily malware batches. The DATE_STR argument needs to be in the format of YYYY-mm-dd.")
@click.option("-o", "--output", type=click.Path(writable=True), help="Output file name.")
@click.option("-q", "--quiet", is_flag=True, help="Do not display any output. This also uses shutil to write the "
                                                  "downloaded chunks direct to the file which *might* speed up your "
                                                  "download.")
@click.option("-c", "--chunk-size", type=int, default=1024 ** 2, help="Sets chunk size of downloaded chunks "
                                                                      f"(default is {1024 ** 2}). "
                                                                      f"This can help speed-up your downloads.")
@click.argument("date_str")
def batch(output: str, quiet: bool, chunk_size: int, date_str: str):
    ec = Console(stderr=True, style="bold red")
    url = f"https://mb-api.abuse.ch/downloads/{date_str}.zip"
    head = requests.head(url)
    if head.status_code != 200:
        ec.print(f"No batch file for given date found. "
                 f"Maybe you used the wrong date format? (Tried {head.request.url})")
        exit(-1)

    total = int(head.headers.get("Content-Length", 0))
    filename = output or f"{date_str}.zip"
    if quiet:
        with requests.get(url, stream=True) as r:
            with io.open(filename, "wb") as file:
                shutil.copyfileobj(r.raw, file, length=chunk_size)
    else:
        with Progress() as progress:
            progress.print(f"Downloading {date_str}.zip with a size of {total / 1024 ** 2:.2f}MB.")
            task = progress.add_task(f"Downloading {filename}...", total=total)
            with requests.get(url, stream=True) as r:
                with io.open(filename, "wb") as file:
                    for data_chunk in r.iter_content(chunk_size=chunk_size):
                        num_bytes = file.write(data_chunk)
                        progress.update(task, advance=num_bytes)
