# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['osrsbox',
 'osrsbox.items_api',
 'osrsbox.items_api_examples',
 'osrsbox.monsters_api',
 'osrsbox.monsters_api_examples',
 'osrsbox.prayers_api']

package_data = \
{'': ['*'], 'osrsbox': ['docs/*']}

install_requires = \
['dataclasses']

setup_kwargs = {
    'name': 'osrsbox',
    'version': '2.1.4',
    'description': 'A complete and up-to-date database of Old School Runescape (OSRS) items, monsters and prayers accessible using a Python API.',
    'long_description': '# osrsbox-db \n\n![build](https://img.shields.io/github/workflow/status/osrsbox/osrsbox-db/Build%20and%20Deploy%20to%20PyPI) ![PyPI - Python Version](https://img.shields.io/pypi/pyversions/osrsbox.svg) \n\n[![PyPI version](https://badge.fury.io/py/osrsbox.svg)](https://badge.fury.io/py/osrsbox) ![PyPI - Downloads](https://img.shields.io/pypi/dm/osrsbox.svg)\n\n[![Discord Chat](https://img.shields.io/discord/598412106118987787.svg)](https://discord.gg/HFynKyr)\n\n[![Donate](https://img.shields.io/badge/Donate-PayPal-green.svg)](https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=9J44ADGJQ5BC6&source=url)\n\n## A complete and up-to-date database of Old School Runescape (OSRS) items, monsters and prayers\n\nThis project hosts a complete and up-to-date database items, monsters and prayers in OSRS. **Complete** means it holds every single item, monster and prayer in OSRS. **Up-to-date** means this database is updated after every weekly game update to ensure accurate information.\n\nThe item database has extensive properties for each item: a total of 27 properties for every item, an additional 16 properties for equipable items, and an additional 3 properties for equipable weapons. These properties include the item ID and name, whether an item is tradeable, stackable, or equipable or if the item is members only. For any equipable item, there are additional properties about combat stats; for example, what slash attack bonus, magic defence bonus or prayer bonus that an item provides. For weapons, additional properties are added which include attack speed, combat stance and weapon type information.\n\nThe monster database also has extensive properties: a total of 44 unique properties for each monster, as well as an array of item drops for each monster that has 6 additional properties per item drop. The base properties include the monster ID, name, member status, slayer properties, attack type, max hit, attack types and all monster combat stats. Each monster also has an associated array of drops which document the item ID, name, rarity, quantity, and any requirements to get the drop.\n\nThe prayer database documents each prayer that available in-game and has detailed properties: a total of 8 properties for every prayer. The base properties include the prayer name, members status, description, requirements, and bonuses that it provides.\n\n## Table of Contents\n\n- [Project Summary](#project-summary)\n- [The `osrsbox` Python PyPi Package](#the-osrsbox-python-pypi-package)\n- [The osrsbox RESTful API](#the-osrsbox-restful-api)\n- [The osrsbox Static JSON API](#the-osrsbox-static-json-api)\n- [The `osrsbox-db` GitHub Repository](#the-osrsbox-db-github-repository)\n- [The Item Database](#the-item-database)\n- [The Monster Database](#the-monster-database)\n- [The Prayer Database](#the-prayer-database)\n- [Project Contribution](#project-contribution)\n- [Additional Project Information](#additional-project-information)\n\n## Project Summary\n\nThe osrsbox-db project provides data for three different categories:\n\n1. **Items**\n1. **Monsters**\n1. **Prayers**\n\nThe osrsbox-db project and data is accessible using four methods:\n\n1. [**The Python PyPi package**](https://pypi.org/project/osrsbox/)\n1. [**The RESTful API**](https://github.com/osrsbox/osrsbox-api/)\n1. [**The Static JSON API**](https://github.com/osrsbox/osrsbox-db/tree/master/docs)\n1. [**The GitHub development repository**](https://github.com/osrsbox/osrsbox-db/)\n\nWith four different methods to access data... most people will have the following question: _Which one should I use?_ The following list is a short-sharp summary of the options:\n\n1. [**The Python PyPi package**](https://pypi.org/project/osrsbox/): Use this if you are programming anything in Python - as it is the simplest option. Install using `pip`, and you are ready to do anything from experimenting and prototyping, to building a modern web app using something like Flask.\n1. [**The RESTful API**](https://github.com/osrsbox/osrsbox-api/): Use this if you are not programming in Python, and want an Internet-accessible API with rich-quering including filtering, sorting and projection functionality.\n1. [**The Static JSON API**](https://github.com/osrsbox/osrsbox-db/tree/master/docs): Use this if you want Internet-accessible raw data (JSON files and PNG images) and don\'t need queries to filter data. This is a good option if you want to _dump_ the entire database contents, and saves the RESTful API from un-needed traffic.\n1. [**The GitHub development repository**](https://github.com/osrsbox/osrsbox-db/): The development repository provides the code and data to build the database. I would not recommend using the development repository unless you are (really) interested in the project or you want to contribute to the project.\n\n## The `osrsbox` Python PyPi Package\n\nIf you want to access the item and monster database programmatically using Python, the simplest option is to use the [`osrsbox` package available from PyPi](https://pypi.org/project/osrsbox/). You can load the item and/or monster database and process item objects, monster objects, and their properties. \n\n### Package Quick Start\n\n- Make sure you have >= Python 3.6\n- Install package using: `pip install osrsbox`\n- Item database quick start:\n    - Import items API using: `from osrsbox import items_api`\n    - Load all items using: `items = items_api.load()`\n    - Loop items using: `for item in items: print(item.name)`\n- Monster database quick start:\n    - Import monsters API using: `from osrsbox import monsters_api`\n    - Load all monsters using: `monsters = monsters_api.load()`\n    - Loop monsters using: `for monster in monsters: print(monster.name)`\n- Prayer database quick start:\n    - Import prayers API using: `from osrsbox import prayers_api`\n    - Load all prayers using: `prayers = prayers_api.load()`\n    - Loop prayers using: `for prayer in prayers: print(prayer.name)`\n\n### Package Requirements\n\nFor the `osrsbox` PyPi package you must meet the following requirements:\n\n- Python 3.6 or above\n- Pip package manager\n- Dataclasses package (if Python is below 3.7)\n\nIf you are using Python 3.6, the `dataclasses` package will automatically be installed. If you are using Python 3.7 or above, the `dataclasses` package is part of the standard library and will not be installed automatically.\n\n### Package Installation\n\nThe easiest way to install the osrsbox package is through the [Python Package Index](http://pypi.python.org/) using the `pip` command. You need to have `pip` installed - and make sure it is updated (especially on Windows). Then you can install the `osrsbox` package using the following `pip` command:\n\n```\npip install osrsbox\n```\n\n### Package Upgrading\n\nThe package is consistently updated - usually after each weekly in-game update. This is because the in-game update usually introduces additional items into the game or changes existing items. Therefore, you should regularly check and update the `osrsbox` package. To achieve this, run `pip` with the `upgrade` flag, as demonstrated in the following command:\n\n```\npip install --upgrade osrsbox\n```\n\n### Package Usage \n\nThe key use of the `osrsbox` package is to load and automate the processing of OSRS items and their associated metadata. You can load the package using `import osrsbox`, however, you probably want to load the `items_api` module or `monsters_api` module. A simple example of using the package to `load` all the items, then loop and print out the item ID and name of every item in OSRS is provided below:\n\n```\nphoil@gilenor ~ $ python3.6\nPython 3.6.8 (default, Jan 14 2019, 11:02:34) \n[GCC 8.0.1 20180414 (experimental) [trunk revision 259383]] on linux\nType "help", "copyright", "credits" or "license" for more information.\n>>> from osrsbox import items_api\n>>> items = items_api.load()\n>>> for item in items:\n...     print(item.id, item.name)\n```\n\nInstead of using the Python interpreter, you can also write a simple script and import the `osrsbox` Python package. An example script is provided below, this time for the `monsters_api`:\n\n```\n#!/usr/bin/python3\n\nfrom osrsbox import monsters_api\n\nmonsters = monsters_api.load()\nfor monster in monsters:\n    print(monster.id, monster.name)\n```\n\nIf you would like to review additional examples of using the `osrsbox` Python API, have a look at the [`items_api_examples` folder](https://github.com/osrsbox/osrsbox-db/tree/master/osrsbox/items_api_examples) and [`monsters_api_examples` folder](https://github.com/osrsbox/osrsbox-db/tree/master/osrsbox/monsters_api_examples). There are a number of scripts available that provide examples of loading and processing data using the Python API. \n\n## The osrsbox RESTful API\n\nThe [official osrsbox-api GitHub repository](https://github.com/osrsbox/osrsbox-api) hosts the source code used for the RESTful API. The official `osrsbox-api` project is available from:\n\n- [https://api.osrsbox.com](https://api.osrsbox.com)\n\nThe link provided above has an API landing page with detailed information on the project including a project summary, API endpoints, and links to useful documentation. Also, have a look at the [official `osrsbox-api` project README](https://github.com/osrsbox/osrsbox-api/blob/master/README.md) for more information. The README has a tutorial on how to build the API docker environment locally for testing purposes which might be useful.\n\n## The `osrsbox` Static JSON API\n\nThis project also includes an Internet-accessible, static JSON API for all items/monsters in the database. The JSON API was originally written for the [`osrsbox-tooltips` project](https://github.com/osrsbox/osrsbox-tooltips) but has since been used for a variety of other projects. The JSON API is useful when you do not want to write a program in Python (as using the PyPi package is probably easier), but would like to fetch the database information programmatically over the Internet, and receive the data back in nicely structured JSON syntax. A key example is a web application. \n\n### Static JSON API Files\n\nThe JSON API is available in the [`docs` folder](https://github.com/osrsbox/osrsbox-db/tree/master/docs/) in the osrsbox-db project repository. This folder contains the publicly available database. Every file inside this specific folder can be fetched using HTTP GET requests. The base URL for this folder is `https://www.osrsbox.com/osrsbox-db/`. Simply append any name of any file from the `docs` folder to the base URL, and you can fetch this data. A summary of the folders/files provided in the JSON API are listed below with descriptions:\n\n- `items-complete.json`: A single JSON file that combines all single JSON files from `items-json` folder. This file contains the entire osrsbox-db items database in one file. This is useful if you want to get the data for every single item.\n- `items-icons`: Collection of PNG files (20K+) for every item inventory icon in OSRS. Each inventory icon is named using the unique item ID number.\n- `items-json`: Collection of JSON files (20K+) of extensive item metadata for every item in OSRS. This folder contains the entire osrsbox-db item database where each item has an individual JSON file, named using the unique item ID number. This is useful when you want to fetch data for a single item where you already know the item ID number.\n- `items-json-slot`: Collection of JSON files extracted from the database that are specific for each equipment slot (e.g., head, legs). This is useful when you want to only get item data for equipable items for one, or multiple, specific item slot.\n- `items-summary.json`: A single JSON file that contains only the item names and item ID numbers. This file is useful when you want to download a small file (1.1MB) to quickly scan/process item data when you only need the item name and/or ID number.\n- `models-summary.json`: A single JSON file that contains model ID numbers for items, objects, and NPCs. This file is useful to determine the model ID number for a specific item, object or NPC.\n- `monsters-complete.json`: A single JSON file that combines all single JSON files from the `monsters-json` folder. This file contains the entire osrsbox-db monster database in one file. This is useful if you want to get the data for every single monster in one file.\n- `monsters-json`: Collection of JSON files (2.5K+) of extensive monster metadata for every monster in OSRS. This folder contains the entire osrsbox-db monster database where each monster has an individual JSON file, named using the unique monster ID number. This is useful when you want to fetch data for a single monster where you already know the item ID number.\n- `npcs-summary.json`: A single JSON file that contains only the NPC names and NPC ID numbers. This file is useful when you want to download a small file (0.35MB) to quickly scan/process NPC data when you only need the NPC name and/or ID number. Note that this file contains both attackable, and non-attackable (monster) NPCs.\n- `objects-summary.json`: A single JSON file that contains only the object names and object ID numbers. This file is useful when you want to download a small file (0.86MB) to quickly scan/process in-game object data when you only need the object name and/or ID number.\n- `prayer-icon`: Collection of PNG files for each prayer in OSRS.\n- `prayer-json`: Collection of individual JSON files with properties and metadata about OSRS prayers.\n\n### Accessing the Static JSON API\n\nThe JSON file for each OSRS item can be directly accessed using unique URLs provide through the [`osrsbox.com`](https://www.osrsbox.com/osrsbox-db/) base URL. As mentioned, you can fetch JSON files using a unique URL, but cannot modify any JSON content. Below is a list of URL examples for items and monsters in the osrsbox-db database:\n\n- [`https://www.osrsbox.com/osrsbox-db/items-json/2.json`](https://www.osrsbox.com/osrsbox-db/items-json/2.json)\n- [`https://www.osrsbox.com/osrsbox-db/items-json/74.json`](https://www.osrsbox.com/osrsbox-db/items-json/74.json)\n- [`https://www.osrsbox.com/osrsbox-db/items-json/35.json`](https://www.osrsbox.com/osrsbox-db/items-json/35.json)\n- [`https://www.osrsbox.com/osrsbox-db/items-json/415.json`](https://www.osrsbox.com/osrsbox-db/items-json/415.json)\n- [`https://www.osrsbox.com/osrsbox-db/items-json/239.json`](https://www.osrsbox.com/osrsbox-db/items-json/239.json)\n\nAs displayed by the links above, each item or monster is stored in the `osrsbox-db` repository, under the [`items-json`](https://github.com/osrsbox/osrsbox-db/tree/master/docs/items-json) folder or [`monsters-json`](https://github.com/osrsbox/osrsbox-db/tree/master/docs/monsters-json) folder. In addition to the single JSON files for each item, many other JSON files can be fetched. Some more examples are provided below:\n\n- [`https://www.osrsbox.com/osrsbox-db/items-complete.json`](https://www.osrsbox.com/osrsbox-db/items-complete.json)\n- [`https://www.osrsbox.com/osrsbox-db/monsters-complete.json`](https://www.osrsbox.com/osrsbox-db/monsters-complete.json)\n- [`https://www.osrsbox.com/osrsbox-db/items-summary.json`](https://www.osrsbox.com/osrsbox-db/items-summary.json)\n- [`https://www.osrsbox.com/osrsbox-db/items-json-slot/items-cape.json`](https://www.osrsbox.com/osrsbox-db/items-json-slot/items-cape.json)\n- [`https://www.osrsbox.com/osrsbox-db/prayer-json/protect-from-magic.json`](https://www.osrsbox.com/osrsbox-db/prayer-json/protect-from-magic.json)\n\nSo how can you get and use these JSON files about OSRS items? It is pretty easy but depends on what you are trying to accomplish and what programming language you are using. Some examples are provided in the following subsections.\n\n### Accessing the JSON API using Command Line Tools\n\nTake a simple example of downloading a single JSON file. In a Linux system, we could use the `wget` command to download a single JSON file, as illustrated in the example code below:\n\n```\nwget https://www.osrsbox.com/osrsbox-db/items-json/12453.json\n```\n\nYou could perform a similar technique using the `curl` tool:\n\n```\ncurl https://www.osrsbox.com/osrsbox-db/items-json/12453.json\n```\n\nFor Windows users, you could use PowerShell:\n\n```\nInvoke-WebRequest -Uri "https://www.osrsbox.com/osrsbox-db/items-json/12453.json" -OutFile "12453.json"\n```\n\n### Accessing the JSON API using Python\n\nMaybe you are interested in downloading a single (or potentially multiple) JSON files about OSRS items and processing the information in a Python program. The short script below downloads the `12453.json` file using Python\'s `urllib` library, loads the data as a JSON object and prints the contents to the console. The code is a little messy, primarily due to supporting both Python 2 and 3 - as you can see from the `try` and `except` importing method implemented.\n\n```\nimport json\n\ntry:\n    from urllib.request import urlopen\nexcept ImportError:\n    from urllib2 import urlopen\n\nurl = ("https://www.osrsbox.com/osrsbox-db/items-json/12453.json")\nresponse = urlopen(url)\ndata = response.read().decode("utf-8")\njson_obj = json.loads(data)\nprint(json_obj)\n```\n\n### Accessing the JSON API using JavaScript\n\nFinally, let\'s have a look at JavaScript (specifically jQuery) example to fetch a JSON file from the osrsbox-db and build an HTML element to display in a web page. The example below is a very simple method to download the JSON file using the jQuery `getJSON` function. Once we get the JSON file, we loop through the JSON entries and print each key and value (e.g., `name` and _Black wizard hat (g)_) on its own line in a `div` element.\n\n```\n<!DOCTYPE html>\n<html>\n  <head>\n    <script src="https://ajax.googleapis.com/ajax/libs/jquery/3.1.1/jquery.min.js"></script>\n    <script>\n      $(document).ready(function(){\n          $("button").click(function(){\n              $.getJSON("https://www.osrsbox.com/osrsbox-db/items-json/12453.json", function(result){\n                  $.each(result, function(i, field){\n                      $("div").append(i + " " + field + "<br>");\n                  });\n              });\n          });\n      });\n    </script>\n  </head>\n  <body>\n    <button>Get JSON data</button>\n    <div></div>\n  </body>\n</html>\n```\n\n## The `osrsbox-db` GitHub Repository \n\nThe [official osrsbox-db GitHub repository](https://github.com/osrsbox/osrsbox-db) hosts the source code for the entire osrsbox-db project. The Python PyPi package is located in the `osrsbox` folder of the official development repository, while the other folders in this repository are used to store essential data and Python modules to build the item database. \n\n### Using the Development Repository\n\nIf using this repository (the development version), you will need to fulfill some specific requirements. This includes having the following tools available on your system:\n\n- Python 3.6 or above\n- Pip - the standard package manager for Python\n- A selection of additional Python packages\n\nAs a short example, I configured my Ubuntu 18.04 system to run the development repository code using the following steps:\n\n```\nsudo apt update\nsudo apt install python3-pip\n```\n\nThese two commands will install the `pip3` command, allowing the installation of Python packages. Then you can use `pip3` to install additional packages. The development repository requires a variety of Python packages in addition to the mandatory `dataclasses` package. These package requirements are documented in the [`requirements.txt`](https://github.com/osrsbox/osrsbox-db/tree/master/requirements.txt) file. It is recommended to use the `venv` module to set up your environment, then install the specified requirements. As an example, the following workflow is provided for Linux-based environments (make sure `python3` is available first):\n\n```\ngit clone --recursive https://github.com/osrsbox/osrsbox-db.git\ncd osrsbox-db\npython -m venv venv\nsource venv/bin/activate\npip3 install -r requirements.txt\n```\n\nWhen you have finished with working in the `osrsbox-db` repository, make sure to deactivate the current `venv` environment using:\n\n```\ndeactivate\n```\n\n### Summary of Repository Structure\n\n- `builders`: The builders are the code that performs automatic regeneration of the databases. These builders read in a variety of data and produce a JSON file for each item or monster.\n    - `items`: The item database builder that uses a collection of Python scripts to build the item database. The `builder.py` script is the primary entry point, and the `build_item.py` module does the processing of each item.\n    - `monsters`: The monster database builder that uses a collection of Python scripts to build the monster database. The `builder.py` script is the primary entry point, and the `build_monster.py` module does the processing of each monster. Additionally, the `drop_table.py` module contains a selection of hard-coded drop tables for the various OSRS Wiki drop table templates such as the rare, herb, seed, gem and catacombs drop tables.\n- `data`: Collection of useful data files used in the osrsbox-db project.\n    - `cache`: OSRS client cache dump (not present in repository due to size, but populated using the `scripts/cache` scripts).\n    - `icons`: Item and prayer icons in base64.\n    - `items`: Data used for item database generation.\n    - `monsters`: Data used for monster database generation.\n    - `schemas`: JSON schemas for the item and monster database, as well as schemas for item, npc and object definitions from cache data.\n    - `wiki`: OSRS Wiki data dump including all item and monster page titles and page data.\n- `docs`: The publicly accessible item database available through this repo or by using the static JSON API. This folder contains the actual item database that is publicly available, or browsable within this repository (see section above for more information).\n- `osrsbox`: The Python PyPi package:\n    - `items_api`: The Python API for interacting with the items database. The API has modules to load all items in the database, iterate through items, and access the different item properties.\n    - `items_api_examples`: A collection of simple Python scripts that use the `items_api` to provide an example of what can be achieved and how to use the items database.\n    - `monsters_api`: The Python API for interacting with the monster database. The API has modules to load all monsters in the database, iterate through items, and access different monster properties.\n    - `monsters_api_examples`: A collection of simple Python scripts that use the `monsters_api` to provide an example of what can be achieved and how to use the monster\'s database.\n- `scripts`: A collection of scripts (using Python and BASH) to help automate common tasks including dumping the OSRS cache, scraping the OSRS wiki, generating schemas, updating the databases, and inserting data into a MongoDB database.\n    - `cache`: A collection of scripts to extract useful data from the OSRS cache item, npc and object definition files.\n    - `icons`: Various scripts to help process, check or update item icons.\n    - `items`: A collection of scripts to help process data for the item builder.\n    - `monsters`: A collection of scripts to help process data for the monster builder.\n    - `update`: A collection of scripts for automating the data collection and database regeneration.\n    - `wiki`: A collection of scripts for automating data extraction from the OSRS Wiki using the MediaWiki API.\n- `test`: A collection of PyTest tests.\n\n### Item, Monster and Prayer Database Schemas\n\nTechnically, the `osrsbox-db` is not really a database - more specifically it should be called a data set. Anyway... the contents in the item/monster/prayer database need to adhere to a specified structure, as well as specified data types for each property. This is achieved (documented and tested) using the [Cerberus project](https://docs.python-cerberus.org/en/stable/). The Cerberus schema is useful to determine the properties that are available for each entity, and the types and requirements for each property, including:\n\n- `type`: Specifies the data type (e.g., boolean, integer, string)\n- `required`: If the property must be populated (true or false)\n- `nullable`: If the property can be set to `null` or `None`\n\nThe Cerberus schemas are provided in a dedicated repository called [`osrsbox/schemas`](https://github.com/osrsbox/schemas), and implorted into this project as a submodule - this is because the schemas are used in other repositories and central management is required. The schemas are loaded into the `data/schemas` folder and includes:\n\n1. [`schema-items.json`](https://github.com/osrsbox/schemas/blob/master/schema-items.json): This file defines the item schema, the defined properties, the property types, and some additional specifications including regex validation, and/or property type specification.\n1. [`schema-monsters.json`](https://github.com/osrsbox/schemas/blob/master/schema-monsters.json): This file defines the monster schema, the defined properties, the property types, and some additional specifications including regex validation, and/or property type specification.\n1. [`schema-prayers.json`](https://github.com/osrsbox/schemas/blob/master/schema-prayers.json): This file defines the prayer schema, the defined properties, the property types, and some additional specifications including regex validation, and/or property type specification.\n\nAll Cerberus schema files are authored using Cerberus version 1.3.2. This project uses the [`Cerberus` PyPi package](https://pypi.org/project/Cerberus/).\n\n## The Item Database\n\nEach item is represented by Python objects when using the PyPi `osrsbox` package, specifically using Python dataclass objects. Additionally, the data is accessible directly by parsing the raw JSON files. There are three types of objects, or classifications of data, that can be used to represent part of an in-game OSRS item, each outlined in the following subsections.\n\n### Item Properties\n\nAn `ItemProperties` object type includes basic item metadata such as `id`, `name`, `examine` text, store `cost`, `highalch` and `lowalch` values and `quest_item` association. Every item object in the item database has all of these properties. If you are parsing the raw JSON files all of these properties are in the root of the JSON document - so they are not nested. All of the properties available are listed in the table below including the property name, the data types used, a description of the property, if the property is required to be populated, and if the property is nullable (able to be set to `null` or `None`).\n\n| Property | Data type | Description | Required | Nullable |\n| -------- | --------- | ----------- | -------- |----------|\n| id | integer | Unique OSRS item ID number. | True | False |\n| name | string | The name of the item. | True | False |\n| last_updated | string | The last time (UTC) the item was updated (in ISO8601 date format). | True | False |\n| incomplete | boolean | If the item has incomplete wiki data. | True | False |\n| members | boolean | If the item is a members-only. | True | False |\n| tradeable | boolean | If the item is tradeable (between players and on the GE). | True | False |\n| tradeable_on_ge | boolean | If the item is tradeable (only on GE). | True | False |\n| stackable | boolean | If the item is stackable (in inventory). | True | False |\n| stacked | integer | If the item is stacked, indicated by the stack count. | True | True |\n| noted | boolean | If the item is noted. | True | False |\n| noteable | boolean | If the item is noteable. | True | False |\n| linked_id_item | integer | The linked ID of the actual item (if noted/placeholder). | True | True |\n| linked_id_noted | integer | The linked ID of an item in noted form. | True | True |\n| linked_id_placeholder | integer | The linked ID of an item in placeholder form. | True | True |\n| placeholder | boolean | If the item is a placeholder. | True | False |\n| equipable | boolean | If the item is equipable (based on right-click menu entry). | True | False |\n| equipable_by_player | boolean | If the item is equipable in-game by a player. | True | False |\n| equipable_weapon | boolean | If the item is an equipable weapon. | True | False |\n| cost | integer | The store price of an item. | True | False |\n| lowalch | integer | The low alchemy value of the item (cost * 0.4). | True | True |\n| highalch | integer | The high alchemy value of the item (cost * 0.6). | True | True |\n| weight | float | The weight (in kilograms) of the item. | True | True |\n| buy_limit | integer | The Grand Exchange buy limit of the item. | True | True |\n| quest_item | boolean | If the item is associated with a quest. | True | False |\n| release_date | string | Date the item was released (in ISO8601 format). | True | True |\n| duplicate | boolean | If the item is a duplicate. | True | False |\n| examine | string | The examine text for the item. | True | True |\n| icon | string | The item icon (in base64 encoding). | True | False |\n| wiki_name | string | The OSRS Wiki name for the item. | True | True |\n| wiki_url | string | The OSRS Wiki URL (possibly including anchor link). | True | True |\n| equipment | dict | The equipment bonuses of equipable armour/weapons. | True | True |\n| weapon | dict | The weapon bonuses including attack speed, type and stance. | True | True |\n\n### Item Equipment\n\nMany items in OSRS are equipable, this includes armor, weapons, and other _wearable_ items. Any equipable item has additional properties stored as an `ItemEquipment` object type - including attributes such as `attack_slash`, `defence_crush` and `melee_strength` values. The `ItemEquipment` object is nested within an `ItemProperties`. If you are parsing the raw JSON files, this data is nested under the `equipment` key. It is very important to note that not all items in OSRS are equipable. Only items with the `equipable_by_player` property set to `true` are equipable. The `equipable` property is similar, but this is the raw data extracted from the game cache - and can sometimes be incorrect (not equipable by a player). All of the properties available for equipable items are listed in the table below including the property name, the data types used, a description of the property, if the property is required to be populated, and if the property is nullable (able to be set to `null` or `None`).\n\n| Property | Data type | Description | Required | Nullable |\n| -------- | --------- | ----------- | -------- |----------|\n| attack_stab | integer | The attack stab bonus of the item. | True | False |\n| attack_slash | integer | The attack slash bonus of the item. | True | False |\n| attack_crush | integer | The attack crush bonus of the item. | True | False |\n| attack_magic | integer | The attack magic bonus of the item. | True | False |\n| attack_ranged | integer | The attack ranged bonus of the item. | True | False |\n| defence_stab | integer | The defence stab bonus of the item. | True | False |\n| defence_slash | integer | The defence slash bonus of the item. | True | False |\n| defence_crush | integer | The defence crush bonus of the item. | True | False |\n| defence_magic | integer | The defence magic bonus of the item. | True | False |\n| defence_ranged | integer | The defence ranged bonus of the item. | True | False |\n| melee_strength | integer | The melee strength bonus of the item. | True | False |\n| ranged_strength | integer | The ranged strength bonus of the item. | True | False |\n| magic_damage | integer | The magic damage bonus of the item. | True | False |\n| prayer | integer | The prayer bonus of the item. | True | False |\n| slot | string | The equipment slot associated with the item (e.g., head). | True | False |\n| requirements | dict | An object of requirements {skill: level}. | True | True |\n\n### Item Weapon\n\nA select number of items in OSRS are equipable weapons. Any equipable item that is a weapon has additional properties stored as an `ItemWeapon` type object including attributes such as `attack_speed` and `weapon_types` values. Additionally, each weapon has an array of combat stances associated with it to determine the `combat_style`, `attack_type`, `attack_style` and any `bonuses` or combat `experience` association. The `ItemWeapon` object is nested within an `ItemProperties` object when using the Python API. If you are parsing the raw JSON files, this data is nested under the `weapon` key.  It is very important to note that not all items in OSRS are equipable weapons. Only items with the `equipable_weapon` property set to `true` are equipable. All of the properties available for equipable weapons are listed in the table below including the property name, the data types used, a description of the property, if the property is required to be populated, and if the property is nullable (able to be set to `null` or `None`).\n\n| Property | Data type | Description | Required | Nullable |\n| -------- | --------- | ----------- | -------- |----------|\n| attack_speed | integer | The attack speed of a weapon (in game ticks). | True | False |\n| weapon_type | string | The weapon classification (e.g., axes) | True | False |\n| stances | list | An array of weapon stance information. | True | False |\n\n### Item: Python Object Example\n\nA description of the properties that each item in the database can have is useful, but sometimes it is simpler to provide an example. Below is a full example of an item as loaded in a Python object, specifically the _Abyssal whip_ item. Since this item is a type of equipment, there is an `EquipmentProperties` object nested with combat bonuses. Additionally, this item is also a weapon, so there is a `WeaponProperties` object with extra information. If the item was not equipable, the `EquipmentProperties` property would be `None` and the `equipable_by_player` would be `False`. If the item was not a weapon, the `WeaponProperties` key would be `None` and the `equipable_weapon` would be `False`.\n\n```\nItemProperties(\n    id=4151,\n    name=\'Abyssal whip\',\n    last_updated=\'2020-12-27\',\n    incomplete=False,\n    members=True,\n    tradeable=True,\n    tradeable_on_ge=True,\n    stackable=False,\n    stacked=None,\n    noted=False,\n    noteable=True,\n    linked_id_item=None,\n    linked_id_noted=4152,\n    linked_id_placeholder=14032,\n    placeholder=False,\n    equipable=True,\n    equipable_by_player=True,\n    equipable_weapon=True,\n    cost=120001,\n    lowalch=48000,\n    highalch=72000,\n    weight=0.453,\n    buy_limit=70,\n    quest_item=False,\n    release_date=\'2005-01-26\',\n    duplicate=False,\n    examine=\'A weapon from the abyss.\',\n    icon=\'iVBORw0KGgoAAAANSUhEUgAAACQAAAAgCAYAAAB6kdqOAAABvUlEQVR4Xu2Xv26DMBDG4QEyZECKIkVCKIoyVR26dOrQpUOHDn3/V3F7nL7689kGAo6z9JNuiH1wP+6PIU3zr2Jq3bRVkQ/Y7feBvfcn93o8upfDwT11XQKwOGQMwfYx9O7rcnaf52GEezuFgNdfn4JQ7RjAQojZLAAMcPJbAOH73PcloBTIQiEAG8MB7Pt6MQ+wSW1QAs6Kh1A/NgtnHyKMcZMUCP3AIBw0XcqmgU9qb4W0JwTIwuRAYHETF4FSIMkORjn1xCnjayy8lN/vLVY7TglfvBRGTJpZrpXM+my1f6DhPRdJs8M3nAPibGDseY9hVgHxzbh3chC22dkPQ8EnufddpBgI69Lk3B8hnPrwOsPEw7FYqUzoOsqBUtps8XUASh0bYbxZxUCcJZzCNnjOBGgDDBRD8d4tQAVgRAqEs2jusLOGEhaCgTQTgApLp/s5A0RBGMg3shx2YZb0fZUz9issD4VpsR4PkJ+uYbczJQr94rW7KT3yDJcegLtqeuRlAFa+0bdoeuTxPV2538Ixt1D86Vutr8IR16CSndzfoSpQLIDh09dmscL5lJICMUClw3JKD8tGXiVgfgACr1tEhnw7UAAAAABJRU5ErkJggg==\',\n    wiki_name=\'Abyssal whip\',\n    wiki_url=\'https://oldschool.runescape.wiki/w/Abyssal_whip\',\n    equipment=ItemEquipment(\n        attack_stab=0,\n        attack_slash=82,\n        attack_crush=0,\n        attack_magic=0,\n        attack_ranged=0,\n        defence_stab=0,\n        defence_slash=0,\n        defence_crush=0,\n        defence_magic=0,\n        defence_ranged=0,\n        melee_strength=82,\n        ranged_strength=0,\n        magic_damage=0,\n        prayer=0,\n        slot=\'weapon\',\n        requirements={\'attack\': 70}\n    ),\n    weapon=ItemWeapon(\n        attack_speed=4,\n        weapon_type=\'whip\',\n        stances=[\n            {\n                \'combat_style\': \'flick\',\n                \'attack_type\': \'slash\',\n                \'attack_style\': \'accurate\',\n                \'experience\': \'attack\',\n                \'boosts\': None\n            },\n            {\n                \'combat_style\': \'lash\',\n                \'attack_type\': \'slash\',\n                \'attack_style\': \'controlled\',\n                \'experience\': \'shared\',\n                \'boosts\': None},\n            {\n                \'combat_style\': \'deflect\',\n                \'attack_type\': \'slash\',\n                \'attack_style\': \'defensive\',\n                \'experience\': \'defence\',\n                \'boosts\': None\n            }\n        ]\n    )\n)\n```\n\n### Item: JSON Example\n\nA description of the properties that each item in the database can have is useful, but sometimes it is simpler to provide an example. Below is a full example of an item, specifically the _Abyssal whip_ item. Since this item is a type of equipment, there is an `equipment` key with combat bonuses. Additionally, this item is also a weapon, so there is a `weapon` key with extra information. If the item was not equipable, the `equipment` key would be `null` and the `equipable_by_player` would be `false`. If the item was not a weapon, the `weapon` key would be `null` and the `equipable_weapon` would be `false`.\n\n```\n{\n    "id": 4151,\n    "name": "Abyssal whip",\n    "last_updated": "2020-12-27",\n    "incomplete": false,\n    "members": true,\n    "tradeable": true,\n    "tradeable_on_ge": true,\n    "stackable": false,\n    "stacked": null,\n    "noted": false,\n    "noteable": true,\n    "linked_id_item": null,\n    "linked_id_noted": 4152,\n    "linked_id_placeholder": 14032,\n    "placeholder": false,\n    "equipable": true,\n    "equipable_by_player": true,\n    "equipable_weapon": true,\n    "cost": 120001,\n    "lowalch": 48000,\n    "highalch": 72000,\n    "weight": 0.453,\n    "buy_limit": 70,\n    "quest_item": false,\n    "release_date": "2005-01-26",\n    "duplicate": false,\n    "examine": "A weapon from the abyss.",\n    "icon": "iVBORw0KGgoAAAANSUhEUgAAACQAAAAgCAYAAAB6kdqOAAABvUlEQVR4Xu2Xv26DMBDG4QEyZECKIkVCKIoyVR26dOrQpUOHDn3/V3F7nL7689kGAo6z9JNuiH1wP+6PIU3zr2Jq3bRVkQ/Y7feBvfcn93o8upfDwT11XQKwOGQMwfYx9O7rcnaf52GEezuFgNdfn4JQ7RjAQojZLAAMcPJbAOH73PcloBTIQiEAG8MB7Pt6MQ+wSW1QAs6Kh1A/NgtnHyKMcZMUCP3AIBw0XcqmgU9qb4W0JwTIwuRAYHETF4FSIMkORjn1xCnjayy8lN/vLVY7TglfvBRGTJpZrpXM+my1f6DhPRdJs8M3nAPibGDseY9hVgHxzbh3chC22dkPQ8EnufddpBgI69Lk3B8hnPrwOsPEw7FYqUzoOsqBUtps8XUASh0bYbxZxUCcJZzCNnjOBGgDDBRD8d4tQAVgRAqEs2jusLOGEhaCgTQTgApLp/s5A0RBGMg3shx2YZb0fZUz9issD4VpsR4PkJ+uYbczJQr94rW7KT3yDJcegLtqeuRlAFa+0bdoeuTxPV2538Ixt1D86Vutr8IR16CSndzfoSpQLIDh09dmscL5lJICMUClw3JKD8tGXiVgfgACr1tEhnw7UAAAAABJRU5ErkJggg==",\n    "wiki_name": "Abyssal whip",\n    "wiki_url": "https://oldschool.runescape.wiki/w/Abyssal_whip",\n    "equipment": {\n        "attack_stab": 0,\n        "attack_slash": 82,\n        "attack_crush": 0,\n        "attack_magic": 0,\n        "attack_ranged": 0,\n        "defence_stab": 0,\n        "defence_slash": 0,\n        "defence_crush": 0,\n        "defence_magic": 0,\n        "defence_ranged": 0,\n        "melee_strength": 82,\n        "ranged_strength": 0,\n        "magic_damage": 0,\n        "prayer": 0,\n        "slot": "weapon",\n        "requirements": {\n            "attack": 70\n        }\n    },\n    "weapon": {\n        "attack_speed": 4,\n        "weapon_type": "whip",\n        "stances": [\n            {\n                "combat_style": "flick",\n                "attack_type": "slash",\n                "attack_style": "accurate",\n                "experience": "attack",\n                "boosts": null\n            },\n            {\n                "combat_style": "lash",\n                "attack_type": "slash",\n                "attack_style": "controlled",\n                "experience": "shared",\n                "boosts": null\n            },\n            {\n                "combat_style": "deflect",\n                "attack_type": "slash",\n                "attack_style": "defensive",\n                "experience": "defence",\n                "boosts": null\n            }\n        ]\n    }\n}\n```\n\n## The Monster Database\n\nEach monster is represented by Python objects when using the PyPi `osrsbox` package, specifically using Python dataclass objects. Additionally, the data is accessible directly by parsing the raw JSON files. There are two types of objects, or classifications of data, that can be used to represent part of an in-game OSRS monster, each outlined in the following subsections. \n\n### Monster Properties\n\nA `MonsterProperties` object type includes basic monster metadata such as `id`, `name`, `examine` text, `combat_level`, `attack_speed` and `hitpoints` values and slayer association such as `slayer_masters` who give this monster as a task. Every monster object in the monster database has all of these properties. If you are parsing the raw JSON files all of these properties are in the root of the JSON document - so they are not nested. All of the properties available are listed in the table below including the property name, the data types used, a description of the property, if the property is required to be populated, and if the property is nullable (able to be set to `null` or `None`).\n\n| Property | Data type | Description | Required | Nullable |\n| -------- | --------- | ----------- | -------- |----------|\n| id | integer | Unique OSRS monster ID number. | True | False |\n| name | string | The name of the monster. | True | False |\n| last_updated | string | The last time (UTC) the monster was updated (in ISO8601 date format). | True | True |\n| incomplete | boolean | If the monster has incomplete wiki data. | True | False |\n| members | boolean | If the monster is members only, or not. | True | False |\n| release_date | string | The release date of the monster (in ISO8601 date format). | True | True |\n| combat_level | integer | The combat level of the monster. | True | False |\n| size | integer | The size, in tiles, of the monster. | True | False |\n| hitpoints | integer | The number of hitpoints a monster has. | True | True |\n| max_hit | integer | The maximum hit of the monster. | True | True |\n| attack_type | list | The attack style (e.g., melee, magic, range) of the monster. | True | False |\n| attack_speed | integer | The attack speed (in game ticks) of the monster. | True | True |\n| aggressive | boolean | If the monster is aggressive, or not. | True | False |\n| poisonous | boolean | If the monster poisons, or not | True | False |\n| venomous | boolean | If the monster poisons using venom, or not | True | False |\n| immune_poison | boolean | If the monster is immune to poison, or not | True | False |\n| immune_venom | boolean | If the monster is immune to venom, or not | True | False |\n| attributes | list | An array of monster attributes. | True | False |\n| category | list | An array of monster category. | True | False |\n| slayer_monster | boolean | If the monster is a potential slayer task. | True | False |\n| slayer_level | integer | The slayer level required to kill the monster. | True | True |\n| slayer_xp | float | The slayer XP rewarded for a monster kill. | True | True |\n| slayer_masters | list | The slayer masters who can assign the monster. | True | False |\n| duplicate | boolean | If the monster is a duplicate. | True | False |\n| examine | string | The examine text of the monster. | True | False |\n| wiki_name | string | The OSRS Wiki name for the monster. | True | False |\n| wiki_url | string | The OSRS Wiki URL (possibly including anchor link). | True | False |\n| attack_level | integer | The attack level of the monster. | True | False |\n| strength_level | integer | The strength level of the monster. | True | False |\n| defence_level | integer | The defence level of the monster. | True | False |\n| magic_level | integer | The magic level of the monster. | True | False |\n| ranged_level | integer | The ranged level of the monster. | True | False |\n| attack_bonus | integer | The attack bonus of the monster. | True | False |\n| strength_bonus | integer | The strength bonus of the monster. | True | False |\n| attack_magic | integer | The magic attack of the monster. | True | False |\n| magic_bonus | integer | The magic bonus of the monster. | True | False |\n| attack_ranged | integer | The ranged attack of the monster. | True | False |\n| ranged_bonus | integer | The ranged bonus of the monster. | True | False |\n| defence_stab | integer | The defence stab bonus of the monster. | True | False |\n| defence_slash | integer | The defence slash bonus of the monster. | True | False |\n| defence_crush | integer | The defence crush bonus of the monster. | True | False |\n| defence_magic | integer | The defence magic bonus of the monster. | True | False |\n| defence_ranged | integer | The defence ranged bonus of the monster. | True | False |\n| drops | list | An array of monster drop objects. | True | False |\n\n### Monster Drops\n\nMost monsters in OSRS drop items when they have been defeated (killed). All monster drops are stored in the `drops` property in an array containing properties about the item drop. When using the PyPi `osrsbox` package, these drops are represented by a list of `MonsterDrops` object type. When parsing the raw JSON files, the drops are stored in an array, that are nested under the `drops` key. The data included with the monster drops are the item `id`, item `name`, the drop `rarity`, whether the drop is `noted` and any `drop_requirements`. All of the properties available for item drops are listed in the table below including the property name, the data types used, a description of the property, if the property is required to be populated, and if the property is nullable (able to be set to `null` or `None`).\n\n| Property | Data type | Description | Required | Nullable |\n| -------- | --------- | ----------- | -------- |----------|\n| id | integer | The ID number of the item drop. | True | False |\n| name | string | The name of the item drop. | True | False |\n| members | boolean | If the drop is a members-only item. | True | False |\n| quantity | string | The quantity of the item drop (integer, comma-separated or range). | True | True |\n| noted | boolean | If the item drop is noted, or not. | True | False |\n| rarity | float | The rarity of the item drop (as a float out of 1.0). | True | False |\n| rolls | integer | Number of rolls from the drop. | True | False |\n\n### Monster: Python Object Example \n\nA description of the properties that each monster in the database can have is useful, but sometimes it is simpler to provide an example. Below is a full example of a monster, specifically the _Abyssal demon_ monster. Please note that the number of item `drops` key data has been reduced to make the data more readable.\n\n```\nMonsterProperties(\n    id=415,\n    name=\'Abyssal demon\',\n    last_updated=\'2020-12-25\',\n    incomplete=False,\n    members=True,\n    release_date=\'2005-01-26\',\n    combat_level=124,\n    size=1,\n    hitpoints=150,\n    max_hit=8,\n    attack_type=[\'stab\'],\n    attack_speed=4,\n    aggressive=False,\n    poisonous=False,\n    venomous=False,\n    immune_poison=False,\n    immune_venom=False,\n    attributes=[\'demon\'],\n    category=[\'abyssal demon\'],\n    slayer_monster=True,\n    slayer_level=85,\n    slayer_xp=150.0,\n    slayer_masters=[\n        \'vannaka\',\n        \'chaeldar\',\n        \'konar\',\n        \'nieve\',\n        \'duradel\'\n    ],\n    duplicate=False,\n    examine=\'A denizen of the Abyss!\',\n    wiki_name=\'Abyssal demon (Standard)\',\n    wiki_url=\'https://oldschool.runescape.wiki/w/Abyssal_demon#Standard\',\n    attack_level=97,\n    strength_level=67,\n    defence_level=135,\n    magic_level=1,\n    ranged_level=1,\n    attack_bonus=0,\n    strength_bonus=0,\n    attack_magic=0,\n    magic_bonus=0,\n    attack_ranged=0,\n    ranged_bonus=0,\n    defence_stab=20,\n    defence_slash=20,\n    defence_crush=20,\n    defence_magic=0,\n    defence_ranged=20,\n    drops=\n    [\n        MonsterDrop(\n            id=592,\n            name=\'Ashes\',\n            members=False,\n            quantity=\'1\',\n            noted=False,\n            rarity=1.0,\n            rolls=1\n        ),\n        ...\n        MonsterDrop(\n            id=4151,\n            name=\'Abyssal whip\',\n            members=True,\n            quantity=\'1\',\n            noted=False,\n            rarity=0.001953125,\n            rolls=1\n        )\n    ]\n)\n```\n\n### Monster: JSON Example\n\nA description of the properties that each monster in the database can have is useful, but sometimes it is simpler to provide an example. Below is a full example of a monster, specifically the _Abyssal demon_ monster. Please note that the number of item `drops` key data has been reduced to make the data more readable.\n\n```\n{\n    "id": 415,\n    "name": "Abyssal demon",\n    "last_updated": "2020-12-25",\n    "incomplete": false,\n    "members": true,\n    "release_date": "2005-01-26",\n    "combat_level": 124,\n    "size": 1,\n    "hitpoints": 150,\n    "max_hit": 8,\n    "attack_type": [\n        "stab"\n    ],\n    "attack_speed": 4,\n    "aggressive": false,\n    "poisonous": false,\n    "venomous": false,\n    "immune_poison": false,\n    "immune_venom": false,\n    "attributes": [\n        "demon"\n    ],\n    "category": [\n        "abyssal demon"\n    ],\n    "slayer_monster": true,\n    "slayer_level": 85,\n    "slayer_xp": 150.0,\n    "slayer_masters": [\n        "vannaka",\n        "chaeldar",\n        "konar",\n        "nieve",\n        "duradel"\n    ],\n    "duplicate": false,\n    "examine": "A denizen of the Abyss!",\n    "wiki_name": "Abyssal demon (Standard)",\n    "wiki_url": "https://oldschool.runescape.wiki/w/Abyssal_demon#Standard",\n    "attack_level": 97,\n    "strength_level": 67,\n    "defence_level": 135,\n    "magic_level": 1,\n    "ranged_level": 1,\n    "attack_bonus": 0,\n    "strength_bonus": 0,\n    "attack_magic": 0,\n    "magic_bonus": 0,\n    "attack_ranged": 0,\n    "ranged_bonus": 0,\n    "defence_stab": 20,\n    "defence_slash": 20,\n    "defence_crush": 20,\n    "defence_magic": 0,\n    "defence_ranged": 20,\n    "drops": [\n        {\n            "id": 1623,\n            "name": "Uncut sapphire",\n            "members": true,\n            "quantity": "1",\n            "noted": false,\n            "rarity": 0.009765625,\n            "rolls": 1\n        },\n        ...\n        {\n            "id": 4151,\n            "name": "Abyssal whip",\n            "members": true,\n            "quantity": "1",\n            "noted": false,\n            "rarity": 0.001953125,\n            "rolls": 1\n        }\n    ]\n}\n```\n\n## The Prayer Database\n\nEach prayer is represented by Python objects when using the PyPi `osrsbox` package, specifically using Python dataclass objects Additionally, the data is accessible directly by parsing the raw JSON files. All prayer data is stored in a single object to represent the properties of an in-game OSRS prayer, which is outlined in the following subsection. \n\n### Prayer Properties\n\nA `PrayerProperties` object type includes basic prayer metadata such as `id`, `name`, `description` text, `drain_per_minute`, `requirements` and `bonuses` values. Every prayer object in the prayer database has all of these properties. If you are parsing the raw JSON files all of these properties are in the root of the JSON document - so they are not nested. All of the properties available are listed in the table below including the property name, the data types used, a description of the property, if the property is required to be populated, and if the property is nullable (able to be set to `null` or `None`).\n\n| Property | Data type | Description | Required | Nullable |\n| -------- | --------- | ----------- | -------- |----------|\n| id | integer | Unique prayer ID number. | True | False |\n| name | string | The name of the prayer. | True | False |\n| members | boolean | If the prayer is members-only. | True | False |\n| description | string | The prayer description (as show in-game). | True | False |\n| drain_per_minute | float | The prayer point drain rate per minute. | True | False |\n| wiki_url | string | The OSRS Wiki URL. | True | False |\n| requirements | dict | The stat requirements to use the prayer. | True | False |\n| bonuses | dict | The bonuses a prayer provides. | True | False |\n| icon | string | The prayer icon. | True | False |\n\n### Prayer: Python Object Example \n\nA description of the properties that each prayer in the database can have is useful, but sometimes it is simpler to provide an example. Below is a full example of a prayer, specifically the _Rigour_ prayer, as loaded in the `osrsbox` PyPi Python package.\n\n```\nPrayerProperties(\n    id=28,\n    name=\'Rigour\',\n    members=True,\n    description=\'Increases your Ranged attack by 20% and damage by 23%, and your defence by 25%.\',\n    drain_per_minute=40.0,\n    wiki_url=\'https://oldschool.runescape.wiki/w/Rigour\',\n    requirements={\'prayer\': 74, \'defence\': 70},\n    bonuses={\'ranged\': 20, \'ranged_strength\': 25, \'defence\': 23}, \n    icon=\'iVBORw0KGgoAAAANSUhEUgAAABwAAAAYCAYAAADpnJ2CAAABMklEQVR42rWW3Q0CIRCEjwJ8tgBrMPHZFmzAIny0gOvA+izAGjCYcMwNswucSrLJHT/7McvyM02bSojTfwo7Tv8h3h/PqKFfTSTEYqUuwTRQ9R8Erp0XdV79ANBWw7Y/7Mw2W7mhqDSGxTkC0vf5eqqg2I99CKAOVXZ0mY+Lw/SdgFjHk7DD3xE+hLZsINR2+BQMlXG7Gu8Cc2jyt4KketWWw22HWYTUWqcMsYzVcmIBMFQZqBSxmvl1+xj2Z7XdQByQHbKSDET1qNCB1uuHs1ZhY2NgQ2W9fpbCEaAT0jpLeZAKaQvmJI3OUs5QhHoZqoDuWcr7jDe4lURqL3bcIOsTxwJbxmvdcV0FeQPgPmydpZ3KJvcg904bVNi+GwegCPYgG2D1g6nXfvCu4cdRy9rlDXGzl98mKbMMAAAAAElFTkSuQmCC\'\n)\n```\n\n### Prayer: JSON Example\n\nA description of the properties that each prayer in the database can have is useful, but sometimes it is simpler to provide an example. Below is a full example of a prayer, specifically the _Rigour_ prayer, as a JSON object.\n\n```\n    "id": 28,\n    "name": "Rigour",\n    "members": true,\n    "description": "Increases your Ranged attack by 20% and damage by 23%, and your defence by 25%.",\n    "drain_per_minute": 40.0,\n    "wiki_url": "https://oldschool.runescape.wiki/w/Rigour",\n    "requirements": {\n        "prayer": 74,\n        "defence": 70\n    },\n    "bonuses": {\n        "ranged": 20,\n        "ranged_strength": 25,\n        "defence": 23\n    },\n    "icon": "iVBORw0KGgoAAAANSUhEUgAAABwAAAAYCAYAAADpnJ2CAAABMklEQVR42rWW3Q0CIRCEjwJ8tgBrMPHZFmzAIny0gOvA+izAGjCYcMwNswucSrLJHT/7McvyM02bSojTfwo7Tv8h3h/PqKFfTSTEYqUuwTRQ9R8Erp0XdV79ANBWw7Y/7Mw2W7mhqDSGxTkC0vf5eqqg2I99CKAOVXZ0mY+Lw/SdgFjHk7DD3xE+hLZsINR2+BQMlXG7Gu8Cc2jyt4KketWWw22HWYTUWqcMsYzVcmIBMFQZqBSxmvl1+xj2Z7XdQByQHbKSDET1qNCB1uuHs1ZhY2NgQ2W9fpbCEaAT0jpLeZAKaQvmJI3OUs5QhHoZqoDuWcr7jDe4lURqL3bcIOsTxwJbxmvdcV0FeQPgPmydpZ3KJvcg904bVNi+GwegCPYgG2D1g6nXfvCu4cdRy9rlDXGzl98mKbMMAAAAAElFTkSuQmCC"\n}\n```\n\n## Project Contribution\n\nThis project would thoroughly benefit from contributions from additional developers. Please feel free to submit a pull request if you have code that you wish to contribute - I would thoroughly appreciate the helping hand. For any code contributions, the best method is to [open a new GitHub pull request](https://github.com/osrsbox/osrsbox-db/pulls) in the project repository. Also, feel free to contact me (e.g., on the Discord server) if you wish to discuss contribution before making a pull request. If you are not a software developer and want to contribute, even something as small as _Staring_ this repository really makes my day and keeps me motivated!\n\n### Crowd Sourcing Item Skill Requirements\n\nA really manual part of the item database is the `item.equipment.requirements` data. So far, I have manually populated this data... for over 3,500 items! To keep this project alive, I have stopped adding in this data (as it takes a lot of time). Here is a summary of how the item skill requirements work:\n\n- All item requirements are stored in the [`skill-requirements.json`](https://github.com/osrsbox/osrsbox-db/blob/master/data/items/items-skill-requirements.json) file\n\n- They have a structure of:\n\n```\n    "item_id": {\n        "skill_name": integer\n    },\n```\n\n- For example, the Abyssal whip item:\n\n```\n    "4151": {\n        "attack": 70\n    },\n```\n\n- For the `skill_name`, the [`schema-items.json`](https://github.com/osrsbox/schemas/blob/67b062b8d8499f80f43a95ff3b72cc40a6a833c9/schema-items.json#L293) file has a list of the allowed values - to help get the correct skill name. For example, `runecraft` and not `runecrafting`!\n\nWith some community help (by crowd sourcing) we could keep this data point fresh. If you find an error or want to add in a requirement, and want to contribute, here are the best ways to help:\n\n- GitHub PR: Clone the project repo, make changes to `skill-requirements.json`, submit PR\n- GitHub Issue: Submit an issue with the fix. It would really help me if you put the request in the correct JSON format as described above!\n\nFYI - there is currently no quest-associated requirements. This would be a great addition to the project, but seems to be a very complex thing to add.\n\n## Additional Project Information\n\nThis section contains additional information about the osrsbox-db project. For detailed information about the project see the [`osrsbox.com`](https://www.osrsbox.com/) website for the official project page, and the _Database_ tag to find blog posts about the project: \n\n- https://www.osrsbox.com/projects/osrsbox-db/\n- https://www.osrsbox.com/blog/tags/Database/\n\n### Project Feedback\n\nI would thoroughly appreciate any feedback regarding the osrsbox-db project, especially problems with the inaccuracies of the data provided. So if you notice any problem with the accuracy of item property data, could you please let me know. The same goes for any discovered bugs, or if you have a specific feature request. The best method is to [open a new Github issue](https://github.com/osrsbox/osrsbox-db/issues) in the project repository. \n\n### Project License\n\nThe osrsbox-db project is released under the GNU General Public License version 3 as published by the Free Software Foundation. You can read the [LICENSE](LICENSE) file for the full license, check the [GNU GPL](https://www.gnu.org/licenses/gpl-3.0.en.html) page for additional information, or check the [tl;drLegal](https://tldrlegal.com/license/gnu-general-public-license-v3-(gpl-3)) documentation for the license explained in simple English. The GPL license is specified for all source code contained in this project. Other content is specified under GPL if not listed in the **Exceptions to GPL** below.\n\n#### Exceptions to GPL\n\nOld School RuneScape (OSRS) content and materials are trademarks and copyrights of JaGeX or its licensors. All rights reserved. OSRSBox and the osrsbox-db project is not associated or affiliated with JaGeX or its licensors. \n\nAdditional data to help build this project is sourced from the [OSRS Wiki](https://oldschool.runescape.wiki/). This primarily includes item and monster metadata that is not available in the OSRS cache. As specified by the [Weird Gloop Copyright](https://meta.weirdgloop.org/w/Meta:Copyrights) page, this content is licensed under CC BY-NC-SA 3.0 - [Attribution-NonCommercial-ShareAlike 3.0 Unported](https://creativecommons.org/licenses/by-nc-sa/3.0/) license.\n\n### Project Attribution\n\nThe osrsbox-db project is a labor of love. I put a huge amount of time and effort into the project, and I want people to use it. That is the entire reason for its existence. I am not too fussed about attribution guidelines... but if you want to use the project please adhere to the licenses used. Please feel free to link to this repository or my [OSRSBox website](https://www.osrsbox.com/) if you use it in your project - mainly so others can find it, and hopefully use it too!\n',
    'author': 'PH01L',
    'author_email': 'phoil@osrsbox.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/osrsbox/osrsbox-db',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
