# -*- coding: utf-8 -*-
# Copyright (c) 2020 Matt Martz <matt@sivel.net>
# BSD-3-Clause (see LICENSE or https://opensource.org/licenses/BSD-3-Clause)

import inspect
import sys
from modulefinder import ModuleFinder

from IPython.core.magic import Magics
from IPython.core.magic import line_magic
from IPython.core.magic import magics_class
from IPython.core.magic import needs_local_scope

try:
    from importlib import reload
except ImportError:
    from imp import reload


Sentinel = object()


def _reload(obj, name=None):
    obj_type = type(obj)
    module = inspect.getmodule(obj)
    sys_modules_frozen = sys.modules.copy()
    sys_modules_sorted = sorted(
        sys_modules_frozen.items(),
        key=lambda t: len(t[0]),
        reverse=True
    )
    if not module:
        for modname, module in sys_modules_sorted:
            if modname.split('.')[0] in STDLIB_NAMES:
                continue
            candidate = getattr(module, name, Sentinel)
            if (candidate is not Sentinel and
                    isinstance(candidate, obj_type)):
                break
        else:
            module = None

    if not module:
        raise TypeError()

    finder = ModuleFinder()
    finder.run_script(module.__file__)

    for m in finder.modules:
        if m.split('.')[0] in STDLIB_NAMES:
            continue
        try:
            reload(sys_modules_frozen[m])
        except KeyError:
            pass

    return reload(module)


@magics_class
class ReloadMagics(Magics):
    @line_magic
    @needs_local_scope
    def reload(self, line, local_ns=None):
        line = line.strip()

        if line in local_ns:
            module = _reload(local_ns[line], line)
            local_ns[line] = getattr(module, line)
        elif line in sys.modules:
            _reload(sys.modules[line])
        else:
            raise TypeError()


def load_ipython_extension(ipython):
    ipython.register_magics(ReloadMagics)


STDLIB_NAMES = {
    'AL',
    'BaseHTTPServer',
    'Bastion',
    'Binary',
    'Boolean',
    'CGIHTTPServer',
    'ColorPicker',
    'ConfigParser',
    'Cookie',
    'DEVICE',
    'DocXMLRPCServer',
    'EasyDialogs',
    'FL',
    'FrameWork',
    'GL',
    'HTMLParser',
    'LICENSE',
    'MacOS',
    'Mapping',
    'MimeWriter',
    'MiniAEFrame',
    'Numeric',
    'Queue',
    'SUNAUDIODEV',
    'ScrolledText',
    'Sequence',
    'Set',
    'SimpleHTTPServer',
    'SimpleXMLRPCServer',
    'SocketServer',
    'StringIO',
    'Text',
    'Tix',
    'Tkinter',
    'UserDict',
    'UserList',
    'UserString',
    '__builtin__',
    '__future__',
    '__main__',
    '_abc',
    '_ast',
    '_asyncio',
    '_bisect',
    '_blake2',
    '_bootlocale',
    '_bz2',
    '_codecs',
    '_codecs_cn',
    '_codecs_hk',
    '_codecs_iso2022',
    '_codecs_jp',
    '_codecs_kr',
    '_codecs_tw',
    '_collections',
    '_collections_abc',
    '_compat_pickle',
    '_compression',
    '_contextvars',
    '_crypt',
    '_csv',
    '_ctypes',
    '_ctypes_test',
    '_curses',
    '_curses_panel',
    '_datetime',
    '_dbm',
    '_decimal',
    '_dummy_thread',
    '_elementtree',
    '_functools',
    '_gdbm',
    '_hashlib',
    '_heapq',
    '_imp',
    '_io',
    '_json',
    '_locale',
    '_lsprof',
    '_lzma',
    '_markupbase',
    '_md5',
    '_multibytecodec',
    '_multiprocessing',
    '_opcode',
    '_operator',
    '_osx_support',
    '_pickle',
    '_posixshmem',
    '_posixsubprocess',
    '_py_abc',
    '_pydecimal',
    '_pyio',
    '_queue',
    '_random',
    '_scproxy',
    '_sha1',
    '_sha256',
    '_sha3',
    '_sha512',
    '_signal',
    '_sitebuiltins',
    '_socket',
    '_sqlite3',
    '_sre',
    '_ssl',
    '_stat',
    '_statistics',
    '_string',
    '_strptime',
    '_struct',
    '_sysconfigdata__darwin_darwin',
    '_testbuffer',
    '_testcapi',
    '_testimportmultiple',
    '_testinternalcapi',
    '_testmultiphase',
    '_thread',
    '_threading_local',
    '_tracemalloc',
    '_uuid',
    '_warnings',
    '_weakref',
    '_weakrefset',
    '_xxsubinterpreters',
    '_xxtestfuzz',
    'abc',
    'aepack',
    'aetools',
    'aetypes',
    'aifc',
    'al',
    'antigravity',
    'anydbm',
    'argparse',
    'array',
    'ast',
    'asynchat',
    'asyncio',
    'asyncore',
    'atexit',
    'audioop',
    'autoGIL',
    'base64',
    'bdb',
    'binascii',
    'binhex',
    'bisect',
    'bsddb',
    'builtins',
    'bz2',
    'cPickle',
    'cProfile',
    'cStringIO',
    'calendar',
    'cd',
    'cgi',
    'cgitb',
    'chunk',
    'cmath',
    'cmd',
    'code',
    'codecs',
    'codeop',
    'collections',
    'colorsys',
    'commands',
    'compileall',
    'concurrent',
    'config-3',
    'configparser',
    'contextlib',
    'contextvars',
    'cookielib',
    'copy',
    'copy_reg',
    'copyreg',
    'crypt',
    'csv',
    'ctypes',
    'curses',
    'dataclasses',
    'datetime',
    'dbhash',
    'dbm',
    'decimal',
    'difflib',
    'dircache',
    'dis',
    'distutils',
    'dl',
    'doctest',
    'dumbdbm',
    'dummy_thread',
    'dummy_threading',
    'email',
    'encodings',
    'ensurepip',
    'enum',
    'errno',
    'faulthandler',
    'fcntl',
    'filecmp',
    'fileinput',
    'findertools',
    'fl',
    'flp',
    'fm',
    'fnmatch',
    'formatter',
    'fpectl',
    'fpformat',
    'fractions',
    'ftplib',
    'functools',
    'future_builtins',
    'gc',
    'gdbm',
    'genericpath',
    'gensuitemodule',
    'getopt',
    'getpass',
    'gettext',
    'gl',
    'glob',
    'grp',
    'gzip',
    'hashlib',
    'heapq',
    'hmac',
    'hotshot',
    'html',
    'htmlentitydefs',
    'htmllib',
    'http',
    'httplib',
    'ic',
    'idlelib',
    'imageop',
    'imaplib',
    'imgfile',
    'imghdr',
    'imp',
    'importlib',
    'imputil',
    'inspect',
    'io',
    'ipaddress',
    'itertools',
    'jpeg',
    'json',
    'keyword',
    'lib-dynload',
    'lib2to3',
    'linecache',
    'locale',
    'logging',
    'lzma',
    'macostools',
    'macpath',
    'macurl2path',
    'mailbox',
    'mailcap',
    'marshal',
    'math',
    'md5',
    'mhlib',
    'mimetools',
    'mimetypes',
    'mimify',
    'mmap',
    'modulefinder',
    'msilib',
    'multifile',
    'multiprocessing',
    'mutex',
    'netrc',
    'new',
    'nis',
    'nntplib',
    'ntpath',
    'nturl2path',
    'numbers',
    'opcode',
    'operator',
    'optparse',
    'os',
    'os2emxpath',
    'ossaudiodev',
    'parser',
    'pathlib',
    'pdb',
    'pickle',
    'pickletools',
    'pipes',
    'pkgutil',
    'platform',
    'plistlib',
    'popen2',
    'poplib',
    'posix',
    'posixfile',
    'posixpath',
    'pprint',
    'profile',
    'pstats',
    'pty',
    'pwd',
    'py_compile',
    'pyclbr',
    'pydoc',
    'pydoc_data',
    'pyexpat',
    'queue',
    'quopri',
    'random',
    're',
    'readline',
    'repr',
    'reprlib',
    'resource',
    'rexec',
    'rfc822',
    'rlcompleter',
    'robotparser',
    'runpy',
    'sched',
    'secrets',
    'select',
    'selectors',
    'sets',
    'sgmllib',
    'sha',
    'shelve',
    'shlex',
    'shutil',
    'signal',
    'site',
    'site-packages',
    'smtpd',
    'smtplib',
    'sndhdr',
    'socket',
    'socketserver',
    'spwd',
    'sqlite3',
    'sre_compile',
    'sre_constants',
    'sre_parse',
    'ssl',
    'stat',
    'statistics',
    'statvfs',
    'string',
    'stringprep',
    'struct',
    'subprocess',
    'sunau',
    'sunaudiodev',
    'symbol',
    'symtable',
    'sys',
    'sysconfig',
    'syslog',
    'tabnanny',
    'tarfile',
    'telnetlib',
    'tempfile',
    'termios',
    'test',
    'textwrap',
    'this',
    'thread',
    'threading',
    'time',
    'timeit',
    'tkinter',
    'token',
    'tokenize',
    'trace',
    'traceback',
    'tracemalloc',
    'ttk',
    'tty',
    'turtle',
    'turtledemo',
    'types',
    'typing',
    'unicodedata',
    'unittest',
    'urllib',
    'urllib2',
    'urlparse',
    'user',
    'uu',
    'uuid',
    'venv',
    'warnings',
    'wave',
    'weakref',
    'webbrowser',
    'whichdb',
    'winsound',
    'wsgiref',
    'xdrlib',
    'xml',
    'xmlrpc',
    'xmlrpclib',
    'xxlimited',
    'zipapp',
    'zipfile',
    'zipimport',
    'zlib'
}
