# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['automapper', 'automapper.extensions']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'py-automapper',
    'version': '1.2.3',
    'description': 'Library for automatically mapping one object to another',
    'long_description': '<img src="logo.png" style="width:128px; margin-right: 20px;" />\n\n# py-automapper <!-- omit in toc -->\n\n**Build Status**\n[![Main branch status](https://github.com/anikolaienko/py-automapper/actions/workflows/run_code_checks.yml/badge.svg?branch=main)](https://github.com/anikolaienko/py-automapper/actions?query=branch%3Amain)\n\n---\n\nTable of Contents:\n- [Versions](#versions)\n- [About](#about)\n- [Contribute](#contribute)\n- [Usage](#usage)\n  - [Installation](#installation)\n  - [Get started](#get-started)\n  - [Map dictionary source to target object](#map-dictionary-source-to-target-object)\n  - [Different field names](#different-field-names)\n  - [Overwrite field value in mapping](#overwrite-field-value-in-mapping)\n  - [Disable Deepcopy](#disable-deepcopy)\n  - [Extensions](#extensions)\n  - [Pydantic/FastAPI Support](#pydanticfastapi-support)\n  - [TortoiseORM Support](#tortoiseorm-support)\n  - [SQLAlchemy Support](#sqlalchemy-support)\n  - [Create your own extension (Advanced)](#create-your-own-extension-advanced)\n\n# Versions\nCheck [CHANGELOG.md](/CHANGELOG.md)\n\n# About\n\n**Python auto mapper** is useful for multilayer architecture which requires constant mapping between objects from separate layers (data layer, presentation layer, etc).\n\nInspired by: [object-mapper](https://github.com/marazt/object-mapper)\n\nThe major advantage of py-automapper is its extensibility, that allows it to map practically any type, discover custom class fields and customize mapping rules. Read more in [documentation](https://anikolaienko.github.io/py-automapper).\n\n# Contribute\nRead [CONTRIBUTING.md](/CONTRIBUTING.md) guide.\n\n# Usage\n## Installation\nInstall package:\n```bash\npip install py-automapper\n```\n\n## Get started\nLet\'s say we have domain model `UserInfo` and its API representation `PublicUserInfo` without exposing user `age`:\n```python\nclass UserInfo:\n    def __init__(self, name: str, profession: str, age: int):\n        self.name = name\n        self.profession = profession\n        self.age = age\n\nclass PublicUserInfo:\n    def __init__(self, name: str, profession: str):\n        self.name = name\n        self.profession = profession\n\nuser_info = UserInfo("John Malkovich", "engineer", 35)\n```\nTo create `PublicUserInfo` object:\n```python\nfrom automapper import mapper\n\npublic_user_info = mapper.to(PublicUserInfo).map(user_info)\n\nprint(vars(public_user_info))\n# {\'name\': \'John Malkovich\', \'profession\': \'engineer\'}\n```\nYou can register which class should map to which first:\n```python\n# Register\nmapper.add(UserInfo, PublicUserInfo)\n\npublic_user_info = mapper.map(user_info)\n\nprint(vars(public_user_info))\n# {\'name\': \'John Malkovich\', \'profession\': \'engineer\'}\n```\n\n## Map dictionary source to target object\nIf source object is dictionary:\n```python\nsource = {\n    "name": "John Carter",\n    "profession": "hero"\n}\npublic_info = mapper.to(PublicUserInfo).map(source)\n\nprint(vars(public_info))\n# {\'name\': \'John Carter\', \'profession\': \'hero\'}\n```\n\n## Different field names\nIf your target class field name is different from source class.\n```python\nclass PublicUserInfo:\n    def __init__(self, full_name: str, profession: str):\n        self.full_name = full_name       # UserInfo has `name` instead\n        self.profession = profession\n```\nSimple map:\n```python\npublic_user_info = mapper.to(PublicUserInfo).map(user_info, fields_mapping={\n    "full_name": user_info.name\n})\n```\nPreregister and map. Source field should start with class name followed by period sign and field name:\n```python\nmapper.add(UserInfo, PublicUserInfo, fields_mapping={"full_name": "UserInfo.name"})\npublic_user_info = mapper.map(user_info)\n\nprint(vars(public_user_info))\n# {\'full_name\': \'John Malkovich\', \'profession\': \'engineer\'}\n```\n\n## Overwrite field value in mapping\nVery easy if you want to field just have different value, you provide a new value:\n```python\npublic_user_info = mapper.to(PublicUserInfo).map(user_info, fields_mapping={\n    "full_name": "John Cusack"\n})\n\nprint(vars(public_user_info))\n# {\'full_name\': \'John Cusack\', \'profession\': \'engineer\'}\n```\n\n## Disable Deepcopy\nBy default, py-automapper performs a recursive `copy.deepcopy()` call on all attributes when copying from source object into target class instance.\nThis makes sure that changes in the attributes of the source do not affect the target and vice versa.\nIf you need your target and source class share same instances of child objects, set `use_deepcopy=False` in `map` function.\n\n```python\nfrom dataclasses import dataclass\nfrom automapper import mapper\n\n@dataclass\nclass Address:\n    street: str\n    number: int\n    zip_code: int\n    city: str\n  \nclass PersonInfo:\n    def __init__(self, name: str, age: int, address: Address):\n        self.name = name\n        self.age = age\n        self.address = address\n\nclass PublicPersonInfo:\n    def __init__(self, name: str, address: Address):\n        self.name = name\n        self.address = address\n\naddress = Address(street="Main Street", number=1, zip_code=100001, city=\'Test City\')\ninfo = PersonInfo(\'John Doe\', age=35, address=address)\n\n# default deepcopy behavior\npublic_info = mapper.to(PublicPersonInfo).map(info)\nprint("Target public_info.address is same as source address: ", address is public_info.address)\n# Target public_info.address is same as source address: False\n\n# disable deepcopy\npublic_info = mapper.to(PublicPersonInfo).map(info, use_deepcopy=False)\nprint("Target public_info.address is same as source address: ", address is public_info.address)\n# Target public_info.address is same as source address: True\n```\n\n## Extensions\n`py-automapper` has few predefined extensions for mapping support to classes for frameworks:\n* [FastAPI](https://github.com/tiangolo/fastapi) and [Pydantic](https://github.com/samuelcolvin/pydantic)\n* [TortoiseORM](https://github.com/tortoise/tortoise-orm)\n* [SQLAlchemy](https://www.sqlalchemy.org/)\n\n## Pydantic/FastAPI Support\nOut of the box Pydantic models support:\n```python\nfrom pydantic import BaseModel\nfrom typing import List\nfrom automapper import mapper\n\nclass UserInfo(BaseModel):\n    id: int\n    full_name: str\n    public_name: str\n    hobbies: List[str]\n\nclass PublicUserInfo(BaseModel):\n    id: int\n    public_name: str\n    hobbies: List[str]\n\nobj = UserInfo(\n    id=2,\n    full_name="Danny DeVito",\n    public_name="dannyd",\n    hobbies=["acting", "comedy", "swimming"]\n)\n\nresult = mapper.to(PublicUserInfo).map(obj)\n# same behaviour with preregistered mapping\n\nprint(vars(result))\n# {\'id\': 2, \'public_name\': \'dannyd\', \'hobbies\': [\'acting\', \'comedy\', \'swimming\']}\n```\n\n## TortoiseORM Support\nOut of the box TortoiseORM models support:\n```python\nfrom tortoise import Model, fields\nfrom automapper import mapper\n\nclass UserInfo(Model):\n    id = fields.IntField(pk=True)\n    full_name = fields.TextField()\n    public_name = fields.TextField()\n    hobbies = fields.JSONField()\n\nclass PublicUserInfo(Model):\n    id = fields.IntField(pk=True)\n    public_name = fields.TextField()\n    hobbies = fields.JSONField()\n\nobj = UserInfo(\n    id=2,\n    full_name="Danny DeVito",\n    public_name="dannyd",\n    hobbies=["acting", "comedy", "swimming"],\n    using_db=True\n)\n\nresult = mapper.to(PublicUserInfo).map(obj)\n# same behaviour with preregistered mapping\n\n# filtering out protected fields that start with underscore "_..."\nprint({key: value for key, value in vars(result) if not key.startswith("_")})\n# {\'id\': 2, \'public_name\': \'dannyd\', \'hobbies\': [\'acting\', \'comedy\', \'swimming\']}\n```\n\n## SQLAlchemy Support\nOut of the box SQLAlchemy models support:\n```python\nfrom sqlalchemy.orm import declarative_base\nfrom sqlalchemy import Column, Integer, String\nfrom automapper import mapper\n\nBase = declarative_base()\n\nclass UserInfo(Base):\n    __tablename__ = "users"\n    id = Column(Integer, primary_key=True)\n    full_name = Column(String)\n    public_name = Column(String)\n    hobbies = Column(String)\n    def __repr__(self):\n        return "<User(full_name=\'%s\', public_name=\'%s\', hobbies=\'%s\')>" % (\n            self.full_name,\n            self.public_name,\n            self.hobbies,\n        )\n\nclass PublicUserInfo(Base):\n    __tablename__ = \'public_users\'\n    id = Column(Integer, primary_key=True)\n    public_name = Column(String)\n    hobbies = Column(String)\n    \nobj = UserInfo(\n            id=2,\n            full_name="Danny DeVito",\n            public_name="dannyd",\n            hobbies="acting, comedy, swimming",\n        )\n\nresult = mapper.to(PublicUserInfo).map(obj)\n# same behaviour with preregistered mapping\n\n# filtering out protected fields that start with underscore "_..."\nprint({key: value for key, value in vars(result) if not key.startswith("_")})\n# {\'id\': 2, \'public_name\': \'dannyd\', \'hobbies\': "acting, comedy, swimming"}\n```\n\n## Create your own extension (Advanced)\nWhen you first time import `mapper` from `automapper` it checks default extensions and if modules are found for these extensions, then they will be automatically loaded for default `mapper` object.\n\n**What does extension do?** To know what fields in Target class are available for mapping, `py-automapper` needs to know how to extract the list of fields. There is no generic way to do that for all Python objects. For this purpose `py-automapper` uses extensions.\n\nList of default extensions can be found in [/automapper/extensions](/automapper/extensions) folder. You can take a look how it\'s done for a class with `__init__` method or for Pydantic or TortoiseORM models.\n\nYou can create your own extension and register in `mapper`:\n```python\nfrom automapper import mapper\n\nclass TargetClass:\n    def __init__(self, **kwargs):\n        self.name = kwargs["name"]\n        self.age = kwargs["age"]\n    \n    @staticmethod\n    def get_fields(cls):\n        return ["name", "age"]\n\nsource_obj = {"name": "Andrii", "age": 30}\n\ntry:\n    # Map object\n    target_obj = mapper.to(TargetClass).map(source_obj)\nexcept Exception as e:\n    print(f"Exception: {repr(e)}")\n    # Output:\n    # Exception: KeyError(\'name\')\n\n    # mapper could not find list of fields from BaseClass\n    # let\'s register extension for class BaseClass and all inherited ones\n    mapper.add_spec(TargetClass, TargetClass.get_fields)\n    target_obj = mapper.to(TargetClass).map(source_obj)\n\n    print(f"Name: {target_obj.name}; Age: {target_obj.age}")\n```\n\nYou can also create your own clean Mapper without any extensions and define extension for very specific classes, e.g. if class accepts `kwargs` parameter in `__init__` method and you want to copy only specific fields. Next example is a bit complex but probably rarely will be needed:\n```python\nfrom typing import Type, TypeVar\n\nfrom automapper import Mapper\n\n# Create your own Mapper object without any predefined extensions\nmapper = Mapper()\n\nclass TargetClass:\n    def __init__(self, **kwargs):\n        self.data = kwargs.copy()\n\n    @classmethod\n    def fields(cls):\n        return ["name", "age", "profession"]\n\nsource_obj = {"name": "Andrii", "age": 30, "profession": None}\n\ntry:\n    target_obj = mapper.to(TargetClass).map(source_obj)\nexcept Exception as e:\n    print(f"Exception: {repr(e)}")\n    # Output:\n    # Exception: MappingError("No spec function is added for base class of <class \'type\'>")\n\n# Instead of using base class, we define spec for all classes that have `fields` property\nT = TypeVar("T")\n\ndef class_has_fields_property(target_cls: Type[T]) -> bool:\n    return callable(getattr(target_cls, "fields", None))\n    \nmapper.add_spec(class_has_fields_property, lambda t: getattr(t, "fields")())\n\ntarget_obj = mapper.to(TargetClass).map(source_obj)\nprint(f"Name: {target_obj.data[\'name\']}; Age: {target_obj.data[\'age\']}; Profession: {target_obj.data[\'profession\']}")\n# Output:\n# Name: Andrii; Age: 30; Profession: None\n\n# Skip `None` value\ntarget_obj = mapper.to(TargetClass).map(source_obj, skip_none_values=True)\nprint(f"Name: {target_obj.data[\'name\']}; Age: {target_obj.data[\'age\']}; Has profession: {hasattr(target_obj, \'profession\')}")\n# Output:\n# Name: Andrii; Age: 30; Has profession: False\n```\n',
    'author': 'Andrii Nikolaienko',
    'author_email': 'anikolaienko14@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/anikolaienko/py-automapper',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
