#!/usr/bin/env node
"use strict";
/**
 * automatically creates a module for any CloudFormation namespaces that do not
 * have an AWS construct library.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const pkglint = require("@aws-cdk/pkglint");
const fs = require("fs-extra");
const cfnspec = require("../lib");
// don't be a prude:
/* eslint-disable no-console */
/* eslint-disable quote-props */
async function main() {
    const root = path.join(__dirname, '..', '..');
    if (path.basename(root) !== '@aws-cdk') {
        throw new Error(`Something went wrong. We expected ${root} to be the "packages/@aws-cdk" directory. Did you move me?`);
    }
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const version = require('../package.json').version;
    // iterate over all cloudformation namespaces
    for (const namespace of cfnspec.namespaces()) {
        const module = pkglint.createModuleDefinitionFromCfnNamespace(namespace);
        const lowcaseModuleName = module.moduleBaseName.toLocaleLowerCase();
        const packagePath = path.join(root, module.moduleName);
        // we already have a module for this namesapce, move on.
        if (await fs.pathExists(packagePath)) {
            const packageJsonPath = path.join(packagePath, 'package.json');
            // eslint-disable-next-line @typescript-eslint/no-require-imports
            const packageJson = require(packageJsonPath);
            let scopes = packageJson['cdk-build'].cloudformation;
            if (typeof scopes === 'string') {
                scopes = [scopes];
            }
            if (scopes.indexOf(namespace) !== -1) {
                // V2-style module is already modeled in the root package, nothing to be done!
                continue;
            }
            else if (await fs.pathExists(path.join(root, `${module.moduleFamily}-${module.moduleBaseName}`.toLocaleLowerCase()))) {
                // V2-style package already has it's own package (legacy behavior), nothing to be done!
                continue;
            }
            else {
                // V2-style package needs to be added to it's "V1" package... Get down to business!
                console.error(`Adding ${namespace} to ${module.packageName}`);
                scopes.push(namespace);
                packageJson['cdk-build'].cloudformation = scopes;
                await fs.writeJson(packageJsonPath, packageJson, { encoding: 'utf-8', spaces: 2 });
                const indexTsPath = path.join(packagePath, 'lib', 'index.ts');
                const indexTs = [
                    (await fs.readFile(indexTsPath, { encoding: 'utf8' })).trimRight(),
                    `// ${namespace} CloudFormation Resources:`,
                    `export * from './${lowcaseModuleName}.generated';`,
                ].join('\n');
                await fs.writeFile(indexTsPath, indexTs, { encoding: 'utf8' });
                continue;
            }
        }
        async function write(relativePath, contents) {
            const fullPath = path.join(packagePath, relativePath);
            const dir = path.dirname(fullPath);
            await fs.mkdirp(dir);
            let data;
            if (typeof contents === 'string') {
                data = contents.trimLeft(); // trim first newline
            }
            else if (Array.isArray(contents)) {
                data = contents.join('\n');
            }
            else if (typeof contents === 'object') {
                data = JSON.stringify(contents, undefined, 2);
            }
            else {
                throw new Error('Invalid type of contents: ' + contents);
            }
            await fs.writeFile(fullPath, data + '\n');
        }
        console.log(`generating module for ${module.packageName}...`);
        const description = `${namespace} Construct Library`;
        await write('package.json', {
            name: module.packageName,
            version,
            description,
            main: 'lib/index.js',
            types: 'lib/index.d.ts',
            jsii: {
                outdir: 'dist',
                projectReferences: true,
                targets: {
                    dotnet: {
                        namespace: module.dotnetPackage,
                        packageId: module.dotnetPackage,
                        signAssembly: true,
                        assemblyOriginatorKeyFile: '../../key.snk',
                        iconUrl: 'https://raw.githubusercontent.com/aws/aws-cdk/master/logo/default-256-dark.png',
                    },
                    java: {
                        package: `${module.javaGroupId}.${module.javaPackage}`,
                        maven: {
                            groupId: module.javaGroupId,
                            artifactId: module.javaArtifactId,
                        },
                    },
                    python: {
                        classifiers: [
                            'Framework :: AWS CDK',
                            'Framework :: AWS CDK :: 1',
                        ],
                        distName: module.pythonDistName,
                        module: module.pythonModuleName,
                    },
                },
                metadata: {
                    jsii: {
                        rosetta: {
                            strict: true,
                        },
                    },
                },
            },
            repository: {
                type: 'git',
                url: 'https://github.com/aws/aws-cdk.git',
                directory: `packages/${module.packageName}`,
            },
            homepage: 'https://github.com/aws/aws-cdk',
            scripts: {
                build: 'cdk-build',
                watch: 'cdk-watch',
                lint: 'cdk-lint',
                test: 'cdk-test',
                integ: 'cdk-integ',
                pkglint: 'pkglint -f',
                package: 'cdk-package',
                awslint: 'cdk-awslint',
                cfn2ts: 'cfn2ts',
                'build+test': 'yarn build && yarn test',
                'build+test+package': 'yarn build+test && yarn package',
                compat: 'cdk-compat',
                gen: 'cfn2ts',
                'rosetta:extract': 'yarn --silent jsii-rosetta extract',
                'build+extract': 'yarn build && yarn rosetta:extract',
                'build+test+extract': 'yarn build+test && yarn rosetta:extract',
            },
            'cdk-build': {
                cloudformation: namespace,
                jest: true,
                env: {
                    AWSLINT_BASE_CONSTRUCT: 'true',
                },
            },
            keywords: [
                'aws',
                'cdk',
                'constructs',
                namespace,
                module.moduleName,
            ],
            author: {
                name: 'Amazon Web Services',
                url: 'https://aws.amazon.com',
                organization: true,
            },
            license: 'Apache-2.0',
            devDependencies: {
                '@aws-cdk/assertions': version,
                '@aws-cdk/cdk-build-tools': version,
                '@aws-cdk/cfn2ts': version,
                '@aws-cdk/pkglint': version,
                '@types/jest': '^26.0.22',
            },
            dependencies: {
                '@aws-cdk/core': version,
            },
            peerDependencies: {
                '@aws-cdk/core': version,
            },
            engines: {
                node: '>= 10.13.0 <13 || >=13.7.0',
            },
            stability: 'experimental',
            maturity: 'cfn-only',
            awscdkio: {
                announce: false,
            },
            publishConfig: {
                tag: 'latest',
            },
        });
        await write('.gitignore', [
            '*.js',
            '*.js.map',
            '*.d.ts',
            'tsconfig.json',
            'node_modules',
            '*.generated.ts',
            'dist',
            '.jsii',
            '',
            '.LAST_BUILD',
            '.nyc_output',
            'coverage',
            '.nycrc',
            '.LAST_PACKAGE',
            '*.snk',
            'nyc.config.js',
            '!.eslintrc.js',
            '!jest.config.js',
            'junit.xml',
        ]);
        await write('.npmignore', [
            '# Don\'t include original .ts files when doing `npm pack`',
            '*.ts',
            '!*.d.ts',
            'coverage',
            '.nyc_output',
            '*.tgz',
            '',
            'dist',
            '.LAST_PACKAGE',
            '.LAST_BUILD',
            '!*.js',
            '',
            '# Include .jsii',
            '!.jsii',
            '',
            '*.snk',
            '',
            '*.tsbuildinfo',
            '',
            'tsconfig.json',
            '',
            '.eslintrc.js',
            'jest.config.js',
            '',
            '# exclude cdk artifacts',
            '**/cdk.out',
            'junit.xml',
            'test/',
            '!*.lit.ts',
        ]);
        await write('lib/index.ts', [
            `// ${namespace} CloudFormation Resources:`,
            `export * from './${lowcaseModuleName}.generated';`,
        ]);
        await write(`test/${lowcaseModuleName}.test.ts`, [
            "import '@aws-cdk/assertions';",
            "import {} from '../lib';",
            '',
            "test('No tests are specified for this package', () => {",
            '  expect(true).toBe(true);',
            '});',
        ]);
        await pkglint.createLibraryReadme(namespace, path.join(packagePath, 'README.md'));
        await write('.eslintrc.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/eslintrc');",
            "baseConfig.parserOptions.project = __dirname + '/tsconfig.json';",
            'module.exports = baseConfig;',
        ]);
        await write('jest.config.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/jest.config');",
            'module.exports = baseConfig;',
        ]);
        await write('rosetta/default.ts-fixture', [
            "import { Construct } from 'constructs';",
            "import { Stack } from '@aws-cdk/core';",
            '',
            'class MyStack extends Stack {',
            '  constructor(scope: Construct, id: string) {',
            '    /// here',
            '  }',
            '}',
        ]);
        const templateDir = path.join(__dirname, 'template');
        for (const file of await fs.readdir(templateDir)) {
            await fs.copy(path.join(templateDir, file), path.join(packagePath, file));
        }
        await addDependencyToMegaPackage(path.join('@aws-cdk', 'cloudformation-include'), module.packageName, version, ['dependencies', 'peerDependencies']);
        await addDependencyToMegaPackage('aws-cdk-lib', module.packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('monocdk', module.packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('decdk', module.packageName, version, ['dependencies']);
    }
}
/**
 * A few of our packages (e.g., decdk, aws-cdk-lib) require a dependency on every service package.
 * This automates adding the dependency (and peer dependency) to the package.json.
 */
async function addDependencyToMegaPackage(megaPackageName, packageName, version, dependencyTypes) {
    const packageJsonPath = path.join(__dirname, '..', '..', '..', megaPackageName, 'package.json');
    const packageJson = JSON.parse(await fs.readFile(packageJsonPath, 'utf8'));
    dependencyTypes.forEach(dependencyType => {
        const unorderedDeps = {
            ...packageJson[dependencyType],
            [packageName]: version,
        };
        packageJson[dependencyType] = {};
        Object.keys(unorderedDeps).sort().forEach(k => {
            packageJson[dependencyType][k] = unorderedDeps[k];
        });
    });
    await fs.writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2) + '\n');
}
main().catch(e => {
    console.error(e);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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