"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynthUtils = void 0;
const fs = require("fs");
const path = require("path");
const core = require("@aws-cdk/core");
class SynthUtils {
    /**
     * Returns the cloud assembly template artifact for a stack.
     */
    static synthesize(stack, options = {}) {
        // always synthesize against the root (be it an App or whatever) so all artifacts will be included
        const assembly = synthesizeApp(stack, options);
        return assembly.getStackArtifact(stack.artifactId);
    }
    /**
     * Synthesizes the stack and returns the resulting CloudFormation template.
     */
    static toCloudFormation(stack, options = {}) {
        const synth = this._synthesizeWithNested(stack, options);
        if (isStackArtifact(synth)) {
            return synth.template;
        }
        else {
            return synth;
        }
    }
    /**
     * @returns Returns a subset of the synthesized CloudFormation template (only specific resource types).
     */
    static subset(stack, options) {
        const template = this.toCloudFormation(stack);
        if (template.Resources) {
            for (const [key, resource] of Object.entries(template.Resources)) {
                if (options.resourceTypes && !options.resourceTypes.includes(resource.Type)) {
                    delete template.Resources[key];
                }
            }
        }
        return template;
    }
    /**
     * Synthesizes the stack and returns a `CloudFormationStackArtifact` which can be inspected.
     * Supports nested stacks as well as normal stacks.
     *
     * @return CloudFormationStackArtifact for normal stacks or the actual template for nested stacks
     * @internal
     */
    static _synthesizeWithNested(stack, options = {}) {
        // always synthesize against the root (be it an App or whatever) so all artifacts will be included
        const assembly = synthesizeApp(stack, options);
        // if this is a nested stack (it has a parent), then just read the template as a string
        if (stack.nestedStackParent) {
            return JSON.parse(fs.readFileSync(path.join(assembly.directory, stack.templateFile)).toString('utf-8'));
        }
        return assembly.getStackArtifact(stack.artifactId);
    }
}
exports.SynthUtils = SynthUtils;
/**
 * Synthesizes the app in which a stack resides and returns the cloud assembly object.
 */
function synthesizeApp(stack, options) {
    const root = stack.node.root;
    if (!core.Stage.isStage(root)) {
        throw new Error('unexpected: all stacks must be part of a Stage or an App');
    }
    // to support incremental assertions (i.e. "expect(stack).toNotContainSomething(); doSomething(); expect(stack).toContainSomthing()")
    const force = true;
    return root.synth({
        force,
        ...options,
    });
}
function isStackArtifact(x) {
    return 'template' in x;
}
//# sourceMappingURL=data:application/json;base64,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