#include "TLS.h"

#if defined(ENABLE_WOLFSSL) && defined(ENABLE_BOOST)

#ifndef WOLFSSL_LOGGING
#define LOG(X)
#endif
#include <fstream>

extern "C" {
#include "wolfssl/error-ssl.h"
}

namespace osuCrypto
{

    //namespace {

        //std::mutex mtx;
        error_code readFile(const std::string& file, std::vector<u8>& buffer)
        {
            std::ifstream t(file);

            if (t.is_open() == false)
                return boost::system::errc::make_error_code(boost::system::errc::no_such_file_or_directory);

            t.seekg(0, std::ios::end);
            buffer.resize(0);
            buffer.reserve(t.tellg());
            t.seekg(0, std::ios::beg);

            buffer.insert(buffer.end(), (std::istreambuf_iterator<char>(t)),
                std::istreambuf_iterator<char>());


            //std::lock_guard<std::mutex> lock(mtx);
            //std::cout << file << "\nstd::array<char, "<<buffer.size()<<"> name = {0x" << std::hex << int(buffer[0]);

            //for (u64 i = 1; i < buffer.size(); ++i)
            //    std::cout << ", 0x" << std::hex << int(buffer[i]);
            //std::cout << "};\n";

            return {};
        }
    //}

    std::mutex WolfInitMtx;

    WolfContext::Base::Base(Mode mode)
    {
        {
            std::lock_guard<std::mutex> lock(WolfInitMtx);
            switch (mode)
            {
            case Mode::Server:
                mMethod = (wolfSSLv23_server_method());
                break;
            case Mode::Client:
                mMethod = (wolfSSLv23_client_method());
                break;
            default:
                mMethod = wolfSSLv23_method();
            }

            mCtx = (wolfSSL_CTX_new(mMethod));
        }
        wolfSSL_SetIOSend(mCtx, WolfSocket::sendCallback);
        wolfSSL_SetIORecv(mCtx, WolfSocket::recvCallback);
        mMode = mode;

    }
    WolfContext::Base::~Base()
    {
        wolfSSL_CTX_free(mCtx);
    }

    void WolfContext::init(Mode mode, error_code& ec)
    {
        if (isInit())
        {
            ec = make_error_code(TLS_errc::ContextAlreadyInit);
            return;
        }
        mBase = std::make_shared<Base>(mode);

        if (isInit() == false)
            ec = make_error_code(TLS_errc::ContextFailedToInit);
    }

    void WolfContext::loadCertFile(std::string path, error_code& ec)
    {
        if (isInit() == false)
            ec = make_error_code(TLS_errc::ContextNotInit);
        else
        {
            std::vector<unsigned char> data;
            ec = readFile(path, data);
            if (!ec) 
                loadCert(data, ec);
        }
    }

    void WolfContext::loadCert(span<u8> data, error_code& ec)
    {
        if (isInit() == false)
            ec = make_error_code(TLS_errc::ContextNotInit);
        else
            ec = wolfssl_error_code(
                wolfSSL_CTX_load_verify_buffer(*this, data.data(), static_cast<long>(data.size()), WOLFSSL_FILETYPE_PEM));
    }

    void WolfContext::loadKeyPairFile(std::string pkPath, std::string skPath, error_code& ec)
    {
        if (isInit() == false) {
            ec = make_error_code(TLS_errc::ContextNotInit);
            return;
        }
        
        std::vector<unsigned char> pk, sk;
        ec = readFile(pkPath, pk);
        if (ec) return;
        ec = readFile(skPath, sk);
        if (ec) return;

        loadKeyPair(pk, sk, ec);
    }

    void WolfContext::loadKeyPair(span<u8> pk, span<u8> sk, error_code& ec)
    {
        if (isInit() == false) {
            ec = make_error_code(TLS_errc::ContextNotInit);
            return;
        }

        ec = wolfssl_error_code(
            wolfSSL_CTX_use_certificate_buffer(*this, pk.data(), static_cast<long>(pk.size()), WOLFSSL_FILETYPE_PEM));
        if (ec) return;

        ec = wolfssl_error_code(
            wolfSSL_CTX_use_PrivateKey_buffer(*this, sk.data(), static_cast<long>(sk.size()), WOLFSSL_FILETYPE_PEM));
    }

    void WolfContext::requestClientCert(error_code& ec)
    {
        if (isInit() == false) 
            ec = make_error_code(TLS_errc::ContextNotInit);
        else  if (mBase->mMode == Mode::Client) 
            ec = make_error_code(TLS_errc::OnlyValidForServerContext);
        else
        {
            wolfSSL_CTX_set_verify(*this, SSL_VERIFY_PEER, 0);
            ec = {};
        }
    }

    WolfSocket::WolfSocket(boost::asio::io_context& ios, WolfContext& ctx)
        : mSock(ios)
        , mStrand(ios.get_executor())
        , mIos(ios)
        , mSSL(wolfSSL_new(ctx))
    {
#ifdef WOLFSSL_LOGGING
        setLog(mLog_);
#endif
        if (mSSL)
        {
            wolfSSL_SetIOWriteCtx(mSSL, this);
            wolfSSL_SetIOReadCtx(mSSL, this);
        }
        else
        {
            throw std::runtime_error("Context not init correctly.");
        }
    }

    WolfSocket::WolfSocket(boost::asio::io_context& ios, boost::asio::ip::tcp::socket&& sock, WolfContext& ctx)
        : mSock(std::move(sock))
        , mStrand(ios.get_executor())
        , mIos(ios)
        , mSSL(wolfSSL_new(ctx))
    {
#ifdef WOLFSSL_LOGGING
        setLog(mLog_);
#endif
        wolfSSL_SetIOWriteCtx(mSSL, this);
        wolfSSL_SetIOReadCtx(mSSL, this);
    }

    void WolfSocket::close()
    {
        LOG("WolfSocket::close()");
        boost::system::error_code ec;
        mSock.close(ec);
    }

    void WolfSocket::cancel()
    {
        LOG("WolfSocket::cancel()");
        boost::system::error_code ec;
        mSock.cancel(ec);
    }

    void WolfSocket::send(
        span<buffer> buffers,
        error_code& ec,
        u64& bt)
    {
        std::promise<void> prom;
        async_send(buffers, [&](const error_code& ecc, u64 btt) {
            ec = ecc;
            bt = btt;
            prom.set_value(); });
        prom.get_future().get();
    }

    void WolfSocket::async_send(
        span<buffer> buffers,
        io_completion_handle&& fn)
    {
#ifdef WOLFSSL_LOGGING
        std::stringstream ss;
        ss << "async_send ";
        for (auto b : buffers)
            ss << b.size() << " ";
        LOG(ss.str());
#endif

        assert(mSendBufs.size() == 0);

        assert(buffers.size());
        for (u64 i = 0; i < buffers.size(); ++i)
            assert(buffers[i].size());

        mSendCB = std::move(fn);
        mSendBufs.insert(mSendBufs.end(), buffers.begin(), buffers.end());
        mSendBufIdx = 0;

        sendNext();
    }

    void WolfSocket::sendNext()
    {
        LOG("sendNext");
        assert(hasSendBuffer());

        boost::asio::dispatch(mStrand, [this]() {

            auto buf = (char*)curSendBuffer().data();
            auto size = curSendBuffer().size();

            int err = 0, ret = 0;
            //auto wasPending = mState.hasPendingSend();

            // this will call sslRequextSendCB(...)
            ret = wolfSSL_write(mSSL, buf, static_cast<int>(size));
            if (ret <= 0)
            {
                err = wolfSSL_get_error(mSSL, 0);
                if (err == WOLFSSL_ERROR_WANT_WRITE)
                {
                    assert(mState.hasPendingSend() == true);
                }
                else
                {
                    assert(err);
                    mSendEC = wolfssl_error_code(err);
                    auto fn = mSendCB;
                    auto bt = mSendBT;
                    mSendBT = 0;
                    fn(mSendEC, bt);
                }
            }
            else
            {
                if (ret == size)
                {

                    ++mSendBufIdx;
                    if (hasSendBuffer())
                    {
                        LOG("next send buffer");
                        sendNext();
                    }
                    else
                    {
                        mSendBufIdx = 0;
                        mSendBufs.resize(0);
                        auto fn = mSendCB;
                        auto bt = mSendBT;
                        mSendBT = 0;
                        fn(mSendEC, bt);
                    }
                }
                else
                {
                    LOG("redo send buffer");
                    curSendBuffer() = buffer(buf + ret, size - ret);
                    sendNext();
                }
            }

            });

    }

    int WolfSocket::sslRquestSendCB(char* buf, int size)
    {
        LOG("sslRquestSendCB " + std::string(mState.hasPendingSend() ? "complete" : "init"));

        assert(mStrand.running_in_this_thread());

        if (mState.hasPendingSend() == false)
        {
            mState.mPendingSendBuf = { buf, size };
            buffer b(buf, size);
            boost::asio::async_write(mSock, b, [this](const error_code& ec, u64 bt) {
                boost::asio::dispatch(mStrand, [this, ec, bt]() {
                    LOG("data Sent : " + std::to_string(bt) + " bytes");

                    mSendEC = ec;
                    mSendBT += bt;

                    assert(mState.hasPendingSend());
                    
                    switch (mState.mPhase)
                    {
                    case WolfState::Phase::Normal:
                    {
                        sendNext();
                        break;
                    }
                    case WolfState::Phase::Connect:
                    {
                        connectNext();
                        break;
                    }
                    case WolfState::Phase::Accept:
                    {
                        acceptNext();
                        break;
                    }
                    default:
                        std::terminate();
                    }
                    }
                );
                }
            );

            return WOLFSSL_CBIO_ERR_WANT_WRITE;
        }
        else
        {
            if (mSendEC)
                return WOLFSSL_CBIO_ERR_GENERAL;
            assert(mState.mPendingSendBuf.data() == buf && mState.mPendingSendBuf.size() == size);
            mState.mPendingSendBuf = {};
            return size;
        }

    }

    void WolfSocket::recv(
        span<buffer> buffers,
        error_code& ec,
        u64& bt)
    {
        std::promise<void> prom;
        async_recv(buffers, [&](const error_code& ecc, u64 btt) {
            ec = ecc;
            bt = btt;
            prom.set_value(); });
        prom.get_future().get();
    }

    void WolfSocket::async_recv(
        span<buffer> buffers,
        io_completion_handle&& fn)
    {

#ifdef WOLFSSL_LOGGING
        std::stringstream ss;
        ss << "async_recv ";
        for (auto b : buffers)
            ss << b.size() << " ";
        LOG(ss.str());
#endif

        assert(mRecvBufs.size() == 0);
        assert(buffers.size());
        for (u64 i = 0; i < buffers.size(); ++i)
            assert(buffers[i].size());

        mRecvCB = std::move(fn);
        mRecvBufs.insert(mRecvBufs.end(), buffers.begin(), buffers.end());
        mRecvBufIdx = 0;

        recvNext();
    }

    void osuCrypto::WolfSocket::setDHParamFile(std::string path, error_code& ec)
    {
        std::vector<u8> paramData;
        if (!(ec = readFile(path, paramData)))
            setDHParam(paramData, ec);
    }

    void osuCrypto::WolfSocket::setDHParam(span<u8> data, error_code& ec)
    {
        ec = wolfssl_error_code(
            wolfSSL_SetTmpDH_buffer(mSSL, data.data(), static_cast<long>(data.size()), WOLFSSL_FILETYPE_PEM));
    }

    WolfCertX509 osuCrypto::WolfSocket::getCert()
    {
        auto ptr = wolfSSL_get_peer_certificate(mSSL);
        if (ptr == nullptr)
        {
            throw std::runtime_error("please compile wolfssl with KEEP_PEER_CERT defined. For example `./configure --enable-all; make`");
        }
        return { ptr };
    }


    void WolfSocket::recvNext()
    {
        LOG("recvNext");

        boost::asio::dispatch(mStrand, [this]() {

            assert(hasRecvBuffer());

            auto buf = (char*)curRecvBuffer().data();
            auto size = curRecvBuffer().size();

            int err = 0, ret = 0;
            //auto wasPending = mState.hasPendingRecv();

            // this will call sslRequextRecvCB(...)
            ret = wolfSSL_read(mSSL, buf, static_cast<int>(size));

            if (ret <= 0)
            {
                err = wolfSSL_get_error(mSSL, 0);
                assert(err);
                if (err == WOLFSSL_ERROR_WANT_READ)
                {
                    assert(mState.hasPendingRecv() == true);
                    // no op
                }
                else
                {
                    mRecvEC = wolfssl_error_code(err);
                    auto fn = std::move(mRecvCB);
                    auto bt = mRecvBT;
                    mRecvBT = 0;
                    fn(mRecvEC, bt);
                }
            }
            else
            {
                if (ret == size)
                {
                    ++mRecvBufIdx;
                    if (hasRecvBuffer())
                    {
                        LOG("next recv buffer");
                        recvNext();
                    }
                    else
                    {
                        mRecvBufIdx = 0;
                        mRecvBufs.resize(0);
                        auto fn = std::move(mRecvCB);
                        auto bt = mRecvBT;
                        mRecvBT = 0;
                        fn(mRecvEC, bt);
                    }
                }
                else
                {
                    // update the current buffer to point to what is left...
                    LOG("redo recv buffer");
                    curRecvBuffer() = buffer(buf + ret, size - ret);
                    recvNext();
                }
            }
            }
        );
    }

    int WolfSocket::sslRquestRecvCB(char* buf, int size)
    {
        LOG("sslRquestRecvCB " + std::string(mState.hasPendingRecv() ? "complete" : "init"));

        assert(mStrand.running_in_this_thread());

        if (mState.hasPendingRecv() == false)
        {
            mState.mPendingRecvBuf = { buf,  size };
            buffer b(buf, size);
            boost::asio::async_read(mSock, b, [this](const error_code& ec, u64 bt) {
                LOG("data recvd ****");
                boost::asio::dispatch(mStrand, [this, ec, bt]() {
                    LOG("data recvd: " + std::to_string(bt) + " bytes");

                    mRecvEC = ec;
                    mRecvBT += bt;

                    assert(mState.hasPendingRecv());

                    switch (mState.mPhase)
                    {
                    case WolfState::Phase::Normal:
                    {
                        recvNext();
                        break;
                    }
                    case WolfState::Phase::Connect:
                    {
                        connectNext();
                        break;
                    }
                    case WolfState::Phase::Accept:
                    {
                        acceptNext();
                        break;
                    }
                    default:
                        std::terminate();
                    }
                    }
                );
                }
            );
            return WOLFSSL_CBIO_ERR_WANT_READ;
        }
        else
        {
            if (mRecvEC)
                return WOLFSSL_CBIO_ERR_GENERAL;
            assert(mState.mPendingRecvBuf.data() == buf && mState.mPendingRecvBuf.size() == size);
            mState.mPendingRecvBuf = {};
            return size;
        }
    }

    void WolfSocket::connect(error_code& ec)
    {
        std::promise<error_code> prom;
        async_connect([&prom](const error_code& ec) { prom.set_value(ec); });
        ec = prom.get_future().get();
    }

    void WolfSocket::async_connect(completion_handle&& cb)
    {
        LOG("async_connect");

        assert(mState.mPhase == WolfState::Phase::Uninit);
        mState.mPhase = WolfState::Phase::Connect;
        mSetupCB = std::move(cb);
        connectNext();
    }


    void WolfSocket::connectNext()
    {
        LOG("connectNext");
        boost::asio::dispatch(mStrand, [this]() {

            assert(mState.mPhase == WolfState::Phase::Connect);

            if (mCancelingPending)
            {
                auto fn = std::move(mSetupCB);
                fn(mSetupEC);
            }
            else
            {
                int ret, err = 0;
                ret = wolfSSL_connect(mSSL);
                if (ret != WOLFSSL_SUCCESS)
                    err = wolfSSL_get_error(mSSL, 0);

                if (ret == WOLFSSL_SUCCESS) {
                    mState.mPhase = WolfState::Phase::Normal;
                    auto fn = std::move(mSetupCB);
                    fn(mSetupEC);
                }
                else if (
                    err != WOLFSSL_ERROR_WANT_READ &&
                    err != WOLFSSL_ERROR_WANT_WRITE)
                {

                    if (mSendEC)
                    {
                        assert(err == SOCKET_ERROR_E);
                        mSetupEC = mSendEC;
                    }
                    else if (mRecvEC)
                    {
                        assert(err == SOCKET_ERROR_E);
                        mSetupEC = mRecvEC;
                    }
                    else
                    {
                        mSetupEC = wolfssl_error_code(err);

                        if (mState.hasPendingRecv() || mState.hasPendingSend())
                        {
                            // make sure they both aren't pending. No logic for this case.
                            assert(mState.hasPendingRecv() != mState.hasPendingSend());

                            mSock.close();
                            mCancelingPending = true;
                            return;
                        }
                    }
                    auto fn = std::move(mSetupCB);
                    fn(mSetupEC);
                }
            }
            }
        );
    }

    void WolfSocket::accept(error_code& ec)
    {
        std::promise<error_code> prom;
        async_accept([&prom](const error_code& ec) {
            prom.set_value(ec);
            }
        );
        ec = prom.get_future().get();
    }

    void WolfSocket::async_accept(completion_handle&& cb)
    {
        LOG("async_accept");
        assert(mState.mPhase == WolfState::Phase::Uninit);
        mState.mPhase = WolfState::Phase::Accept;
        mSetupCB = std::move(cb);
        acceptNext();
    }

    void WolfSocket::acceptNext()
    {
        LOG("acceptNext");
        boost::asio::dispatch(mStrand, [this]() {

            assert(mState.mPhase == WolfState::Phase::Accept);

            if (mCancelingPending)
            {
                auto fn = std::move(mSetupCB);
                fn(mSetupEC);
            }
            else
            {
                int ret, err = 0;
                ret = wolfSSL_accept(mSSL);
                if (ret != WOLFSSL_SUCCESS)
                    err = wolfSSL_get_error(mSSL, 0);

                if (ret == WOLFSSL_SUCCESS) {
                    mState.mPhase = WolfState::Phase::Normal;
                    auto fn = std::move(mSetupCB);
                    fn(mSetupEC);
                }
                else if (
                    err != WOLFSSL_ERROR_WANT_READ &&
                    err != WOLFSSL_ERROR_WANT_WRITE)
                {
                    if (mSendEC)
                    {
                        assert(err == SOCKET_ERROR_E);
                        mSetupEC = mSendEC;
                    }
                    else if (mRecvEC)
                    {
                        assert(err == SOCKET_ERROR_E);
                        mSetupEC = mRecvEC;
                    }
                    else
                    {
                        mSetupEC = wolfssl_error_code(err);
                        if (mState.hasPendingRecv() || mState.hasPendingSend())
                        {
                            // make sure they both aren't pending. No logic for this case.
                            assert(mState.hasPendingRecv() != mState.hasPendingSend());

                            mSock.close();
                            mCancelingPending = true;
                            return;
                        }
                    }
                    auto fn = std::move(mSetupCB);
                    fn(mSetupEC);
                }
            }
            }
        );
    }
#ifdef WOLFSSL_LOGGING
    void WolfSocket::LOG(std::string X) {
#ifdef WOLFSSL_LOGGING_VERBODSE
        if (wolfSSL_is_server(mSSL) == false)
            lout << Color::Red << "client " << (X) << std::endl << Color::Default;
        else
            lout << Color::Green << "server " << (X) << std::endl << Color::Default;
#endif
        push(X);
    }
#endif

    std::array<u8, 5010>   sample_ca_cert_pem     = { 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x44, 0x61, 0x74, 0x61, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x3a, 0x20, 0x33, 0x20, 0x28, 0x30, 0x78, 0x32, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x53, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x20, 0x4e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x3a, 0x20, 0x39, 0x37, 0x32, 0x37, 0x37, 0x36, 0x33, 0x37, 0x31, 0x30, 0x36, 0x36, 0x30, 0x37, 0x35, 0x33, 0x36, 0x35, 0x39, 0x20, 0x28, 0x30, 0x78, 0x38, 0x36, 0x66, 0x66, 0x66, 0x35, 0x38, 0x65, 0x31, 0x30, 0x64, 0x65, 0x62, 0x38, 0x66, 0x62, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x53, 0x69, 0x67, 0x6e, 0x61, 0x74, 0x75, 0x72, 0x65, 0x20, 0x41, 0x6c, 0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x3a, 0x20, 0x73, 0x68, 0x61, 0x32, 0x35, 0x36, 0x57, 0x69, 0x74, 0x68, 0x52, 0x53, 0x41, 0x45, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x49, 0x73, 0x73, 0x75, 0x65, 0x72, 0x3a, 0x20, 0x43, 0x3d, 0x55, 0x53, 0x2c, 0x20, 0x53, 0x54, 0x3d, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x2c, 0x20, 0x4c, 0x3d, 0x42, 0x6f, 0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x2c, 0x20, 0x4f, 0x3d, 0x53, 0x61, 0x77, 0x74, 0x6f, 0x6f, 0x74, 0x68, 0x2c, 0x20, 0x4f, 0x55, 0x3d, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x43, 0x4e, 0x3d, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x65, 0x6d, 0x61, 0x69, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3d, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x56, 0x61, 0x6c, 0x69, 0x64, 0x69, 0x74, 0x79, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x4e, 0x6f, 0x74, 0x20, 0x42, 0x65, 0x66, 0x6f, 0x72, 0x65, 0x3a, 0x20, 0x41, 0x70, 0x72, 0x20, 0x31, 0x33, 0x20, 0x31, 0x35, 0x3a, 0x32, 0x33, 0x3a, 0x30, 0x39, 0x20, 0x32, 0x30, 0x31, 0x38, 0x20, 0x47, 0x4d, 0x54, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x4e, 0x6f, 0x74, 0x20, 0x41, 0x66, 0x74, 0x65, 0x72, 0x20, 0x3a, 0x20, 0x4a, 0x61, 0x6e, 0x20, 0x20, 0x37, 0x20, 0x31, 0x35, 0x3a, 0x32, 0x33, 0x3a, 0x30, 0x39, 0x20, 0x32, 0x30, 0x32, 0x31, 0x20, 0x47, 0x4d, 0x54, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x3a, 0x20, 0x43, 0x3d, 0x55, 0x53, 0x2c, 0x20, 0x53, 0x54, 0x3d, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x2c, 0x20, 0x4c, 0x3d, 0x42, 0x6f, 0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x2c, 0x20, 0x4f, 0x3d, 0x53, 0x61, 0x77, 0x74, 0x6f, 0x6f, 0x74, 0x68, 0x2c, 0x20, 0x4f, 0x55, 0x3d, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x43, 0x4e, 0x3d, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x65, 0x6d, 0x61, 0x69, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3d, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x49, 0x6e, 0x66, 0x6f, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x41, 0x6c, 0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x3a, 0x20, 0x72, 0x73, 0x61, 0x45, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x2d, 0x4b, 0x65, 0x79, 0x3a, 0x20, 0x28, 0x32, 0x30, 0x34, 0x38, 0x20, 0x62, 0x69, 0x74, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x4d, 0x6f, 0x64, 0x75, 0x6c, 0x75, 0x73, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x30, 0x30, 0x3a, 0x62, 0x66, 0x3a, 0x30, 0x63, 0x3a, 0x63, 0x61, 0x3a, 0x32, 0x64, 0x3a, 0x31, 0x34, 0x3a, 0x62, 0x32, 0x3a, 0x31, 0x65, 0x3a, 0x38, 0x34, 0x3a, 0x34, 0x32, 0x3a, 0x35, 0x62, 0x3a, 0x63, 0x64, 0x3a, 0x33, 0x38, 0x3a, 0x31, 0x66, 0x3a, 0x34, 0x61, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x32, 0x3a, 0x34, 0x64, 0x3a, 0x37, 0x35, 0x3a, 0x31, 0x30, 0x3a, 0x66, 0x31, 0x3a, 0x62, 0x36, 0x3a, 0x33, 0x35, 0x3a, 0x39, 0x66, 0x3a, 0x64, 0x66, 0x3a, 0x63, 0x61, 0x3a, 0x37, 0x64, 0x3a, 0x30, 0x33, 0x3a, 0x39, 0x38, 0x3a, 0x64, 0x33, 0x3a, 0x61, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x65, 0x3a, 0x30, 0x33, 0x3a, 0x36, 0x36, 0x3a, 0x65, 0x65, 0x3a, 0x32, 0x61, 0x3a, 0x66, 0x31, 0x3a, 0x64, 0x38, 0x3a, 0x62, 0x30, 0x3a, 0x37, 0x64, 0x3a, 0x36, 0x65, 0x3a, 0x30, 0x37, 0x3a, 0x35, 0x34, 0x3a, 0x30, 0x62, 0x3a, 0x31, 0x30, 0x3a, 0x39, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x32, 0x31, 0x3a, 0x34, 0x64, 0x3a, 0x38, 0x30, 0x3a, 0x63, 0x62, 0x3a, 0x31, 0x32, 0x3a, 0x32, 0x30, 0x3a, 0x65, 0x37, 0x3a, 0x63, 0x63, 0x3a, 0x34, 0x66, 0x3a, 0x64, 0x65, 0x3a, 0x34, 0x35, 0x3a, 0x37, 0x64, 0x3a, 0x63, 0x39, 0x3a, 0x37, 0x32, 0x3a, 0x37, 0x37, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x32, 0x3a, 0x65, 0x61, 0x3a, 0x63, 0x61, 0x3a, 0x39, 0x30, 0x3a, 0x62, 0x62, 0x3a, 0x36, 0x39, 0x3a, 0x35, 0x32, 0x3a, 0x31, 0x30, 0x3a, 0x30, 0x33, 0x3a, 0x32, 0x66, 0x3a, 0x61, 0x38, 0x3a, 0x66, 0x33, 0x3a, 0x39, 0x35, 0x3a, 0x63, 0x35, 0x3a, 0x66, 0x31, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x62, 0x3a, 0x36, 0x32, 0x3a, 0x35, 0x36, 0x3a, 0x31, 0x62, 0x3a, 0x65, 0x66, 0x3a, 0x36, 0x37, 0x3a, 0x36, 0x66, 0x3a, 0x61, 0x34, 0x3a, 0x31, 0x30, 0x3a, 0x34, 0x31, 0x3a, 0x39, 0x35, 0x3a, 0x61, 0x64, 0x3a, 0x30, 0x61, 0x3a, 0x39, 0x62, 0x3a, 0x65, 0x33, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x35, 0x3a, 0x63, 0x30, 0x3a, 0x62, 0x30, 0x3a, 0x64, 0x32, 0x3a, 0x37, 0x30, 0x3a, 0x37, 0x36, 0x3a, 0x35, 0x30, 0x3a, 0x33, 0x30, 0x3a, 0x35, 0x62, 0x3a, 0x61, 0x38, 0x3a, 0x65, 0x38, 0x3a, 0x30, 0x38, 0x3a, 0x32, 0x63, 0x3a, 0x37, 0x63, 0x3a, 0x65, 0x64, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x37, 0x3a, 0x61, 0x32, 0x3a, 0x37, 0x61, 0x3a, 0x38, 0x64, 0x3a, 0x33, 0x38, 0x3a, 0x32, 0x39, 0x3a, 0x31, 0x63, 0x3a, 0x61, 0x63, 0x3a, 0x63, 0x37, 0x3a, 0x65, 0x64, 0x3a, 0x66, 0x32, 0x3a, 0x37, 0x63, 0x3a, 0x39, 0x35, 0x3a, 0x62, 0x30, 0x3a, 0x39, 0x35, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x32, 0x3a, 0x37, 0x64, 0x3a, 0x34, 0x39, 0x3a, 0x35, 0x63, 0x3a, 0x33, 0x38, 0x3a, 0x63, 0x64, 0x3a, 0x37, 0x37, 0x3a, 0x32, 0x35, 0x3a, 0x65, 0x66, 0x3a, 0x62, 0x64, 0x3a, 0x38, 0x30, 0x3a, 0x37, 0x35, 0x3a, 0x35, 0x33, 0x3a, 0x39, 0x34, 0x3a, 0x33, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x64, 0x3a, 0x63, 0x61, 0x3a, 0x36, 0x33, 0x3a, 0x35, 0x62, 0x3a, 0x39, 0x66, 0x3a, 0x31, 0x35, 0x3a, 0x62, 0x35, 0x3a, 0x64, 0x33, 0x3a, 0x31, 0x64, 0x3a, 0x31, 0x33, 0x3a, 0x32, 0x66, 0x3a, 0x31, 0x39, 0x3a, 0x64, 0x31, 0x3a, 0x33, 0x63, 0x3a, 0x64, 0x62, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x37, 0x36, 0x3a, 0x33, 0x61, 0x3a, 0x63, 0x63, 0x3a, 0x62, 0x38, 0x3a, 0x37, 0x64, 0x3a, 0x63, 0x39, 0x3a, 0x65, 0x35, 0x3a, 0x63, 0x32, 0x3a, 0x64, 0x37, 0x3a, 0x64, 0x61, 0x3a, 0x34, 0x30, 0x3a, 0x36, 0x66, 0x3a, 0x64, 0x38, 0x3a, 0x32, 0x31, 0x3a, 0x64, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x37, 0x33, 0x3a, 0x31, 0x62, 0x3a, 0x34, 0x32, 0x3a, 0x32, 0x64, 0x3a, 0x35, 0x33, 0x3a, 0x39, 0x63, 0x3a, 0x66, 0x65, 0x3a, 0x31, 0x61, 0x3a, 0x66, 0x63, 0x3a, 0x37, 0x64, 0x3a, 0x61, 0x62, 0x3a, 0x37, 0x61, 0x3a, 0x33, 0x36, 0x3a, 0x33, 0x66, 0x3a, 0x39, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x65, 0x3a, 0x38, 0x34, 0x3a, 0x37, 0x63, 0x3a, 0x30, 0x35, 0x3a, 0x36, 0x37, 0x3a, 0x63, 0x65, 0x3a, 0x36, 0x61, 0x3a, 0x31, 0x34, 0x3a, 0x33, 0x38, 0x3a, 0x38, 0x37, 0x3a, 0x61, 0x39, 0x3a, 0x66, 0x31, 0x3a, 0x38, 0x63, 0x3a, 0x62, 0x35, 0x3a, 0x36, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x62, 0x3a, 0x36, 0x38, 0x3a, 0x37, 0x66, 0x3a, 0x37, 0x31, 0x3a, 0x32, 0x30, 0x3a, 0x32, 0x62, 0x3a, 0x66, 0x35, 0x3a, 0x61, 0x30, 0x3a, 0x36, 0x33, 0x3a, 0x66, 0x35, 0x3a, 0x35, 0x36, 0x3a, 0x32, 0x66, 0x3a, 0x61, 0x33, 0x3a, 0x32, 0x36, 0x3a, 0x64, 0x32, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x37, 0x3a, 0x36, 0x66, 0x3a, 0x62, 0x31, 0x3a, 0x35, 0x61, 0x3a, 0x31, 0x37, 0x3a, 0x64, 0x37, 0x3a, 0x33, 0x38, 0x3a, 0x39, 0x39, 0x3a, 0x30, 0x38, 0x3a, 0x66, 0x65, 0x3a, 0x39, 0x33, 0x3a, 0x35, 0x38, 0x3a, 0x36, 0x66, 0x3a, 0x66, 0x65, 0x3a, 0x63, 0x33, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x3a, 0x34, 0x39, 0x3a, 0x30, 0x38, 0x3a, 0x31, 0x36, 0x3a, 0x30, 0x62, 0x3a, 0x61, 0x37, 0x3a, 0x34, 0x64, 0x3a, 0x36, 0x37, 0x3a, 0x30, 0x30, 0x3a, 0x35, 0x32, 0x3a, 0x33, 0x31, 0x3a, 0x36, 0x37, 0x3a, 0x32, 0x33, 0x3a, 0x34, 0x65, 0x3a, 0x39, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x64, 0x3a, 0x35, 0x31, 0x3a, 0x34, 0x35, 0x3a, 0x31, 0x64, 0x3a, 0x62, 0x39, 0x3a, 0x30, 0x34, 0x3a, 0x64, 0x39, 0x3a, 0x30, 0x62, 0x3a, 0x65, 0x63, 0x3a, 0x64, 0x38, 0x3a, 0x32, 0x38, 0x3a, 0x62, 0x33, 0x3a, 0x34, 0x62, 0x3a, 0x62, 0x64, 0x3a, 0x65, 0x64, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x36, 0x3a, 0x37, 0x39, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x45, 0x78, 0x70, 0x6f, 0x6e, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x36, 0x35, 0x35, 0x33, 0x37, 0x20, 0x28, 0x30, 0x78, 0x31, 0x30, 0x30, 0x30, 0x31, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x58, 0x35, 0x30, 0x39, 0x76, 0x33, 0x20, 0x65, 0x78, 0x74, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x58, 0x35, 0x30, 0x39, 0x76, 0x33, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x49, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x66, 0x69, 0x65, 0x72, 0x3a, 0x20, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x32, 0x37, 0x3a, 0x38, 0x45, 0x3a, 0x36, 0x37, 0x3a, 0x31, 0x31, 0x3a, 0x37, 0x34, 0x3a, 0x43, 0x33, 0x3a, 0x32, 0x36, 0x3a, 0x31, 0x44, 0x3a, 0x33, 0x46, 0x3a, 0x45, 0x44, 0x3a, 0x33, 0x33, 0x3a, 0x36, 0x33, 0x3a, 0x42, 0x33, 0x3a, 0x41, 0x34, 0x3a, 0x44, 0x38, 0x3a, 0x31, 0x44, 0x3a, 0x33, 0x30, 0x3a, 0x45, 0x35, 0x3a, 0x45, 0x38, 0x3a, 0x44, 0x35, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x58, 0x35, 0x30, 0x39, 0x76, 0x33, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x74, 0x79, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x49, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x66, 0x69, 0x65, 0x72, 0x3a, 0x20, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6b, 0x65, 0x79, 0x69, 0x64, 0x3a, 0x32, 0x37, 0x3a, 0x38, 0x45, 0x3a, 0x36, 0x37, 0x3a, 0x31, 0x31, 0x3a, 0x37, 0x34, 0x3a, 0x43, 0x33, 0x3a, 0x32, 0x36, 0x3a, 0x31, 0x44, 0x3a, 0x33, 0x46, 0x3a, 0x45, 0x44, 0x3a, 0x33, 0x33, 0x3a, 0x36, 0x33, 0x3a, 0x42, 0x33, 0x3a, 0x41, 0x34, 0x3a, 0x44, 0x38, 0x3a, 0x31, 0x44, 0x3a, 0x33, 0x30, 0x3a, 0x45, 0x35, 0x3a, 0x45, 0x38, 0x3a, 0x44, 0x35, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x44, 0x69, 0x72, 0x4e, 0x61, 0x6d, 0x65, 0x3a, 0x2f, 0x43, 0x3d, 0x55, 0x53, 0x2f, 0x53, 0x54, 0x3d, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x2f, 0x4c, 0x3d, 0x42, 0x6f, 0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x2f, 0x4f, 0x3d, 0x53, 0x61, 0x77, 0x74, 0x6f, 0x6f, 0x74, 0x68, 0x2f, 0x4f, 0x55, 0x3d, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x2f, 0x43, 0x4e, 0x3d, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x65, 0x6d, 0x61, 0x69, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3d, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x3a, 0x38, 0x36, 0x3a, 0x46, 0x46, 0x3a, 0x46, 0x35, 0x3a, 0x38, 0x45, 0x3a, 0x31, 0x30, 0x3a, 0x44, 0x45, 0x3a, 0x42, 0x38, 0x3a, 0x46, 0x42, 0xa, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x58, 0x35, 0x30, 0x39, 0x76, 0x33, 0x20, 0x42, 0x61, 0x73, 0x69, 0x63, 0x20, 0x43, 0x6f, 0x6e, 0x73, 0x74, 0x72, 0x61, 0x69, 0x6e, 0x74, 0x73, 0x3a, 0x20, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x43, 0x41, 0x3a, 0x54, 0x52, 0x55, 0x45, 0xa, 0x20, 0x20, 0x20, 0x20, 0x53, 0x69, 0x67, 0x6e, 0x61, 0x74, 0x75, 0x72, 0x65, 0x20, 0x41, 0x6c, 0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x3a, 0x20, 0x73, 0x68, 0x61, 0x32, 0x35, 0x36, 0x57, 0x69, 0x74, 0x68, 0x52, 0x53, 0x41, 0x45, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x39, 0x65, 0x3a, 0x32, 0x38, 0x3a, 0x38, 0x38, 0x3a, 0x37, 0x32, 0x3a, 0x30, 0x30, 0x3a, 0x63, 0x61, 0x3a, 0x65, 0x36, 0x3a, 0x65, 0x37, 0x3a, 0x39, 0x37, 0x3a, 0x63, 0x61, 0x3a, 0x63, 0x31, 0x3a, 0x66, 0x31, 0x3a, 0x31, 0x66, 0x3a, 0x39, 0x65, 0x3a, 0x31, 0x32, 0x3a, 0x62, 0x32, 0x3a, 0x62, 0x38, 0x3a, 0x63, 0x37, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x35, 0x31, 0x3a, 0x65, 0x61, 0x3a, 0x32, 0x38, 0x3a, 0x65, 0x31, 0x3a, 0x33, 0x36, 0x3a, 0x62, 0x35, 0x3a, 0x32, 0x64, 0x3a, 0x65, 0x36, 0x3a, 0x32, 0x66, 0x3a, 0x30, 0x38, 0x3a, 0x32, 0x33, 0x3a, 0x63, 0x62, 0x3a, 0x61, 0x39, 0x3a, 0x34, 0x61, 0x3a, 0x38, 0x37, 0x3a, 0x32, 0x35, 0x3a, 0x63, 0x36, 0x3a, 0x35, 0x64, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x39, 0x3a, 0x34, 0x35, 0x3a, 0x65, 0x61, 0x3a, 0x66, 0x35, 0x3a, 0x30, 0x30, 0x3a, 0x39, 0x38, 0x3a, 0x61, 0x63, 0x3a, 0x37, 0x36, 0x3a, 0x66, 0x62, 0x3a, 0x31, 0x62, 0x3a, 0x61, 0x66, 0x3a, 0x66, 0x30, 0x3a, 0x63, 0x65, 0x3a, 0x36, 0x34, 0x3a, 0x39, 0x65, 0x3a, 0x64, 0x61, 0x3a, 0x30, 0x38, 0x3a, 0x62, 0x66, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x36, 0x3a, 0x65, 0x62, 0x3a, 0x62, 0x34, 0x3a, 0x62, 0x35, 0x3a, 0x30, 0x63, 0x3a, 0x61, 0x30, 0x3a, 0x65, 0x37, 0x3a, 0x66, 0x36, 0x3a, 0x34, 0x37, 0x3a, 0x35, 0x39, 0x3a, 0x31, 0x63, 0x3a, 0x36, 0x31, 0x3a, 0x63, 0x66, 0x3a, 0x32, 0x65, 0x3a, 0x30, 0x65, 0x3a, 0x35, 0x38, 0x3a, 0x61, 0x34, 0x3a, 0x38, 0x32, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x63, 0x3a, 0x30, 0x66, 0x3a, 0x33, 0x66, 0x3a, 0x65, 0x63, 0x3a, 0x63, 0x34, 0x3a, 0x61, 0x65, 0x3a, 0x38, 0x30, 0x3a, 0x66, 0x37, 0x3a, 0x62, 0x30, 0x3a, 0x38, 0x61, 0x3a, 0x31, 0x65, 0x3a, 0x38, 0x35, 0x3a, 0x34, 0x31, 0x3a, 0x65, 0x38, 0x3a, 0x66, 0x66, 0x3a, 0x66, 0x65, 0x3a, 0x66, 0x65, 0x3a, 0x34, 0x66, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x31, 0x61, 0x3a, 0x32, 0x34, 0x3a, 0x64, 0x35, 0x3a, 0x34, 0x39, 0x3a, 0x66, 0x61, 0x3a, 0x66, 0x62, 0x3a, 0x66, 0x65, 0x3a, 0x35, 0x65, 0x3a, 0x65, 0x35, 0x3a, 0x64, 0x33, 0x3a, 0x39, 0x31, 0x3a, 0x30, 0x65, 0x3a, 0x34, 0x66, 0x3a, 0x34, 0x65, 0x3a, 0x30, 0x63, 0x3a, 0x32, 0x31, 0x3a, 0x35, 0x31, 0x3a, 0x37, 0x31, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x33, 0x3a, 0x30, 0x34, 0x3a, 0x36, 0x62, 0x3a, 0x36, 0x32, 0x3a, 0x37, 0x62, 0x3a, 0x34, 0x66, 0x3a, 0x35, 0x39, 0x3a, 0x37, 0x36, 0x3a, 0x34, 0x38, 0x3a, 0x38, 0x31, 0x3a, 0x31, 0x65, 0x3a, 0x62, 0x34, 0x3a, 0x66, 0x37, 0x3a, 0x30, 0x34, 0x3a, 0x34, 0x37, 0x3a, 0x38, 0x61, 0x3a, 0x39, 0x31, 0x3a, 0x35, 0x37, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x33, 0x3a, 0x31, 0x31, 0x3a, 0x61, 0x39, 0x3a, 0x66, 0x32, 0x3a, 0x32, 0x30, 0x3a, 0x62, 0x34, 0x3a, 0x37, 0x38, 0x3a, 0x33, 0x33, 0x3a, 0x36, 0x32, 0x3a, 0x33, 0x64, 0x3a, 0x62, 0x30, 0x3a, 0x35, 0x65, 0x3a, 0x30, 0x64, 0x3a, 0x66, 0x39, 0x3a, 0x38, 0x36, 0x3a, 0x33, 0x38, 0x3a, 0x38, 0x32, 0x3a, 0x64, 0x61, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x31, 0x3a, 0x39, 0x38, 0x3a, 0x38, 0x64, 0x3a, 0x31, 0x39, 0x3a, 0x30, 0x36, 0x3a, 0x38, 0x37, 0x3a, 0x32, 0x31, 0x3a, 0x33, 0x39, 0x3a, 0x62, 0x37, 0x3a, 0x30, 0x32, 0x3a, 0x66, 0x37, 0x3a, 0x64, 0x61, 0x3a, 0x37, 0x64, 0x3a, 0x35, 0x38, 0x3a, 0x62, 0x61, 0x3a, 0x35, 0x32, 0x3a, 0x31, 0x35, 0x3a, 0x64, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x62, 0x3a, 0x63, 0x39, 0x3a, 0x37, 0x62, 0x3a, 0x35, 0x38, 0x3a, 0x33, 0x34, 0x3a, 0x61, 0x30, 0x3a, 0x63, 0x37, 0x3a, 0x65, 0x32, 0x3a, 0x37, 0x63, 0x3a, 0x61, 0x39, 0x3a, 0x38, 0x33, 0x3a, 0x31, 0x33, 0x3a, 0x65, 0x31, 0x3a, 0x62, 0x36, 0x3a, 0x65, 0x63, 0x3a, 0x30, 0x31, 0x3a, 0x62, 0x66, 0x3a, 0x35, 0x32, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x33, 0x3a, 0x30, 0x62, 0x3a, 0x63, 0x34, 0x3a, 0x66, 0x65, 0x3a, 0x34, 0x33, 0x3a, 0x64, 0x33, 0x3a, 0x63, 0x36, 0x3a, 0x61, 0x34, 0x3a, 0x38, 0x65, 0x3a, 0x32, 0x66, 0x3a, 0x38, 0x37, 0x3a, 0x37, 0x66, 0x3a, 0x37, 0x61, 0x3a, 0x34, 0x34, 0x3a, 0x65, 0x61, 0x3a, 0x63, 0x61, 0x3a, 0x35, 0x33, 0x3a, 0x36, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x35, 0x3a, 0x65, 0x64, 0x3a, 0x36, 0x35, 0x3a, 0x37, 0x36, 0x3a, 0x37, 0x33, 0x3a, 0x33, 0x31, 0x3a, 0x30, 0x33, 0x3a, 0x34, 0x65, 0x3a, 0x65, 0x61, 0x3a, 0x62, 0x64, 0x3a, 0x33, 0x35, 0x3a, 0x35, 0x34, 0x3a, 0x31, 0x33, 0x3a, 0x66, 0x33, 0x3a, 0x36, 0x34, 0x3a, 0x38, 0x37, 0x3a, 0x36, 0x62, 0x3a, 0x64, 0x66, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x34, 0x3a, 0x64, 0x64, 0x3a, 0x33, 0x34, 0x3a, 0x61, 0x31, 0x3a, 0x38, 0x38, 0x3a, 0x33, 0x62, 0x3a, 0x64, 0x62, 0x3a, 0x34, 0x64, 0x3a, 0x61, 0x66, 0x3a, 0x31, 0x62, 0x3a, 0x36, 0x34, 0x3a, 0x39, 0x30, 0x3a, 0x39, 0x32, 0x3a, 0x37, 0x31, 0x3a, 0x33, 0x30, 0x3a, 0x38, 0x65, 0x3a, 0x63, 0x38, 0x3a, 0x63, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x35, 0x3a, 0x36, 0x30, 0x3a, 0x32, 0x34, 0x3a, 0x61, 0x66, 0x3a, 0x33, 0x31, 0x3a, 0x31, 0x36, 0x3a, 0x33, 0x39, 0x3a, 0x33, 0x33, 0x3a, 0x39, 0x31, 0x3a, 0x35, 0x30, 0x3a, 0x66, 0x39, 0x3a, 0x61, 0x62, 0x3a, 0x36, 0x38, 0x3a, 0x34, 0x32, 0x3a, 0x37, 0x34, 0x3a, 0x37, 0x61, 0x3a, 0x33, 0x35, 0x3a, 0x64, 0x39, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x64, 0x3a, 0x63, 0x38, 0x3a, 0x63, 0x34, 0x3a, 0x35, 0x32, 0xa, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0xa, 0x4d, 0x49, 0x49, 0x45, 0x71, 0x6a, 0x43, 0x43, 0x41, 0x35, 0x4b, 0x67, 0x41, 0x77, 0x49, 0x42, 0x41, 0x67, 0x49, 0x4a, 0x41, 0x49, 0x62, 0x2f, 0x39, 0x59, 0x34, 0x51, 0x33, 0x72, 0x6a, 0x37, 0x4d, 0x41, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62, 0x33, 0x44, 0x51, 0x45, 0x42, 0x43, 0x77, 0x55, 0x41, 0x4d, 0x49, 0x47, 0x55, 0x4d, 0x51, 0x73, 0x77, 0x43, 0x51, 0x59, 0x44, 0xa, 0x56, 0x51, 0x51, 0x47, 0x45, 0x77, 0x4a, 0x56, 0x55, 0x7a, 0x45, 0x51, 0x4d, 0x41, 0x34, 0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x41, 0x77, 0x48, 0x54, 0x57, 0x39, 0x75, 0x64, 0x47, 0x46, 0x75, 0x59, 0x54, 0x45, 0x51, 0x4d, 0x41, 0x34, 0x47, 0x41, 0x31, 0x55, 0x45, 0x42, 0x77, 0x77, 0x48, 0x51, 0x6d, 0x39, 0x36, 0x5a, 0x57, 0x31, 0x68, 0x62, 0x6a, 0x45, 0x52, 0x4d, 0x41, 0x38, 0x47, 0xa, 0x41, 0x31, 0x55, 0x45, 0x43, 0x67, 0x77, 0x49, 0x55, 0x32, 0x46, 0x33, 0x64, 0x47, 0x39, 0x76, 0x64, 0x47, 0x67, 0x78, 0x45, 0x7a, 0x41, 0x52, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x73, 0x4d, 0x43, 0x6b, 0x4e, 0x76, 0x62, 0x6e, 0x4e, 0x31, 0x62, 0x48, 0x52, 0x70, 0x62, 0x6d, 0x63, 0x78, 0x47, 0x44, 0x41, 0x57, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x4d, 0x4d, 0x44, 0x33, 0x64, 0x33, 0xa, 0x64, 0x79, 0x35, 0x33, 0x62, 0x32, 0x78, 0x6d, 0x63, 0x33, 0x4e, 0x73, 0x4c, 0x6d, 0x4e, 0x76, 0x62, 0x54, 0x45, 0x66, 0x4d, 0x42, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62, 0x33, 0x44, 0x51, 0x45, 0x4a, 0x41, 0x52, 0x59, 0x51, 0x61, 0x57, 0x35, 0x6d, 0x62, 0x30, 0x42, 0x33, 0x62, 0x32, 0x78, 0x6d, 0x63, 0x33, 0x4e, 0x73, 0x4c, 0x6d, 0x4e, 0x76, 0x62, 0x54, 0x41, 0x65, 0xa, 0x46, 0x77, 0x30, 0x78, 0x4f, 0x44, 0x41, 0x30, 0x4d, 0x54, 0x4d, 0x78, 0x4e, 0x54, 0x49, 0x7a, 0x4d, 0x44, 0x6c, 0x61, 0x46, 0x77, 0x30, 0x79, 0x4d, 0x54, 0x41, 0x78, 0x4d, 0x44, 0x63, 0x78, 0x4e, 0x54, 0x49, 0x7a, 0x4d, 0x44, 0x6c, 0x61, 0x4d, 0x49, 0x47, 0x55, 0x4d, 0x51, 0x73, 0x77, 0x43, 0x51, 0x59, 0x44, 0x56, 0x51, 0x51, 0x47, 0x45, 0x77, 0x4a, 0x56, 0x55, 0x7a, 0x45, 0x51, 0xa, 0x4d, 0x41, 0x34, 0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x41, 0x77, 0x48, 0x54, 0x57, 0x39, 0x75, 0x64, 0x47, 0x46, 0x75, 0x59, 0x54, 0x45, 0x51, 0x4d, 0x41, 0x34, 0x47, 0x41, 0x31, 0x55, 0x45, 0x42, 0x77, 0x77, 0x48, 0x51, 0x6d, 0x39, 0x36, 0x5a, 0x57, 0x31, 0x68, 0x62, 0x6a, 0x45, 0x52, 0x4d, 0x41, 0x38, 0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x67, 0x77, 0x49, 0x55, 0x32, 0x46, 0x33, 0xa, 0x64, 0x47, 0x39, 0x76, 0x64, 0x47, 0x67, 0x78, 0x45, 0x7a, 0x41, 0x52, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x73, 0x4d, 0x43, 0x6b, 0x4e, 0x76, 0x62, 0x6e, 0x4e, 0x31, 0x62, 0x48, 0x52, 0x70, 0x62, 0x6d, 0x63, 0x78, 0x47, 0x44, 0x41, 0x57, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x4d, 0x4d, 0x44, 0x33, 0x64, 0x33, 0x64, 0x79, 0x35, 0x33, 0x62, 0x32, 0x78, 0x6d, 0x63, 0x33, 0x4e, 0x73, 0xa, 0x4c, 0x6d, 0x4e, 0x76, 0x62, 0x54, 0x45, 0x66, 0x4d, 0x42, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62, 0x33, 0x44, 0x51, 0x45, 0x4a, 0x41, 0x52, 0x59, 0x51, 0x61, 0x57, 0x35, 0x6d, 0x62, 0x30, 0x42, 0x33, 0x62, 0x32, 0x78, 0x6d, 0x63, 0x33, 0x4e, 0x73, 0x4c, 0x6d, 0x4e, 0x76, 0x62, 0x54, 0x43, 0x43, 0x41, 0x53, 0x49, 0x77, 0x44, 0x51, 0x59, 0x4a, 0x4b, 0x6f, 0x5a, 0x49, 0xa, 0x68, 0x76, 0x63, 0x4e, 0x41, 0x51, 0x45, 0x42, 0x42, 0x51, 0x41, 0x44, 0x67, 0x67, 0x45, 0x50, 0x41, 0x44, 0x43, 0x43, 0x41, 0x51, 0x6f, 0x43, 0x67, 0x67, 0x45, 0x42, 0x41, 0x4c, 0x38, 0x4d, 0x79, 0x69, 0x30, 0x55, 0x73, 0x68, 0x36, 0x45, 0x51, 0x6c, 0x76, 0x4e, 0x4f, 0x42, 0x39, 0x4b, 0x38, 0x6b, 0x31, 0x31, 0x45, 0x50, 0x47, 0x32, 0x4e, 0x5a, 0x2f, 0x66, 0x79, 0x6e, 0x30, 0x44, 0xa, 0x6d, 0x4e, 0x4f, 0x73, 0x33, 0x67, 0x4e, 0x6d, 0x37, 0x69, 0x72, 0x78, 0x32, 0x4c, 0x42, 0x39, 0x62, 0x67, 0x64, 0x55, 0x43, 0x78, 0x43, 0x59, 0x49, 0x55, 0x32, 0x41, 0x79, 0x78, 0x49, 0x67, 0x35, 0x38, 0x78, 0x50, 0x33, 0x6b, 0x56, 0x39, 0x79, 0x58, 0x4a, 0x33, 0x4d, 0x75, 0x72, 0x4b, 0x6b, 0x4c, 0x74, 0x70, 0x55, 0x68, 0x41, 0x44, 0x4c, 0x36, 0x6a, 0x7a, 0x6c, 0x63, 0x58, 0x78, 0xa, 0x69, 0x32, 0x4a, 0x57, 0x47, 0x2b, 0x39, 0x6e, 0x62, 0x36, 0x51, 0x51, 0x51, 0x5a, 0x57, 0x74, 0x43, 0x70, 0x76, 0x6a, 0x70, 0x63, 0x43, 0x77, 0x30, 0x6e, 0x42, 0x32, 0x55, 0x44, 0x42, 0x62, 0x71, 0x4f, 0x67, 0x49, 0x4c, 0x48, 0x7a, 0x74, 0x70, 0x36, 0x4a, 0x36, 0x6a, 0x54, 0x67, 0x70, 0x48, 0x4b, 0x7a, 0x48, 0x37, 0x66, 0x4a, 0x38, 0x6c, 0x62, 0x43, 0x56, 0x67, 0x6e, 0x31, 0x4a, 0xa, 0x58, 0x44, 0x6a, 0x4e, 0x64, 0x79, 0x58, 0x76, 0x76, 0x59, 0x42, 0x31, 0x55, 0x35, 0x51, 0x38, 0x50, 0x63, 0x70, 0x6a, 0x57, 0x35, 0x38, 0x56, 0x74, 0x64, 0x4d, 0x64, 0x45, 0x79, 0x38, 0x5a, 0x30, 0x54, 0x7a, 0x62, 0x64, 0x6a, 0x72, 0x4d, 0x75, 0x48, 0x33, 0x4a, 0x35, 0x63, 0x4c, 0x58, 0x32, 0x6b, 0x42, 0x76, 0x32, 0x43, 0x48, 0x63, 0x63, 0x78, 0x74, 0x43, 0x4c, 0x56, 0x4f, 0x63, 0xa, 0x2f, 0x68, 0x72, 0x38, 0x66, 0x61, 0x74, 0x36, 0x4e, 0x6a, 0x2b, 0x59, 0x33, 0x6f, 0x52, 0x38, 0x42, 0x57, 0x66, 0x4f, 0x61, 0x68, 0x51, 0x34, 0x68, 0x36, 0x6e, 0x78, 0x6a, 0x4c, 0x56, 0x6f, 0x79, 0x32, 0x68, 0x2f, 0x63, 0x53, 0x41, 0x72, 0x39, 0x61, 0x42, 0x6a, 0x39, 0x56, 0x59, 0x76, 0x6f, 0x79, 0x62, 0x53, 0x74, 0x32, 0x2b, 0x78, 0x57, 0x68, 0x66, 0x58, 0x4f, 0x4a, 0x6b, 0x49, 0xa, 0x2f, 0x70, 0x4e, 0x59, 0x62, 0x2f, 0x37, 0x44, 0x45, 0x30, 0x6b, 0x49, 0x46, 0x67, 0x75, 0x6e, 0x54, 0x57, 0x63, 0x41, 0x55, 0x6a, 0x46, 0x6e, 0x49, 0x30, 0x36, 0x59, 0x37, 0x56, 0x46, 0x46, 0x48, 0x62, 0x6b, 0x45, 0x32, 0x51, 0x76, 0x73, 0x32, 0x43, 0x69, 0x7a, 0x53, 0x37, 0x33, 0x74, 0x4e, 0x6e, 0x6b, 0x43, 0x41, 0x77, 0x45, 0x41, 0x41, 0x61, 0x4f, 0x42, 0x2f, 0x44, 0x43, 0x42, 0xa, 0x2b, 0x54, 0x41, 0x64, 0x42, 0x67, 0x4e, 0x56, 0x48, 0x51, 0x34, 0x45, 0x46, 0x67, 0x51, 0x55, 0x4a, 0x34, 0x35, 0x6e, 0x45, 0x58, 0x54, 0x44, 0x4a, 0x68, 0x30, 0x2f, 0x37, 0x54, 0x4e, 0x6a, 0x73, 0x36, 0x54, 0x59, 0x48, 0x54, 0x44, 0x6c, 0x36, 0x4e, 0x55, 0x77, 0x67, 0x63, 0x6b, 0x47, 0x41, 0x31, 0x55, 0x64, 0x49, 0x77, 0x53, 0x42, 0x77, 0x54, 0x43, 0x42, 0x76, 0x6f, 0x41, 0x55, 0xa, 0x4a, 0x34, 0x35, 0x6e, 0x45, 0x58, 0x54, 0x44, 0x4a, 0x68, 0x30, 0x2f, 0x37, 0x54, 0x4e, 0x6a, 0x73, 0x36, 0x54, 0x59, 0x48, 0x54, 0x44, 0x6c, 0x36, 0x4e, 0x57, 0x68, 0x67, 0x5a, 0x71, 0x6b, 0x67, 0x5a, 0x63, 0x77, 0x67, 0x5a, 0x51, 0x78, 0x43, 0x7a, 0x41, 0x4a, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x59, 0x54, 0x41, 0x6c, 0x56, 0x54, 0x4d, 0x52, 0x41, 0x77, 0x44, 0x67, 0x59, 0x44, 0xa, 0x56, 0x51, 0x51, 0x49, 0x44, 0x41, 0x64, 0x4e, 0x62, 0x32, 0x35, 0x30, 0x59, 0x57, 0x35, 0x68, 0x4d, 0x52, 0x41, 0x77, 0x44, 0x67, 0x59, 0x44, 0x56, 0x51, 0x51, 0x48, 0x44, 0x41, 0x64, 0x43, 0x62, 0x33, 0x70, 0x6c, 0x62, 0x57, 0x46, 0x75, 0x4d, 0x52, 0x45, 0x77, 0x44, 0x77, 0x59, 0x44, 0x56, 0x51, 0x51, 0x4b, 0x44, 0x41, 0x68, 0x54, 0x59, 0x58, 0x64, 0x30, 0x62, 0x32, 0x39, 0x30, 0xa, 0x61, 0x44, 0x45, 0x54, 0x4d, 0x42, 0x45, 0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x77, 0x77, 0x4b, 0x51, 0x32, 0x39, 0x75, 0x63, 0x33, 0x56, 0x73, 0x64, 0x47, 0x6c, 0x75, 0x5a, 0x7a, 0x45, 0x59, 0x4d, 0x42, 0x59, 0x47, 0x41, 0x31, 0x55, 0x45, 0x41, 0x77, 0x77, 0x50, 0x64, 0x33, 0x64, 0x33, 0x4c, 0x6e, 0x64, 0x76, 0x62, 0x47, 0x5a, 0x7a, 0x63, 0x32, 0x77, 0x75, 0x59, 0x32, 0x39, 0x74, 0xa, 0x4d, 0x52, 0x38, 0x77, 0x48, 0x51, 0x59, 0x4a, 0x4b, 0x6f, 0x5a, 0x49, 0x68, 0x76, 0x63, 0x4e, 0x41, 0x51, 0x6b, 0x42, 0x46, 0x68, 0x42, 0x70, 0x62, 0x6d, 0x5a, 0x76, 0x51, 0x48, 0x64, 0x76, 0x62, 0x47, 0x5a, 0x7a, 0x63, 0x32, 0x77, 0x75, 0x59, 0x32, 0x39, 0x74, 0x67, 0x67, 0x6b, 0x41, 0x68, 0x76, 0x2f, 0x31, 0x6a, 0x68, 0x44, 0x65, 0x75, 0x50, 0x73, 0x77, 0x44, 0x41, 0x59, 0x44, 0xa, 0x56, 0x52, 0x30, 0x54, 0x42, 0x41, 0x55, 0x77, 0x41, 0x77, 0x45, 0x42, 0x2f, 0x7a, 0x41, 0x4e, 0x42, 0x67, 0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77, 0x30, 0x42, 0x41, 0x51, 0x73, 0x46, 0x41, 0x41, 0x4f, 0x43, 0x41, 0x51, 0x45, 0x41, 0x6e, 0x69, 0x69, 0x49, 0x63, 0x67, 0x44, 0x4b, 0x35, 0x75, 0x65, 0x58, 0x79, 0x73, 0x48, 0x78, 0x48, 0x35, 0x34, 0x53, 0x73, 0x72, 0x6a, 0x48, 0xa, 0x55, 0x65, 0x6f, 0x6f, 0x34, 0x54, 0x61, 0x31, 0x4c, 0x65, 0x59, 0x76, 0x43, 0x43, 0x50, 0x4c, 0x71, 0x55, 0x71, 0x48, 0x4a, 0x63, 0x5a, 0x64, 0x69, 0x55, 0x58, 0x71, 0x39, 0x51, 0x43, 0x59, 0x72, 0x48, 0x62, 0x37, 0x47, 0x36, 0x2f, 0x77, 0x7a, 0x6d, 0x53, 0x65, 0x32, 0x67, 0x69, 0x2f, 0x74, 0x75, 0x75, 0x30, 0x74, 0x51, 0x79, 0x67, 0x35, 0x2f, 0x5a, 0x48, 0x57, 0x52, 0x78, 0x68, 0xa, 0x7a, 0x79, 0x34, 0x4f, 0x57, 0x4b, 0x53, 0x43, 0x72, 0x41, 0x38, 0x2f, 0x37, 0x4d, 0x53, 0x75, 0x67, 0x50, 0x65, 0x77, 0x69, 0x68, 0x36, 0x46, 0x51, 0x65, 0x6a, 0x2f, 0x2f, 0x76, 0x35, 0x50, 0x47, 0x69, 0x54, 0x56, 0x53, 0x66, 0x72, 0x37, 0x2f, 0x6c, 0x37, 0x6c, 0x30, 0x35, 0x45, 0x4f, 0x54, 0x30, 0x34, 0x4d, 0x49, 0x56, 0x46, 0x78, 0x67, 0x77, 0x52, 0x72, 0x59, 0x6e, 0x74, 0x50, 0xa, 0x57, 0x58, 0x5a, 0x49, 0x67, 0x52, 0x36, 0x30, 0x39, 0x77, 0x52, 0x48, 0x69, 0x70, 0x46, 0x58, 0x6f, 0x78, 0x47, 0x70, 0x38, 0x69, 0x43, 0x30, 0x65, 0x44, 0x4e, 0x69, 0x50, 0x62, 0x42, 0x65, 0x44, 0x66, 0x6d, 0x47, 0x4f, 0x49, 0x4c, 0x61, 0x6f, 0x5a, 0x69, 0x4e, 0x47, 0x51, 0x61, 0x48, 0x49, 0x54, 0x6d, 0x33, 0x41, 0x76, 0x66, 0x61, 0x66, 0x56, 0x69, 0x36, 0x55, 0x68, 0x58, 0x59, 0xa, 0x4f, 0x38, 0x6c, 0x37, 0x57, 0x44, 0x53, 0x67, 0x78, 0x2b, 0x4a, 0x38, 0x71, 0x59, 0x4d, 0x54, 0x34, 0x62, 0x62, 0x73, 0x41, 0x62, 0x39, 0x53, 0x4d, 0x77, 0x76, 0x45, 0x2f, 0x6b, 0x50, 0x54, 0x78, 0x71, 0x53, 0x4f, 0x4c, 0x34, 0x64, 0x2f, 0x65, 0x6b, 0x54, 0x71, 0x79, 0x6c, 0x4e, 0x73, 0x68, 0x65, 0x31, 0x6c, 0x64, 0x6e, 0x4d, 0x78, 0x41, 0x30, 0x37, 0x71, 0x76, 0x54, 0x56, 0x55, 0xa, 0x45, 0x2f, 0x4e, 0x6b, 0x68, 0x32, 0x76, 0x66, 0x4e, 0x4e, 0x30, 0x30, 0x6f, 0x59, 0x67, 0x37, 0x32, 0x30, 0x32, 0x76, 0x47, 0x32, 0x53, 0x51, 0x6b, 0x6e, 0x45, 0x77, 0x6a, 0x73, 0x6a, 0x4d, 0x35, 0x57, 0x41, 0x6b, 0x72, 0x7a, 0x45, 0x57, 0x4f, 0x54, 0x4f, 0x52, 0x55, 0x50, 0x6d, 0x72, 0x61, 0x45, 0x4a, 0x30, 0x65, 0x6a, 0x58, 0x5a, 0x33, 0x63, 0x6a, 0x45, 0x55, 0x67, 0x3d, 0x3d, 0xa, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0xa };
    std::array<u8, 0x26ef> sample_server_cert_pem = { 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x44, 0x61, 0x74, 0x61, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x3a, 0x20, 0x33, 0x20, 0x28, 0x30, 0x78, 0x32, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x53, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x20, 0x4e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x3a, 0x20, 0x31, 0x20, 0x28, 0x30, 0x78, 0x31, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x53, 0x69, 0x67, 0x6e, 0x61, 0x74, 0x75, 0x72, 0x65, 0x20, 0x41, 0x6c, 0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x3a, 0x20, 0x73, 0x68, 0x61, 0x32, 0x35, 0x36, 0x57, 0x69, 0x74, 0x68, 0x52, 0x53, 0x41, 0x45, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x49, 0x73, 0x73, 0x75, 0x65, 0x72, 0x3a, 0x20, 0x43, 0x3d, 0x55, 0x53, 0x2c, 0x20, 0x53, 0x54, 0x3d, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x2c, 0x20, 0x4c, 0x3d, 0x42, 0x6f, 0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x2c, 0x20, 0x4f, 0x3d, 0x53, 0x61, 0x77, 0x74, 0x6f, 0x6f, 0x74, 0x68, 0x2c, 0x20, 0x4f, 0x55, 0x3d, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x43, 0x4e, 0x3d, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x65, 0x6d, 0x61, 0x69, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3d, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x56, 0x61, 0x6c, 0x69, 0x64, 0x69, 0x74, 0x79, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x4e, 0x6f, 0x74, 0x20, 0x42, 0x65, 0x66, 0x6f, 0x72, 0x65, 0x3a, 0x20, 0x41, 0x70, 0x72, 0x20, 0x31, 0x33, 0x20, 0x31, 0x35, 0x3a, 0x32, 0x33, 0x3a, 0x31, 0x30, 0x20, 0x32, 0x30, 0x31, 0x38, 0x20, 0x47, 0x4d, 0x54, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x4e, 0x6f, 0x74, 0x20, 0x41, 0x66, 0x74, 0x65, 0x72, 0x20, 0x3a, 0x20, 0x4a, 0x61, 0x6e, 0x20, 0x20, 0x37, 0x20, 0x31, 0x35, 0x3a, 0x32, 0x33, 0x3a, 0x31, 0x30, 0x20, 0x32, 0x30, 0x32, 0x31, 0x20, 0x47, 0x4d, 0x54, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x3a, 0x20, 0x43, 0x3d, 0x55, 0x53, 0x2c, 0x20, 0x53, 0x54, 0x3d, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x2c, 0x20, 0x4c, 0x3d, 0x42, 0x6f, 0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x2c, 0x20, 0x4f, 0x3d, 0x77, 0x6f, 0x6c, 0x66, 0x53, 0x53, 0x4c, 0x2c, 0x20, 0x4f, 0x55, 0x3d, 0x53, 0x75, 0x70, 0x70, 0x6f, 0x72, 0x74, 0x2c, 0x20, 0x43, 0x4e, 0x3d, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x65, 0x6d, 0x61, 0x69, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3d, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x49, 0x6e, 0x66, 0x6f, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x41, 0x6c, 0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x3a, 0x20, 0x72, 0x73, 0x61, 0x45, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x2d, 0x4b, 0x65, 0x79, 0x3a, 0x20, 0x28, 0x32, 0x30, 0x34, 0x38, 0x20, 0x62, 0x69, 0x74, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x4d, 0x6f, 0x64, 0x75, 0x6c, 0x75, 0x73, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x30, 0x30, 0x3a, 0x63, 0x30, 0x3a, 0x39, 0x35, 0x3a, 0x30, 0x38, 0x3a, 0x65, 0x31, 0x3a, 0x35, 0x37, 0x3a, 0x34, 0x31, 0x3a, 0x66, 0x32, 0x3a, 0x37, 0x31, 0x3a, 0x36, 0x64, 0x3a, 0x62, 0x37, 0x3a, 0x64, 0x32, 0x3a, 0x34, 0x35, 0x3a, 0x34, 0x31, 0x3a, 0x32, 0x37, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x30, 0x31, 0x3a, 0x36, 0x35, 0x3a, 0x63, 0x36, 0x3a, 0x34, 0x35, 0x3a, 0x61, 0x65, 0x3a, 0x66, 0x32, 0x3a, 0x62, 0x63, 0x3a, 0x32, 0x34, 0x3a, 0x33, 0x30, 0x3a, 0x62, 0x38, 0x3a, 0x39, 0x35, 0x3a, 0x63, 0x65, 0x3a, 0x32, 0x66, 0x3a, 0x34, 0x65, 0x3a, 0x64, 0x36, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x36, 0x3a, 0x31, 0x63, 0x3a, 0x38, 0x38, 0x3a, 0x62, 0x63, 0x3a, 0x37, 0x63, 0x3a, 0x39, 0x66, 0x3a, 0x66, 0x62, 0x3a, 0x61, 0x38, 0x3a, 0x36, 0x37, 0x3a, 0x37, 0x66, 0x3a, 0x66, 0x65, 0x3a, 0x35, 0x63, 0x3a, 0x39, 0x63, 0x3a, 0x35, 0x31, 0x3a, 0x37, 0x35, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x37, 0x3a, 0x38, 0x61, 0x3a, 0x63, 0x61, 0x3a, 0x30, 0x37, 0x3a, 0x65, 0x37, 0x3a, 0x33, 0x35, 0x3a, 0x32, 0x66, 0x3a, 0x38, 0x66, 0x3a, 0x65, 0x31, 0x3a, 0x62, 0x64, 0x3a, 0x37, 0x62, 0x3a, 0x63, 0x30, 0x3a, 0x32, 0x66, 0x3a, 0x37, 0x63, 0x3a, 0x61, 0x62, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x36, 0x34, 0x3a, 0x61, 0x38, 0x3a, 0x31, 0x37, 0x3a, 0x66, 0x63, 0x3a, 0x63, 0x61, 0x3a, 0x35, 0x64, 0x3a, 0x37, 0x62, 0x3a, 0x62, 0x61, 0x3a, 0x65, 0x30, 0x3a, 0x32, 0x31, 0x3a, 0x65, 0x35, 0x3a, 0x37, 0x32, 0x3a, 0x32, 0x65, 0x3a, 0x36, 0x66, 0x3a, 0x32, 0x65, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x36, 0x3a, 0x64, 0x38, 0x3a, 0x39, 0x35, 0x3a, 0x37, 0x33, 0x3a, 0x64, 0x61, 0x3a, 0x61, 0x63, 0x3a, 0x31, 0x62, 0x3a, 0x35, 0x33, 0x3a, 0x62, 0x39, 0x3a, 0x35, 0x66, 0x3a, 0x33, 0x66, 0x3a, 0x64, 0x37, 0x3a, 0x31, 0x39, 0x3a, 0x30, 0x64, 0x3a, 0x32, 0x35, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x34, 0x66, 0x3a, 0x65, 0x31, 0x3a, 0x36, 0x33, 0x3a, 0x36, 0x33, 0x3a, 0x35, 0x31, 0x3a, 0x38, 0x62, 0x3a, 0x30, 0x62, 0x3a, 0x36, 0x34, 0x3a, 0x33, 0x66, 0x3a, 0x61, 0x64, 0x3a, 0x34, 0x33, 0x3a, 0x62, 0x38, 0x3a, 0x61, 0x35, 0x3a, 0x31, 0x63, 0x3a, 0x35, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x34, 0x3a, 0x62, 0x33, 0x3a, 0x61, 0x65, 0x3a, 0x30, 0x30, 0x3a, 0x61, 0x30, 0x3a, 0x36, 0x33, 0x3a, 0x63, 0x35, 0x3a, 0x66, 0x36, 0x3a, 0x37, 0x66, 0x3a, 0x30, 0x62, 0x3a, 0x35, 0x39, 0x3a, 0x36, 0x38, 0x3a, 0x37, 0x38, 0x3a, 0x37, 0x33, 0x3a, 0x61, 0x36, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x63, 0x3a, 0x31, 0x38, 0x3a, 0x61, 0x39, 0x3a, 0x30, 0x32, 0x3a, 0x36, 0x64, 0x3a, 0x61, 0x66, 0x3a, 0x63, 0x33, 0x3a, 0x31, 0x39, 0x3a, 0x30, 0x31, 0x3a, 0x32, 0x65, 0x3a, 0x62, 0x38, 0x3a, 0x31, 0x30, 0x3a, 0x65, 0x33, 0x3a, 0x63, 0x36, 0x3a, 0x63, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x34, 0x30, 0x3a, 0x62, 0x34, 0x3a, 0x36, 0x39, 0x3a, 0x61, 0x33, 0x3a, 0x34, 0x36, 0x3a, 0x33, 0x33, 0x3a, 0x36, 0x39, 0x3a, 0x38, 0x37, 0x3a, 0x36, 0x65, 0x3a, 0x63, 0x34, 0x3a, 0x62, 0x62, 0x3a, 0x31, 0x37, 0x3a, 0x61, 0x36, 0x3a, 0x66, 0x33, 0x3a, 0x65, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x64, 0x3a, 0x61, 0x64, 0x3a, 0x37, 0x33, 0x3a, 0x62, 0x63, 0x3a, 0x37, 0x62, 0x3a, 0x32, 0x66, 0x3a, 0x32, 0x31, 0x3a, 0x62, 0x35, 0x3a, 0x66, 0x64, 0x3a, 0x36, 0x36, 0x3a, 0x35, 0x31, 0x3a, 0x30, 0x63, 0x3a, 0x62, 0x64, 0x3a, 0x35, 0x34, 0x3a, 0x62, 0x33, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x31, 0x3a, 0x36, 0x64, 0x3a, 0x35, 0x66, 0x3a, 0x31, 0x63, 0x3a, 0x62, 0x63, 0x3a, 0x32, 0x33, 0x3a, 0x37, 0x33, 0x3a, 0x64, 0x31, 0x3a, 0x30, 0x39, 0x3a, 0x30, 0x33, 0x3a, 0x38, 0x39, 0x3a, 0x31, 0x34, 0x3a, 0x64, 0x32, 0x3a, 0x31, 0x30, 0x3a, 0x62, 0x39, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x36, 0x34, 0x3a, 0x63, 0x33, 0x3a, 0x32, 0x61, 0x3a, 0x64, 0x30, 0x3a, 0x61, 0x31, 0x3a, 0x39, 0x36, 0x3a, 0x34, 0x61, 0x3a, 0x62, 0x63, 0x3a, 0x65, 0x31, 0x3a, 0x64, 0x34, 0x3a, 0x31, 0x61, 0x3a, 0x35, 0x62, 0x3a, 0x63, 0x37, 0x3a, 0x61, 0x30, 0x3a, 0x63, 0x30, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x31, 0x3a, 0x36, 0x33, 0x3a, 0x37, 0x38, 0x3a, 0x30, 0x66, 0x3a, 0x34, 0x34, 0x3a, 0x33, 0x37, 0x3a, 0x33, 0x30, 0x3a, 0x33, 0x32, 0x3a, 0x39, 0x36, 0x3a, 0x38, 0x30, 0x3a, 0x33, 0x32, 0x3a, 0x32, 0x33, 0x3a, 0x39, 0x35, 0x3a, 0x61, 0x31, 0x3a, 0x37, 0x37, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x61, 0x3a, 0x31, 0x33, 0x3a, 0x64, 0x32, 0x3a, 0x39, 0x37, 0x3a, 0x37, 0x33, 0x3a, 0x65, 0x32, 0x3a, 0x35, 0x64, 0x3a, 0x32, 0x35, 0x3a, 0x63, 0x39, 0x3a, 0x36, 0x61, 0x3a, 0x30, 0x64, 0x3a, 0x63, 0x33, 0x3a, 0x33, 0x39, 0x3a, 0x36, 0x30, 0x3a, 0x61, 0x34, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x34, 0x3a, 0x62, 0x30, 0x3a, 0x36, 0x39, 0x3a, 0x34, 0x32, 0x3a, 0x34, 0x32, 0x3a, 0x30, 0x39, 0x3a, 0x65, 0x39, 0x3a, 0x64, 0x38, 0x3a, 0x30, 0x38, 0x3a, 0x62, 0x63, 0x3a, 0x33, 0x33, 0x3a, 0x32, 0x30, 0x3a, 0x62, 0x33, 0x3a, 0x35, 0x38, 0x3a, 0x32, 0x32, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x37, 0x3a, 0x61, 0x61, 0x3a, 0x65, 0x62, 0x3a, 0x63, 0x34, 0x3a, 0x65, 0x31, 0x3a, 0x65, 0x36, 0x3a, 0x36, 0x31, 0x3a, 0x38, 0x33, 0x3a, 0x63, 0x35, 0x3a, 0x64, 0x32, 0x3a, 0x39, 0x36, 0x3a, 0x64, 0x66, 0x3a, 0x64, 0x39, 0x3a, 0x64, 0x30, 0x3a, 0x34, 0x66, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x64, 0x3a, 0x64, 0x37, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x45, 0x78, 0x70, 0x6f, 0x6e, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x36, 0x35, 0x35, 0x33, 0x37, 0x20, 0x28, 0x30, 0x78, 0x31, 0x30, 0x30, 0x30, 0x31, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x58, 0x35, 0x30, 0x39, 0x76, 0x33, 0x20, 0x65, 0x78, 0x74, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x58, 0x35, 0x30, 0x39, 0x76, 0x33, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x49, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x66, 0x69, 0x65, 0x72, 0x3a, 0x20, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x42, 0x33, 0x3a, 0x31, 0x31, 0x3a, 0x33, 0x32, 0x3a, 0x43, 0x39, 0x3a, 0x39, 0x32, 0x3a, 0x39, 0x38, 0x3a, 0x38, 0x34, 0x3a, 0x45, 0x32, 0x3a, 0x43, 0x39, 0x3a, 0x46, 0x38, 0x3a, 0x44, 0x30, 0x3a, 0x33, 0x42, 0x3a, 0x36, 0x45, 0x3a, 0x30, 0x33, 0x3a, 0x34, 0x32, 0x3a, 0x43, 0x41, 0x3a, 0x31, 0x46, 0x3a, 0x30, 0x45, 0x3a, 0x38, 0x45, 0x3a, 0x33, 0x43, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x58, 0x35, 0x30, 0x39, 0x76, 0x33, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x74, 0x79, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x49, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x66, 0x69, 0x65, 0x72, 0x3a, 0x20, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6b, 0x65, 0x79, 0x69, 0x64, 0x3a, 0x32, 0x37, 0x3a, 0x38, 0x45, 0x3a, 0x36, 0x37, 0x3a, 0x31, 0x31, 0x3a, 0x37, 0x34, 0x3a, 0x43, 0x33, 0x3a, 0x32, 0x36, 0x3a, 0x31, 0x44, 0x3a, 0x33, 0x46, 0x3a, 0x45, 0x44, 0x3a, 0x33, 0x33, 0x3a, 0x36, 0x33, 0x3a, 0x42, 0x33, 0x3a, 0x41, 0x34, 0x3a, 0x44, 0x38, 0x3a, 0x31, 0x44, 0x3a, 0x33, 0x30, 0x3a, 0x45, 0x35, 0x3a, 0x45, 0x38, 0x3a, 0x44, 0x35, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x44, 0x69, 0x72, 0x4e, 0x61, 0x6d, 0x65, 0x3a, 0x2f, 0x43, 0x3d, 0x55, 0x53, 0x2f, 0x53, 0x54, 0x3d, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x2f, 0x4c, 0x3d, 0x42, 0x6f, 0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x2f, 0x4f, 0x3d, 0x53, 0x61, 0x77, 0x74, 0x6f, 0x6f, 0x74, 0x68, 0x2f, 0x4f, 0x55, 0x3d, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x2f, 0x43, 0x4e, 0x3d, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x65, 0x6d, 0x61, 0x69, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3d, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x3a, 0x38, 0x36, 0x3a, 0x46, 0x46, 0x3a, 0x46, 0x35, 0x3a, 0x38, 0x45, 0x3a, 0x31, 0x30, 0x3a, 0x44, 0x45, 0x3a, 0x42, 0x38, 0x3a, 0x46, 0x42, 0xa, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x58, 0x35, 0x30, 0x39, 0x76, 0x33, 0x20, 0x42, 0x61, 0x73, 0x69, 0x63, 0x20, 0x43, 0x6f, 0x6e, 0x73, 0x74, 0x72, 0x61, 0x69, 0x6e, 0x74, 0x73, 0x3a, 0x20, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x43, 0x41, 0x3a, 0x54, 0x52, 0x55, 0x45, 0xa, 0x20, 0x20, 0x20, 0x20, 0x53, 0x69, 0x67, 0x6e, 0x61, 0x74, 0x75, 0x72, 0x65, 0x20, 0x41, 0x6c, 0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x3a, 0x20, 0x73, 0x68, 0x61, 0x32, 0x35, 0x36, 0x57, 0x69, 0x74, 0x68, 0x52, 0x53, 0x41, 0x45, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x34, 0x3a, 0x35, 0x34, 0x3a, 0x36, 0x30, 0x3a, 0x61, 0x64, 0x3a, 0x61, 0x30, 0x3a, 0x30, 0x33, 0x3a, 0x33, 0x32, 0x3a, 0x64, 0x65, 0x3a, 0x30, 0x32, 0x3a, 0x37, 0x66, 0x3a, 0x32, 0x31, 0x3a, 0x34, 0x61, 0x3a, 0x38, 0x31, 0x3a, 0x63, 0x36, 0x3a, 0x65, 0x64, 0x3a, 0x63, 0x64, 0x3a, 0x63, 0x64, 0x3a, 0x64, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x31, 0x32, 0x3a, 0x38, 0x61, 0x3a, 0x63, 0x30, 0x3a, 0x62, 0x61, 0x3a, 0x38, 0x32, 0x3a, 0x35, 0x62, 0x3a, 0x37, 0x35, 0x3a, 0x61, 0x64, 0x3a, 0x35, 0x34, 0x3a, 0x65, 0x33, 0x3a, 0x37, 0x63, 0x3a, 0x38, 0x30, 0x3a, 0x36, 0x61, 0x3a, 0x61, 0x63, 0x3a, 0x32, 0x65, 0x3a, 0x36, 0x63, 0x3a, 0x32, 0x30, 0x3a, 0x34, 0x65, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x65, 0x3a, 0x34, 0x64, 0x3a, 0x38, 0x32, 0x3a, 0x61, 0x37, 0x3a, 0x34, 0x37, 0x3a, 0x31, 0x33, 0x3a, 0x35, 0x63, 0x3a, 0x66, 0x34, 0x3a, 0x63, 0x36, 0x3a, 0x36, 0x61, 0x3a, 0x32, 0x62, 0x3a, 0x31, 0x30, 0x3a, 0x39, 0x39, 0x3a, 0x35, 0x38, 0x3a, 0x64, 0x65, 0x3a, 0x61, 0x62, 0x3a, 0x36, 0x62, 0x3a, 0x37, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x32, 0x32, 0x3a, 0x30, 0x35, 0x3a, 0x63, 0x31, 0x3a, 0x38, 0x33, 0x3a, 0x39, 0x64, 0x3a, 0x63, 0x62, 0x3a, 0x66, 0x66, 0x3a, 0x33, 0x63, 0x3a, 0x65, 0x34, 0x3a, 0x32, 0x64, 0x3a, 0x35, 0x37, 0x3a, 0x36, 0x61, 0x3a, 0x61, 0x36, 0x3a, 0x39, 0x36, 0x3a, 0x64, 0x66, 0x3a, 0x64, 0x33, 0x3a, 0x63, 0x31, 0x3a, 0x36, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x33, 0x3a, 0x64, 0x32, 0x3a, 0x63, 0x36, 0x3a, 0x38, 0x33, 0x3a, 0x34, 0x62, 0x3a, 0x39, 0x37, 0x3a, 0x65, 0x32, 0x3a, 0x63, 0x36, 0x3a, 0x33, 0x32, 0x3a, 0x30, 0x65, 0x3a, 0x62, 0x65, 0x3a, 0x63, 0x34, 0x3a, 0x30, 0x33, 0x3a, 0x62, 0x39, 0x3a, 0x30, 0x37, 0x3a, 0x38, 0x61, 0x3a, 0x35, 0x62, 0x3a, 0x62, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x34, 0x3a, 0x62, 0x61, 0x3a, 0x63, 0x35, 0x3a, 0x33, 0x39, 0x3a, 0x33, 0x66, 0x3a, 0x31, 0x63, 0x3a, 0x35, 0x38, 0x3a, 0x61, 0x37, 0x3a, 0x35, 0x35, 0x3a, 0x64, 0x37, 0x3a, 0x66, 0x30, 0x3a, 0x39, 0x62, 0x3a, 0x65, 0x38, 0x3a, 0x64, 0x32, 0x3a, 0x34, 0x35, 0x3a, 0x62, 0x39, 0x3a, 0x65, 0x33, 0x3a, 0x38, 0x33, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x32, 0x65, 0x3a, 0x65, 0x65, 0x3a, 0x62, 0x36, 0x3a, 0x37, 0x31, 0x3a, 0x35, 0x36, 0x3a, 0x62, 0x39, 0x3a, 0x33, 0x61, 0x3a, 0x65, 0x65, 0x3a, 0x33, 0x66, 0x3a, 0x32, 0x37, 0x3a, 0x64, 0x38, 0x3a, 0x37, 0x37, 0x3a, 0x65, 0x38, 0x3a, 0x66, 0x62, 0x3a, 0x34, 0x34, 0x3a, 0x34, 0x38, 0x3a, 0x36, 0x35, 0x3a, 0x32, 0x37, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x34, 0x37, 0x3a, 0x34, 0x63, 0x3a, 0x66, 0x62, 0x3a, 0x66, 0x65, 0x3a, 0x37, 0x32, 0x3a, 0x63, 0x33, 0x3a, 0x61, 0x63, 0x3a, 0x30, 0x35, 0x3a, 0x37, 0x62, 0x3a, 0x31, 0x64, 0x3a, 0x63, 0x62, 0x3a, 0x65, 0x62, 0x3a, 0x35, 0x65, 0x3a, 0x36, 0x35, 0x3a, 0x39, 0x61, 0x3a, 0x61, 0x62, 0x3a, 0x30, 0x32, 0x3a, 0x65, 0x34, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x38, 0x3a, 0x35, 0x62, 0x3a, 0x33, 0x62, 0x3a, 0x38, 0x62, 0x3a, 0x30, 0x62, 0x3a, 0x63, 0x37, 0x3a, 0x63, 0x63, 0x3a, 0x61, 0x39, 0x3a, 0x61, 0x36, 0x3a, 0x38, 0x62, 0x3a, 0x65, 0x31, 0x3a, 0x38, 0x37, 0x3a, 0x62, 0x30, 0x3a, 0x31, 0x39, 0x3a, 0x31, 0x61, 0x3a, 0x30, 0x63, 0x3a, 0x32, 0x38, 0x3a, 0x35, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x36, 0x66, 0x3a, 0x39, 0x39, 0x3a, 0x35, 0x32, 0x3a, 0x37, 0x65, 0x3a, 0x65, 0x64, 0x3a, 0x62, 0x30, 0x3a, 0x33, 0x61, 0x3a, 0x36, 0x38, 0x3a, 0x33, 0x62, 0x3a, 0x38, 0x63, 0x3a, 0x30, 0x61, 0x3a, 0x30, 0x38, 0x3a, 0x37, 0x34, 0x3a, 0x37, 0x32, 0x3a, 0x61, 0x62, 0x3a, 0x62, 0x39, 0x3a, 0x30, 0x39, 0x3a, 0x63, 0x35, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x64, 0x3a, 0x30, 0x34, 0x3a, 0x37, 0x65, 0x3a, 0x36, 0x66, 0x3a, 0x30, 0x62, 0x3a, 0x31, 0x63, 0x3a, 0x30, 0x39, 0x3a, 0x32, 0x31, 0x3a, 0x64, 0x30, 0x3a, 0x63, 0x64, 0x3a, 0x37, 0x66, 0x3a, 0x66, 0x39, 0x3a, 0x63, 0x34, 0x3a, 0x35, 0x65, 0x3a, 0x32, 0x37, 0x3a, 0x32, 0x30, 0x3a, 0x65, 0x34, 0x3a, 0x38, 0x35, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x37, 0x33, 0x3a, 0x35, 0x32, 0x3a, 0x30, 0x35, 0x3a, 0x64, 0x32, 0x3a, 0x62, 0x61, 0x3a, 0x66, 0x38, 0x3a, 0x64, 0x35, 0x3a, 0x38, 0x66, 0x3a, 0x34, 0x31, 0x3a, 0x63, 0x63, 0x3a, 0x32, 0x33, 0x3a, 0x32, 0x65, 0x3a, 0x31, 0x32, 0x3a, 0x36, 0x64, 0x3a, 0x62, 0x63, 0x3a, 0x33, 0x31, 0x3a, 0x39, 0x38, 0x3a, 0x65, 0x37, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x36, 0x33, 0x3a, 0x61, 0x33, 0x3a, 0x38, 0x65, 0x3a, 0x32, 0x36, 0x3a, 0x63, 0x64, 0x3a, 0x65, 0x38, 0x3a, 0x32, 0x62, 0x3a, 0x38, 0x38, 0x3a, 0x65, 0x65, 0x3a, 0x65, 0x32, 0x3a, 0x66, 0x65, 0x3a, 0x33, 0x61, 0x3a, 0x37, 0x34, 0x3a, 0x35, 0x32, 0x3a, 0x33, 0x34, 0x3a, 0x30, 0x65, 0x3a, 0x66, 0x64, 0x3a, 0x31, 0x32, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x35, 0x3a, 0x35, 0x65, 0x3a, 0x36, 0x39, 0x3a, 0x35, 0x30, 0x3a, 0x32, 0x30, 0x3a, 0x33, 0x31, 0x3a, 0x33, 0x34, 0x3a, 0x65, 0x34, 0x3a, 0x33, 0x31, 0x3a, 0x66, 0x31, 0x3a, 0x65, 0x37, 0x3a, 0x65, 0x34, 0x3a, 0x35, 0x62, 0x3a, 0x30, 0x33, 0x3a, 0x31, 0x33, 0x3a, 0x64, 0x61, 0x3a, 0x61, 0x63, 0x3a, 0x34, 0x31, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x36, 0x63, 0x3a, 0x65, 0x37, 0x3a, 0x63, 0x66, 0x3a, 0x32, 0x62, 0xa, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0xa, 0x4d, 0x49, 0x49, 0x45, 0x6e, 0x6a, 0x43, 0x43, 0x41, 0x34, 0x61, 0x67, 0x41, 0x77, 0x49, 0x42, 0x41, 0x67, 0x49, 0x42, 0x41, 0x54, 0x41, 0x4e, 0x42, 0x67, 0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77, 0x30, 0x42, 0x41, 0x51, 0x73, 0x46, 0x41, 0x44, 0x43, 0x42, 0x6c, 0x44, 0x45, 0x4c, 0x4d, 0x41, 0x6b, 0x47, 0x41, 0x31, 0x55, 0x45, 0x42, 0x68, 0x4d, 0x43, 0x56, 0x56, 0x4d, 0x78, 0xa, 0x45, 0x44, 0x41, 0x4f, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x67, 0x4d, 0x42, 0x30, 0x31, 0x76, 0x62, 0x6e, 0x52, 0x68, 0x62, 0x6d, 0x45, 0x78, 0x45, 0x44, 0x41, 0x4f, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x63, 0x4d, 0x42, 0x30, 0x4a, 0x76, 0x65, 0x6d, 0x56, 0x74, 0x59, 0x57, 0x34, 0x78, 0x45, 0x54, 0x41, 0x50, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x6f, 0x4d, 0x43, 0x46, 0x4e, 0x68, 0xa, 0x64, 0x33, 0x52, 0x76, 0x62, 0x33, 0x52, 0x6f, 0x4d, 0x52, 0x4d, 0x77, 0x45, 0x51, 0x59, 0x44, 0x56, 0x51, 0x51, 0x4c, 0x44, 0x41, 0x70, 0x44, 0x62, 0x32, 0x35, 0x7a, 0x64, 0x57, 0x78, 0x30, 0x61, 0x57, 0x35, 0x6e, 0x4d, 0x52, 0x67, 0x77, 0x46, 0x67, 0x59, 0x44, 0x56, 0x51, 0x51, 0x44, 0x44, 0x41, 0x39, 0x33, 0x64, 0x33, 0x63, 0x75, 0x64, 0x32, 0x39, 0x73, 0x5a, 0x6e, 0x4e, 0x7a, 0xa, 0x62, 0x43, 0x35, 0x6a, 0x62, 0x32, 0x30, 0x78, 0x48, 0x7a, 0x41, 0x64, 0x42, 0x67, 0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77, 0x30, 0x42, 0x43, 0x51, 0x45, 0x57, 0x45, 0x47, 0x6c, 0x75, 0x5a, 0x6d, 0x39, 0x41, 0x64, 0x32, 0x39, 0x73, 0x5a, 0x6e, 0x4e, 0x7a, 0x62, 0x43, 0x35, 0x6a, 0x62, 0x32, 0x30, 0x77, 0x48, 0x68, 0x63, 0x4e, 0x4d, 0x54, 0x67, 0x77, 0x4e, 0x44, 0x45, 0x7a, 0xa, 0x4d, 0x54, 0x55, 0x79, 0x4d, 0x7a, 0x45, 0x77, 0x57, 0x68, 0x63, 0x4e, 0x4d, 0x6a, 0x45, 0x77, 0x4d, 0x54, 0x41, 0x33, 0x4d, 0x54, 0x55, 0x79, 0x4d, 0x7a, 0x45, 0x77, 0x57, 0x6a, 0x43, 0x42, 0x6b, 0x44, 0x45, 0x4c, 0x4d, 0x41, 0x6b, 0x47, 0x41, 0x31, 0x55, 0x45, 0x42, 0x68, 0x4d, 0x43, 0x56, 0x56, 0x4d, 0x78, 0x45, 0x44, 0x41, 0x4f, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x67, 0x4d, 0xa, 0x42, 0x30, 0x31, 0x76, 0x62, 0x6e, 0x52, 0x68, 0x62, 0x6d, 0x45, 0x78, 0x45, 0x44, 0x41, 0x4f, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x63, 0x4d, 0x42, 0x30, 0x4a, 0x76, 0x65, 0x6d, 0x56, 0x74, 0x59, 0x57, 0x34, 0x78, 0x45, 0x44, 0x41, 0x4f, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x6f, 0x4d, 0x42, 0x33, 0x64, 0x76, 0x62, 0x47, 0x5a, 0x54, 0x55, 0x30, 0x77, 0x78, 0x45, 0x44, 0x41, 0x4f, 0xa, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x73, 0x4d, 0x42, 0x31, 0x4e, 0x31, 0x63, 0x48, 0x42, 0x76, 0x63, 0x6e, 0x51, 0x78, 0x47, 0x44, 0x41, 0x57, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x4d, 0x4d, 0x44, 0x33, 0x64, 0x33, 0x64, 0x79, 0x35, 0x33, 0x62, 0x32, 0x78, 0x6d, 0x63, 0x33, 0x4e, 0x73, 0x4c, 0x6d, 0x4e, 0x76, 0x62, 0x54, 0x45, 0x66, 0x4d, 0x42, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0xa, 0x53, 0x49, 0x62, 0x33, 0x44, 0x51, 0x45, 0x4a, 0x41, 0x52, 0x59, 0x51, 0x61, 0x57, 0x35, 0x6d, 0x62, 0x30, 0x42, 0x33, 0x62, 0x32, 0x78, 0x6d, 0x63, 0x33, 0x4e, 0x73, 0x4c, 0x6d, 0x4e, 0x76, 0x62, 0x54, 0x43, 0x43, 0x41, 0x53, 0x49, 0x77, 0x44, 0x51, 0x59, 0x4a, 0x4b, 0x6f, 0x5a, 0x49, 0x68, 0x76, 0x63, 0x4e, 0x41, 0x51, 0x45, 0x42, 0x42, 0x51, 0x41, 0x44, 0x67, 0x67, 0x45, 0x50, 0xa, 0x41, 0x44, 0x43, 0x43, 0x41, 0x51, 0x6f, 0x43, 0x67, 0x67, 0x45, 0x42, 0x41, 0x4d, 0x43, 0x56, 0x43, 0x4f, 0x46, 0x58, 0x51, 0x66, 0x4a, 0x78, 0x62, 0x62, 0x66, 0x53, 0x52, 0x55, 0x45, 0x6e, 0x41, 0x57, 0x58, 0x47, 0x52, 0x61, 0x37, 0x79, 0x76, 0x43, 0x51, 0x77, 0x75, 0x4a, 0x58, 0x4f, 0x4c, 0x30, 0x37, 0x57, 0x39, 0x68, 0x79, 0x49, 0x76, 0x48, 0x79, 0x66, 0x2b, 0x36, 0x68, 0x6e, 0xa, 0x66, 0x2f, 0x35, 0x63, 0x6e, 0x46, 0x46, 0x31, 0x39, 0x34, 0x72, 0x4b, 0x42, 0x2b, 0x63, 0x31, 0x4c, 0x34, 0x2f, 0x68, 0x76, 0x58, 0x76, 0x41, 0x4c, 0x33, 0x79, 0x72, 0x5a, 0x4b, 0x67, 0x58, 0x2f, 0x4d, 0x70, 0x64, 0x65, 0x37, 0x72, 0x67, 0x49, 0x65, 0x56, 0x79, 0x4c, 0x6d, 0x38, 0x75, 0x68, 0x74, 0x69, 0x56, 0x63, 0x39, 0x71, 0x73, 0x47, 0x31, 0x4f, 0x35, 0x58, 0x7a, 0x2f, 0x58, 0xa, 0x47, 0x51, 0x30, 0x6c, 0x54, 0x2b, 0x46, 0x6a, 0x59, 0x31, 0x47, 0x4c, 0x43, 0x32, 0x51, 0x2f, 0x72, 0x55, 0x4f, 0x34, 0x70, 0x52, 0x78, 0x63, 0x4e, 0x4c, 0x4f, 0x75, 0x41, 0x4b, 0x42, 0x6a, 0x78, 0x66, 0x5a, 0x2f, 0x43, 0x31, 0x6c, 0x6f, 0x65, 0x48, 0x4f, 0x6d, 0x6a, 0x42, 0x69, 0x70, 0x41, 0x6d, 0x32, 0x76, 0x77, 0x78, 0x6b, 0x42, 0x4c, 0x72, 0x67, 0x51, 0x34, 0x38, 0x62, 0x4d, 0xa, 0x51, 0x4c, 0x52, 0x70, 0x6f, 0x30, 0x59, 0x7a, 0x61, 0x59, 0x64, 0x75, 0x78, 0x4c, 0x73, 0x58, 0x70, 0x76, 0x50, 0x6f, 0x33, 0x61, 0x31, 0x7a, 0x76, 0x48, 0x73, 0x76, 0x49, 0x62, 0x58, 0x39, 0x5a, 0x6c, 0x45, 0x4d, 0x76, 0x56, 0x53, 0x7a, 0x34, 0x57, 0x31, 0x66, 0x48, 0x4c, 0x77, 0x6a, 0x63, 0x39, 0x45, 0x4a, 0x41, 0x34, 0x6b, 0x55, 0x30, 0x68, 0x43, 0x35, 0x5a, 0x4d, 0x4d, 0x71, 0xa, 0x30, 0x4b, 0x47, 0x57, 0x53, 0x72, 0x7a, 0x68, 0x31, 0x42, 0x70, 0x62, 0x78, 0x36, 0x44, 0x41, 0x77, 0x57, 0x4e, 0x34, 0x44, 0x30, 0x51, 0x33, 0x4d, 0x44, 0x4b, 0x57, 0x67, 0x44, 0x49, 0x6a, 0x6c, 0x61, 0x46, 0x33, 0x75, 0x68, 0x50, 0x53, 0x6c, 0x33, 0x50, 0x69, 0x58, 0x53, 0x58, 0x4a, 0x61, 0x67, 0x33, 0x44, 0x4f, 0x57, 0x43, 0x6b, 0x74, 0x4c, 0x42, 0x70, 0x51, 0x6b, 0x49, 0x4a, 0xa, 0x36, 0x64, 0x67, 0x49, 0x76, 0x44, 0x4d, 0x67, 0x73, 0x31, 0x67, 0x69, 0x70, 0x36, 0x72, 0x72, 0x78, 0x4f, 0x48, 0x6d, 0x59, 0x59, 0x50, 0x46, 0x30, 0x70, 0x62, 0x66, 0x32, 0x64, 0x42, 0x50, 0x72, 0x64, 0x63, 0x43, 0x41, 0x77, 0x45, 0x41, 0x41, 0x61, 0x4f, 0x42, 0x2f, 0x44, 0x43, 0x42, 0x2b, 0x54, 0x41, 0x64, 0x42, 0x67, 0x4e, 0x56, 0x48, 0x51, 0x34, 0x45, 0x46, 0x67, 0x51, 0x55, 0xa, 0x73, 0x78, 0x45, 0x79, 0x79, 0x5a, 0x4b, 0x59, 0x68, 0x4f, 0x4c, 0x4a, 0x2b, 0x4e, 0x41, 0x37, 0x62, 0x67, 0x4e, 0x43, 0x79, 0x68, 0x38, 0x4f, 0x6a, 0x6a, 0x77, 0x77, 0x67, 0x63, 0x6b, 0x47, 0x41, 0x31, 0x55, 0x64, 0x49, 0x77, 0x53, 0x42, 0x77, 0x54, 0x43, 0x42, 0x76, 0x6f, 0x41, 0x55, 0x4a, 0x34, 0x35, 0x6e, 0x45, 0x58, 0x54, 0x44, 0x4a, 0x68, 0x30, 0x2f, 0x37, 0x54, 0x4e, 0x6a, 0xa, 0x73, 0x36, 0x54, 0x59, 0x48, 0x54, 0x44, 0x6c, 0x36, 0x4e, 0x57, 0x68, 0x67, 0x5a, 0x71, 0x6b, 0x67, 0x5a, 0x63, 0x77, 0x67, 0x5a, 0x51, 0x78, 0x43, 0x7a, 0x41, 0x4a, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x59, 0x54, 0x41, 0x6c, 0x56, 0x54, 0x4d, 0x52, 0x41, 0x77, 0x44, 0x67, 0x59, 0x44, 0x56, 0x51, 0x51, 0x49, 0x44, 0x41, 0x64, 0x4e, 0x62, 0x32, 0x35, 0x30, 0x59, 0x57, 0x35, 0x68, 0xa, 0x4d, 0x52, 0x41, 0x77, 0x44, 0x67, 0x59, 0x44, 0x56, 0x51, 0x51, 0x48, 0x44, 0x41, 0x64, 0x43, 0x62, 0x33, 0x70, 0x6c, 0x62, 0x57, 0x46, 0x75, 0x4d, 0x52, 0x45, 0x77, 0x44, 0x77, 0x59, 0x44, 0x56, 0x51, 0x51, 0x4b, 0x44, 0x41, 0x68, 0x54, 0x59, 0x58, 0x64, 0x30, 0x62, 0x32, 0x39, 0x30, 0x61, 0x44, 0x45, 0x54, 0x4d, 0x42, 0x45, 0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x77, 0x77, 0x4b, 0xa, 0x51, 0x32, 0x39, 0x75, 0x63, 0x33, 0x56, 0x73, 0x64, 0x47, 0x6c, 0x75, 0x5a, 0x7a, 0x45, 0x59, 0x4d, 0x42, 0x59, 0x47, 0x41, 0x31, 0x55, 0x45, 0x41, 0x77, 0x77, 0x50, 0x64, 0x33, 0x64, 0x33, 0x4c, 0x6e, 0x64, 0x76, 0x62, 0x47, 0x5a, 0x7a, 0x63, 0x32, 0x77, 0x75, 0x59, 0x32, 0x39, 0x74, 0x4d, 0x52, 0x38, 0x77, 0x48, 0x51, 0x59, 0x4a, 0x4b, 0x6f, 0x5a, 0x49, 0x68, 0x76, 0x63, 0x4e, 0xa, 0x41, 0x51, 0x6b, 0x42, 0x46, 0x68, 0x42, 0x70, 0x62, 0x6d, 0x5a, 0x76, 0x51, 0x48, 0x64, 0x76, 0x62, 0x47, 0x5a, 0x7a, 0x63, 0x32, 0x77, 0x75, 0x59, 0x32, 0x39, 0x74, 0x67, 0x67, 0x6b, 0x41, 0x68, 0x76, 0x2f, 0x31, 0x6a, 0x68, 0x44, 0x65, 0x75, 0x50, 0x73, 0x77, 0x44, 0x41, 0x59, 0x44, 0x56, 0x52, 0x30, 0x54, 0x42, 0x41, 0x55, 0x77, 0x41, 0x77, 0x45, 0x42, 0x2f, 0x7a, 0x41, 0x4e, 0xa, 0x42, 0x67, 0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77, 0x30, 0x42, 0x41, 0x51, 0x73, 0x46, 0x41, 0x41, 0x4f, 0x43, 0x41, 0x51, 0x45, 0x41, 0x74, 0x46, 0x52, 0x67, 0x72, 0x61, 0x41, 0x44, 0x4d, 0x74, 0x34, 0x43, 0x66, 0x79, 0x46, 0x4b, 0x67, 0x63, 0x62, 0x74, 0x7a, 0x63, 0x33, 0x59, 0x45, 0x6f, 0x72, 0x41, 0x75, 0x6f, 0x4a, 0x62, 0x64, 0x61, 0x31, 0x55, 0x34, 0x33, 0x79, 0x41, 0xa, 0x61, 0x71, 0x77, 0x75, 0x62, 0x43, 0x42, 0x4f, 0x76, 0x6b, 0x32, 0x43, 0x70, 0x30, 0x63, 0x54, 0x58, 0x50, 0x54, 0x47, 0x61, 0x69, 0x73, 0x51, 0x6d, 0x56, 0x6a, 0x65, 0x71, 0x32, 0x74, 0x38, 0x49, 0x67, 0x58, 0x42, 0x67, 0x35, 0x33, 0x4c, 0x2f, 0x7a, 0x7a, 0x6b, 0x4c, 0x56, 0x64, 0x71, 0x70, 0x70, 0x62, 0x66, 0x30, 0x38, 0x46, 0x6f, 0x34, 0x39, 0x4c, 0x47, 0x67, 0x30, 0x75, 0x58, 0xa, 0x34, 0x73, 0x59, 0x79, 0x44, 0x72, 0x37, 0x45, 0x41, 0x37, 0x6b, 0x48, 0x69, 0x6c, 0x75, 0x34, 0x68, 0x4c, 0x72, 0x46, 0x4f, 0x54, 0x38, 0x63, 0x57, 0x4b, 0x64, 0x56, 0x31, 0x2f, 0x43, 0x62, 0x36, 0x4e, 0x4a, 0x46, 0x75, 0x65, 0x4f, 0x44, 0x4c, 0x75, 0x36, 0x32, 0x63, 0x56, 0x61, 0x35, 0x4f, 0x75, 0x34, 0x2f, 0x4a, 0x39, 0x68, 0x33, 0x36, 0x50, 0x74, 0x45, 0x53, 0x47, 0x55, 0x6e, 0xa, 0x52, 0x30, 0x7a, 0x37, 0x2f, 0x6e, 0x4c, 0x44, 0x72, 0x41, 0x56, 0x37, 0x48, 0x63, 0x76, 0x72, 0x58, 0x6d, 0x57, 0x61, 0x71, 0x77, 0x4c, 0x6b, 0x69, 0x46, 0x73, 0x37, 0x69, 0x77, 0x76, 0x48, 0x7a, 0x4b, 0x6d, 0x6d, 0x69, 0x2b, 0x47, 0x48, 0x73, 0x42, 0x6b, 0x61, 0x44, 0x43, 0x68, 0x59, 0x62, 0x35, 0x6c, 0x53, 0x66, 0x75, 0x32, 0x77, 0x4f, 0x6d, 0x67, 0x37, 0x6a, 0x41, 0x6f, 0x49, 0xa, 0x64, 0x48, 0x4b, 0x72, 0x75, 0x51, 0x6e, 0x46, 0x37, 0x51, 0x52, 0x2b, 0x62, 0x77, 0x73, 0x63, 0x43, 0x53, 0x48, 0x51, 0x7a, 0x58, 0x2f, 0x35, 0x78, 0x46, 0x34, 0x6e, 0x49, 0x4f, 0x53, 0x46, 0x63, 0x31, 0x49, 0x46, 0x30, 0x72, 0x72, 0x34, 0x31, 0x59, 0x39, 0x42, 0x7a, 0x43, 0x4d, 0x75, 0x45, 0x6d, 0x32, 0x38, 0x4d, 0x5a, 0x6a, 0x6e, 0x59, 0x36, 0x4f, 0x4f, 0x4a, 0x73, 0x33, 0x6f, 0xa, 0x4b, 0x34, 0x6a, 0x75, 0x34, 0x76, 0x34, 0x36, 0x64, 0x46, 0x49, 0x30, 0x44, 0x76, 0x30, 0x53, 0x35, 0x56, 0x35, 0x70, 0x55, 0x43, 0x41, 0x78, 0x4e, 0x4f, 0x51, 0x78, 0x38, 0x65, 0x66, 0x6b, 0x57, 0x77, 0x4d, 0x54, 0x32, 0x71, 0x78, 0x42, 0x62, 0x4f, 0x66, 0x50, 0x4b, 0x77, 0x3d, 0x3d, 0xa, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0xa, 0x43, 0x65, 0x72, 0x74, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x65, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x44, 0x61, 0x74, 0x61, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x3a, 0x20, 0x33, 0x20, 0x28, 0x30, 0x78, 0x32, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x53, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x20, 0x4e, 0x75, 0x6d, 0x62, 0x65, 0x72, 0x3a, 0x20, 0x39, 0x37, 0x32, 0x37, 0x37, 0x36, 0x33, 0x37, 0x31, 0x30, 0x36, 0x36, 0x30, 0x37, 0x35, 0x33, 0x36, 0x35, 0x39, 0x20, 0x28, 0x30, 0x78, 0x38, 0x36, 0x66, 0x66, 0x66, 0x35, 0x38, 0x65, 0x31, 0x30, 0x64, 0x65, 0x62, 0x38, 0x66, 0x62, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x53, 0x69, 0x67, 0x6e, 0x61, 0x74, 0x75, 0x72, 0x65, 0x20, 0x41, 0x6c, 0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x3a, 0x20, 0x73, 0x68, 0x61, 0x32, 0x35, 0x36, 0x57, 0x69, 0x74, 0x68, 0x52, 0x53, 0x41, 0x45, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x49, 0x73, 0x73, 0x75, 0x65, 0x72, 0x3a, 0x20, 0x43, 0x3d, 0x55, 0x53, 0x2c, 0x20, 0x53, 0x54, 0x3d, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x2c, 0x20, 0x4c, 0x3d, 0x42, 0x6f, 0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x2c, 0x20, 0x4f, 0x3d, 0x53, 0x61, 0x77, 0x74, 0x6f, 0x6f, 0x74, 0x68, 0x2c, 0x20, 0x4f, 0x55, 0x3d, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x43, 0x4e, 0x3d, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x65, 0x6d, 0x61, 0x69, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3d, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x56, 0x61, 0x6c, 0x69, 0x64, 0x69, 0x74, 0x79, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x4e, 0x6f, 0x74, 0x20, 0x42, 0x65, 0x66, 0x6f, 0x72, 0x65, 0x3a, 0x20, 0x41, 0x70, 0x72, 0x20, 0x31, 0x33, 0x20, 0x31, 0x35, 0x3a, 0x32, 0x33, 0x3a, 0x30, 0x39, 0x20, 0x32, 0x30, 0x31, 0x38, 0x20, 0x47, 0x4d, 0x54, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x4e, 0x6f, 0x74, 0x20, 0x41, 0x66, 0x74, 0x65, 0x72, 0x20, 0x3a, 0x20, 0x4a, 0x61, 0x6e, 0x20, 0x20, 0x37, 0x20, 0x31, 0x35, 0x3a, 0x32, 0x33, 0x3a, 0x30, 0x39, 0x20, 0x32, 0x30, 0x32, 0x31, 0x20, 0x47, 0x4d, 0x54, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x3a, 0x20, 0x43, 0x3d, 0x55, 0x53, 0x2c, 0x20, 0x53, 0x54, 0x3d, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x2c, 0x20, 0x4c, 0x3d, 0x42, 0x6f, 0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x2c, 0x20, 0x4f, 0x3d, 0x53, 0x61, 0x77, 0x74, 0x6f, 0x6f, 0x74, 0x68, 0x2c, 0x20, 0x4f, 0x55, 0x3d, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x2c, 0x20, 0x43, 0x4e, 0x3d, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x65, 0x6d, 0x61, 0x69, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3d, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x49, 0x6e, 0x66, 0x6f, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x41, 0x6c, 0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x3a, 0x20, 0x72, 0x73, 0x61, 0x45, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x50, 0x75, 0x62, 0x6c, 0x69, 0x63, 0x2d, 0x4b, 0x65, 0x79, 0x3a, 0x20, 0x28, 0x32, 0x30, 0x34, 0x38, 0x20, 0x62, 0x69, 0x74, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x4d, 0x6f, 0x64, 0x75, 0x6c, 0x75, 0x73, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x30, 0x30, 0x3a, 0x62, 0x66, 0x3a, 0x30, 0x63, 0x3a, 0x63, 0x61, 0x3a, 0x32, 0x64, 0x3a, 0x31, 0x34, 0x3a, 0x62, 0x32, 0x3a, 0x31, 0x65, 0x3a, 0x38, 0x34, 0x3a, 0x34, 0x32, 0x3a, 0x35, 0x62, 0x3a, 0x63, 0x64, 0x3a, 0x33, 0x38, 0x3a, 0x31, 0x66, 0x3a, 0x34, 0x61, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x32, 0x3a, 0x34, 0x64, 0x3a, 0x37, 0x35, 0x3a, 0x31, 0x30, 0x3a, 0x66, 0x31, 0x3a, 0x62, 0x36, 0x3a, 0x33, 0x35, 0x3a, 0x39, 0x66, 0x3a, 0x64, 0x66, 0x3a, 0x63, 0x61, 0x3a, 0x37, 0x64, 0x3a, 0x30, 0x33, 0x3a, 0x39, 0x38, 0x3a, 0x64, 0x33, 0x3a, 0x61, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x65, 0x3a, 0x30, 0x33, 0x3a, 0x36, 0x36, 0x3a, 0x65, 0x65, 0x3a, 0x32, 0x61, 0x3a, 0x66, 0x31, 0x3a, 0x64, 0x38, 0x3a, 0x62, 0x30, 0x3a, 0x37, 0x64, 0x3a, 0x36, 0x65, 0x3a, 0x30, 0x37, 0x3a, 0x35, 0x34, 0x3a, 0x30, 0x62, 0x3a, 0x31, 0x30, 0x3a, 0x39, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x32, 0x31, 0x3a, 0x34, 0x64, 0x3a, 0x38, 0x30, 0x3a, 0x63, 0x62, 0x3a, 0x31, 0x32, 0x3a, 0x32, 0x30, 0x3a, 0x65, 0x37, 0x3a, 0x63, 0x63, 0x3a, 0x34, 0x66, 0x3a, 0x64, 0x65, 0x3a, 0x34, 0x35, 0x3a, 0x37, 0x64, 0x3a, 0x63, 0x39, 0x3a, 0x37, 0x32, 0x3a, 0x37, 0x37, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x32, 0x3a, 0x65, 0x61, 0x3a, 0x63, 0x61, 0x3a, 0x39, 0x30, 0x3a, 0x62, 0x62, 0x3a, 0x36, 0x39, 0x3a, 0x35, 0x32, 0x3a, 0x31, 0x30, 0x3a, 0x30, 0x33, 0x3a, 0x32, 0x66, 0x3a, 0x61, 0x38, 0x3a, 0x66, 0x33, 0x3a, 0x39, 0x35, 0x3a, 0x63, 0x35, 0x3a, 0x66, 0x31, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x62, 0x3a, 0x36, 0x32, 0x3a, 0x35, 0x36, 0x3a, 0x31, 0x62, 0x3a, 0x65, 0x66, 0x3a, 0x36, 0x37, 0x3a, 0x36, 0x66, 0x3a, 0x61, 0x34, 0x3a, 0x31, 0x30, 0x3a, 0x34, 0x31, 0x3a, 0x39, 0x35, 0x3a, 0x61, 0x64, 0x3a, 0x30, 0x61, 0x3a, 0x39, 0x62, 0x3a, 0x65, 0x33, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x35, 0x3a, 0x63, 0x30, 0x3a, 0x62, 0x30, 0x3a, 0x64, 0x32, 0x3a, 0x37, 0x30, 0x3a, 0x37, 0x36, 0x3a, 0x35, 0x30, 0x3a, 0x33, 0x30, 0x3a, 0x35, 0x62, 0x3a, 0x61, 0x38, 0x3a, 0x65, 0x38, 0x3a, 0x30, 0x38, 0x3a, 0x32, 0x63, 0x3a, 0x37, 0x63, 0x3a, 0x65, 0x64, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x37, 0x3a, 0x61, 0x32, 0x3a, 0x37, 0x61, 0x3a, 0x38, 0x64, 0x3a, 0x33, 0x38, 0x3a, 0x32, 0x39, 0x3a, 0x31, 0x63, 0x3a, 0x61, 0x63, 0x3a, 0x63, 0x37, 0x3a, 0x65, 0x64, 0x3a, 0x66, 0x32, 0x3a, 0x37, 0x63, 0x3a, 0x39, 0x35, 0x3a, 0x62, 0x30, 0x3a, 0x39, 0x35, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x32, 0x3a, 0x37, 0x64, 0x3a, 0x34, 0x39, 0x3a, 0x35, 0x63, 0x3a, 0x33, 0x38, 0x3a, 0x63, 0x64, 0x3a, 0x37, 0x37, 0x3a, 0x32, 0x35, 0x3a, 0x65, 0x66, 0x3a, 0x62, 0x64, 0x3a, 0x38, 0x30, 0x3a, 0x37, 0x35, 0x3a, 0x35, 0x33, 0x3a, 0x39, 0x34, 0x3a, 0x33, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x64, 0x3a, 0x63, 0x61, 0x3a, 0x36, 0x33, 0x3a, 0x35, 0x62, 0x3a, 0x39, 0x66, 0x3a, 0x31, 0x35, 0x3a, 0x62, 0x35, 0x3a, 0x64, 0x33, 0x3a, 0x31, 0x64, 0x3a, 0x31, 0x33, 0x3a, 0x32, 0x66, 0x3a, 0x31, 0x39, 0x3a, 0x64, 0x31, 0x3a, 0x33, 0x63, 0x3a, 0x64, 0x62, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x37, 0x36, 0x3a, 0x33, 0x61, 0x3a, 0x63, 0x63, 0x3a, 0x62, 0x38, 0x3a, 0x37, 0x64, 0x3a, 0x63, 0x39, 0x3a, 0x65, 0x35, 0x3a, 0x63, 0x32, 0x3a, 0x64, 0x37, 0x3a, 0x64, 0x61, 0x3a, 0x34, 0x30, 0x3a, 0x36, 0x66, 0x3a, 0x64, 0x38, 0x3a, 0x32, 0x31, 0x3a, 0x64, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x37, 0x33, 0x3a, 0x31, 0x62, 0x3a, 0x34, 0x32, 0x3a, 0x32, 0x64, 0x3a, 0x35, 0x33, 0x3a, 0x39, 0x63, 0x3a, 0x66, 0x65, 0x3a, 0x31, 0x61, 0x3a, 0x66, 0x63, 0x3a, 0x37, 0x64, 0x3a, 0x61, 0x62, 0x3a, 0x37, 0x61, 0x3a, 0x33, 0x36, 0x3a, 0x33, 0x66, 0x3a, 0x39, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x65, 0x3a, 0x38, 0x34, 0x3a, 0x37, 0x63, 0x3a, 0x30, 0x35, 0x3a, 0x36, 0x37, 0x3a, 0x63, 0x65, 0x3a, 0x36, 0x61, 0x3a, 0x31, 0x34, 0x3a, 0x33, 0x38, 0x3a, 0x38, 0x37, 0x3a, 0x61, 0x39, 0x3a, 0x66, 0x31, 0x3a, 0x38, 0x63, 0x3a, 0x62, 0x35, 0x3a, 0x36, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x62, 0x3a, 0x36, 0x38, 0x3a, 0x37, 0x66, 0x3a, 0x37, 0x31, 0x3a, 0x32, 0x30, 0x3a, 0x32, 0x62, 0x3a, 0x66, 0x35, 0x3a, 0x61, 0x30, 0x3a, 0x36, 0x33, 0x3a, 0x66, 0x35, 0x3a, 0x35, 0x36, 0x3a, 0x32, 0x66, 0x3a, 0x61, 0x33, 0x3a, 0x32, 0x36, 0x3a, 0x64, 0x32, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x37, 0x3a, 0x36, 0x66, 0x3a, 0x62, 0x31, 0x3a, 0x35, 0x61, 0x3a, 0x31, 0x37, 0x3a, 0x64, 0x37, 0x3a, 0x33, 0x38, 0x3a, 0x39, 0x39, 0x3a, 0x30, 0x38, 0x3a, 0x66, 0x65, 0x3a, 0x39, 0x33, 0x3a, 0x35, 0x38, 0x3a, 0x36, 0x66, 0x3a, 0x66, 0x65, 0x3a, 0x63, 0x33, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x3a, 0x34, 0x39, 0x3a, 0x30, 0x38, 0x3a, 0x31, 0x36, 0x3a, 0x30, 0x62, 0x3a, 0x61, 0x37, 0x3a, 0x34, 0x64, 0x3a, 0x36, 0x37, 0x3a, 0x30, 0x30, 0x3a, 0x35, 0x32, 0x3a, 0x33, 0x31, 0x3a, 0x36, 0x37, 0x3a, 0x32, 0x33, 0x3a, 0x34, 0x65, 0x3a, 0x39, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x64, 0x3a, 0x35, 0x31, 0x3a, 0x34, 0x35, 0x3a, 0x31, 0x64, 0x3a, 0x62, 0x39, 0x3a, 0x30, 0x34, 0x3a, 0x64, 0x39, 0x3a, 0x30, 0x62, 0x3a, 0x65, 0x63, 0x3a, 0x64, 0x38, 0x3a, 0x32, 0x38, 0x3a, 0x62, 0x33, 0x3a, 0x34, 0x62, 0x3a, 0x62, 0x64, 0x3a, 0x65, 0x64, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x36, 0x3a, 0x37, 0x39, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x45, 0x78, 0x70, 0x6f, 0x6e, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x36, 0x35, 0x35, 0x33, 0x37, 0x20, 0x28, 0x30, 0x78, 0x31, 0x30, 0x30, 0x30, 0x31, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x58, 0x35, 0x30, 0x39, 0x76, 0x33, 0x20, 0x65, 0x78, 0x74, 0x65, 0x6e, 0x73, 0x69, 0x6f, 0x6e, 0x73, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x58, 0x35, 0x30, 0x39, 0x76, 0x33, 0x20, 0x53, 0x75, 0x62, 0x6a, 0x65, 0x63, 0x74, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x49, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x66, 0x69, 0x65, 0x72, 0x3a, 0x20, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x32, 0x37, 0x3a, 0x38, 0x45, 0x3a, 0x36, 0x37, 0x3a, 0x31, 0x31, 0x3a, 0x37, 0x34, 0x3a, 0x43, 0x33, 0x3a, 0x32, 0x36, 0x3a, 0x31, 0x44, 0x3a, 0x33, 0x46, 0x3a, 0x45, 0x44, 0x3a, 0x33, 0x33, 0x3a, 0x36, 0x33, 0x3a, 0x42, 0x33, 0x3a, 0x41, 0x34, 0x3a, 0x44, 0x38, 0x3a, 0x31, 0x44, 0x3a, 0x33, 0x30, 0x3a, 0x45, 0x35, 0x3a, 0x45, 0x38, 0x3a, 0x44, 0x35, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x58, 0x35, 0x30, 0x39, 0x76, 0x33, 0x20, 0x41, 0x75, 0x74, 0x68, 0x6f, 0x72, 0x69, 0x74, 0x79, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x49, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x66, 0x69, 0x65, 0x72, 0x3a, 0x20, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x6b, 0x65, 0x79, 0x69, 0x64, 0x3a, 0x32, 0x37, 0x3a, 0x38, 0x45, 0x3a, 0x36, 0x37, 0x3a, 0x31, 0x31, 0x3a, 0x37, 0x34, 0x3a, 0x43, 0x33, 0x3a, 0x32, 0x36, 0x3a, 0x31, 0x44, 0x3a, 0x33, 0x46, 0x3a, 0x45, 0x44, 0x3a, 0x33, 0x33, 0x3a, 0x36, 0x33, 0x3a, 0x42, 0x33, 0x3a, 0x41, 0x34, 0x3a, 0x44, 0x38, 0x3a, 0x31, 0x44, 0x3a, 0x33, 0x30, 0x3a, 0x45, 0x35, 0x3a, 0x45, 0x38, 0x3a, 0x44, 0x35, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x44, 0x69, 0x72, 0x4e, 0x61, 0x6d, 0x65, 0x3a, 0x2f, 0x43, 0x3d, 0x55, 0x53, 0x2f, 0x53, 0x54, 0x3d, 0x4d, 0x6f, 0x6e, 0x74, 0x61, 0x6e, 0x61, 0x2f, 0x4c, 0x3d, 0x42, 0x6f, 0x7a, 0x65, 0x6d, 0x61, 0x6e, 0x2f, 0x4f, 0x3d, 0x53, 0x61, 0x77, 0x74, 0x6f, 0x6f, 0x74, 0x68, 0x2f, 0x4f, 0x55, 0x3d, 0x43, 0x6f, 0x6e, 0x73, 0x75, 0x6c, 0x74, 0x69, 0x6e, 0x67, 0x2f, 0x43, 0x4e, 0x3d, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0x2f, 0x65, 0x6d, 0x61, 0x69, 0x6c, 0x41, 0x64, 0x64, 0x72, 0x65, 0x73, 0x73, 0x3d, 0x69, 0x6e, 0x66, 0x6f, 0x40, 0x77, 0x6f, 0x6c, 0x66, 0x73, 0x73, 0x6c, 0x2e, 0x63, 0x6f, 0x6d, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x3a, 0x38, 0x36, 0x3a, 0x46, 0x46, 0x3a, 0x46, 0x35, 0x3a, 0x38, 0x45, 0x3a, 0x31, 0x30, 0x3a, 0x44, 0x45, 0x3a, 0x42, 0x38, 0x3a, 0x46, 0x42, 0xa, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x58, 0x35, 0x30, 0x39, 0x76, 0x33, 0x20, 0x42, 0x61, 0x73, 0x69, 0x63, 0x20, 0x43, 0x6f, 0x6e, 0x73, 0x74, 0x72, 0x61, 0x69, 0x6e, 0x74, 0x73, 0x3a, 0x20, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x43, 0x41, 0x3a, 0x54, 0x52, 0x55, 0x45, 0xa, 0x20, 0x20, 0x20, 0x20, 0x53, 0x69, 0x67, 0x6e, 0x61, 0x74, 0x75, 0x72, 0x65, 0x20, 0x41, 0x6c, 0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x3a, 0x20, 0x73, 0x68, 0x61, 0x32, 0x35, 0x36, 0x57, 0x69, 0x74, 0x68, 0x52, 0x53, 0x41, 0x45, 0x6e, 0x63, 0x72, 0x79, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x39, 0x65, 0x3a, 0x32, 0x38, 0x3a, 0x38, 0x38, 0x3a, 0x37, 0x32, 0x3a, 0x30, 0x30, 0x3a, 0x63, 0x61, 0x3a, 0x65, 0x36, 0x3a, 0x65, 0x37, 0x3a, 0x39, 0x37, 0x3a, 0x63, 0x61, 0x3a, 0x63, 0x31, 0x3a, 0x66, 0x31, 0x3a, 0x31, 0x66, 0x3a, 0x39, 0x65, 0x3a, 0x31, 0x32, 0x3a, 0x62, 0x32, 0x3a, 0x62, 0x38, 0x3a, 0x63, 0x37, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x35, 0x31, 0x3a, 0x65, 0x61, 0x3a, 0x32, 0x38, 0x3a, 0x65, 0x31, 0x3a, 0x33, 0x36, 0x3a, 0x62, 0x35, 0x3a, 0x32, 0x64, 0x3a, 0x65, 0x36, 0x3a, 0x32, 0x66, 0x3a, 0x30, 0x38, 0x3a, 0x32, 0x33, 0x3a, 0x63, 0x62, 0x3a, 0x61, 0x39, 0x3a, 0x34, 0x61, 0x3a, 0x38, 0x37, 0x3a, 0x32, 0x35, 0x3a, 0x63, 0x36, 0x3a, 0x35, 0x64, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x39, 0x3a, 0x34, 0x35, 0x3a, 0x65, 0x61, 0x3a, 0x66, 0x35, 0x3a, 0x30, 0x30, 0x3a, 0x39, 0x38, 0x3a, 0x61, 0x63, 0x3a, 0x37, 0x36, 0x3a, 0x66, 0x62, 0x3a, 0x31, 0x62, 0x3a, 0x61, 0x66, 0x3a, 0x66, 0x30, 0x3a, 0x63, 0x65, 0x3a, 0x36, 0x34, 0x3a, 0x39, 0x65, 0x3a, 0x64, 0x61, 0x3a, 0x30, 0x38, 0x3a, 0x62, 0x66, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x36, 0x3a, 0x65, 0x62, 0x3a, 0x62, 0x34, 0x3a, 0x62, 0x35, 0x3a, 0x30, 0x63, 0x3a, 0x61, 0x30, 0x3a, 0x65, 0x37, 0x3a, 0x66, 0x36, 0x3a, 0x34, 0x37, 0x3a, 0x35, 0x39, 0x3a, 0x31, 0x63, 0x3a, 0x36, 0x31, 0x3a, 0x63, 0x66, 0x3a, 0x32, 0x65, 0x3a, 0x30, 0x65, 0x3a, 0x35, 0x38, 0x3a, 0x61, 0x34, 0x3a, 0x38, 0x32, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x63, 0x3a, 0x30, 0x66, 0x3a, 0x33, 0x66, 0x3a, 0x65, 0x63, 0x3a, 0x63, 0x34, 0x3a, 0x61, 0x65, 0x3a, 0x38, 0x30, 0x3a, 0x66, 0x37, 0x3a, 0x62, 0x30, 0x3a, 0x38, 0x61, 0x3a, 0x31, 0x65, 0x3a, 0x38, 0x35, 0x3a, 0x34, 0x31, 0x3a, 0x65, 0x38, 0x3a, 0x66, 0x66, 0x3a, 0x66, 0x65, 0x3a, 0x66, 0x65, 0x3a, 0x34, 0x66, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x31, 0x61, 0x3a, 0x32, 0x34, 0x3a, 0x64, 0x35, 0x3a, 0x34, 0x39, 0x3a, 0x66, 0x61, 0x3a, 0x66, 0x62, 0x3a, 0x66, 0x65, 0x3a, 0x35, 0x65, 0x3a, 0x65, 0x35, 0x3a, 0x64, 0x33, 0x3a, 0x39, 0x31, 0x3a, 0x30, 0x65, 0x3a, 0x34, 0x66, 0x3a, 0x34, 0x65, 0x3a, 0x30, 0x63, 0x3a, 0x32, 0x31, 0x3a, 0x35, 0x31, 0x3a, 0x37, 0x31, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x33, 0x3a, 0x30, 0x34, 0x3a, 0x36, 0x62, 0x3a, 0x36, 0x32, 0x3a, 0x37, 0x62, 0x3a, 0x34, 0x66, 0x3a, 0x35, 0x39, 0x3a, 0x37, 0x36, 0x3a, 0x34, 0x38, 0x3a, 0x38, 0x31, 0x3a, 0x31, 0x65, 0x3a, 0x62, 0x34, 0x3a, 0x66, 0x37, 0x3a, 0x30, 0x34, 0x3a, 0x34, 0x37, 0x3a, 0x38, 0x61, 0x3a, 0x39, 0x31, 0x3a, 0x35, 0x37, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x33, 0x3a, 0x31, 0x31, 0x3a, 0x61, 0x39, 0x3a, 0x66, 0x32, 0x3a, 0x32, 0x30, 0x3a, 0x62, 0x34, 0x3a, 0x37, 0x38, 0x3a, 0x33, 0x33, 0x3a, 0x36, 0x32, 0x3a, 0x33, 0x64, 0x3a, 0x62, 0x30, 0x3a, 0x35, 0x65, 0x3a, 0x30, 0x64, 0x3a, 0x66, 0x39, 0x3a, 0x38, 0x36, 0x3a, 0x33, 0x38, 0x3a, 0x38, 0x32, 0x3a, 0x64, 0x61, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x31, 0x3a, 0x39, 0x38, 0x3a, 0x38, 0x64, 0x3a, 0x31, 0x39, 0x3a, 0x30, 0x36, 0x3a, 0x38, 0x37, 0x3a, 0x32, 0x31, 0x3a, 0x33, 0x39, 0x3a, 0x62, 0x37, 0x3a, 0x30, 0x32, 0x3a, 0x66, 0x37, 0x3a, 0x64, 0x61, 0x3a, 0x37, 0x64, 0x3a, 0x35, 0x38, 0x3a, 0x62, 0x61, 0x3a, 0x35, 0x32, 0x3a, 0x31, 0x35, 0x3a, 0x64, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x62, 0x3a, 0x63, 0x39, 0x3a, 0x37, 0x62, 0x3a, 0x35, 0x38, 0x3a, 0x33, 0x34, 0x3a, 0x61, 0x30, 0x3a, 0x63, 0x37, 0x3a, 0x65, 0x32, 0x3a, 0x37, 0x63, 0x3a, 0x61, 0x39, 0x3a, 0x38, 0x33, 0x3a, 0x31, 0x33, 0x3a, 0x65, 0x31, 0x3a, 0x62, 0x36, 0x3a, 0x65, 0x63, 0x3a, 0x30, 0x31, 0x3a, 0x62, 0x66, 0x3a, 0x35, 0x32, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x33, 0x3a, 0x30, 0x62, 0x3a, 0x63, 0x34, 0x3a, 0x66, 0x65, 0x3a, 0x34, 0x33, 0x3a, 0x64, 0x33, 0x3a, 0x63, 0x36, 0x3a, 0x61, 0x34, 0x3a, 0x38, 0x65, 0x3a, 0x32, 0x66, 0x3a, 0x38, 0x37, 0x3a, 0x37, 0x66, 0x3a, 0x37, 0x61, 0x3a, 0x34, 0x34, 0x3a, 0x65, 0x61, 0x3a, 0x63, 0x61, 0x3a, 0x35, 0x33, 0x3a, 0x36, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x38, 0x35, 0x3a, 0x65, 0x64, 0x3a, 0x36, 0x35, 0x3a, 0x37, 0x36, 0x3a, 0x37, 0x33, 0x3a, 0x33, 0x31, 0x3a, 0x30, 0x33, 0x3a, 0x34, 0x65, 0x3a, 0x65, 0x61, 0x3a, 0x62, 0x64, 0x3a, 0x33, 0x35, 0x3a, 0x35, 0x34, 0x3a, 0x31, 0x33, 0x3a, 0x66, 0x33, 0x3a, 0x36, 0x34, 0x3a, 0x38, 0x37, 0x3a, 0x36, 0x62, 0x3a, 0x64, 0x66, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x34, 0x3a, 0x64, 0x64, 0x3a, 0x33, 0x34, 0x3a, 0x61, 0x31, 0x3a, 0x38, 0x38, 0x3a, 0x33, 0x62, 0x3a, 0x64, 0x62, 0x3a, 0x34, 0x64, 0x3a, 0x61, 0x66, 0x3a, 0x31, 0x62, 0x3a, 0x36, 0x34, 0x3a, 0x39, 0x30, 0x3a, 0x39, 0x32, 0x3a, 0x37, 0x31, 0x3a, 0x33, 0x30, 0x3a, 0x38, 0x65, 0x3a, 0x63, 0x38, 0x3a, 0x63, 0x63, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x65, 0x35, 0x3a, 0x36, 0x30, 0x3a, 0x32, 0x34, 0x3a, 0x61, 0x66, 0x3a, 0x33, 0x31, 0x3a, 0x31, 0x36, 0x3a, 0x33, 0x39, 0x3a, 0x33, 0x33, 0x3a, 0x39, 0x31, 0x3a, 0x35, 0x30, 0x3a, 0x66, 0x39, 0x3a, 0x61, 0x62, 0x3a, 0x36, 0x38, 0x3a, 0x34, 0x32, 0x3a, 0x37, 0x34, 0x3a, 0x37, 0x61, 0x3a, 0x33, 0x35, 0x3a, 0x64, 0x39, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x64, 0x3a, 0x63, 0x38, 0x3a, 0x63, 0x34, 0x3a, 0x35, 0x32, 0xa, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0xa, 0x4d, 0x49, 0x49, 0x45, 0x71, 0x6a, 0x43, 0x43, 0x41, 0x35, 0x4b, 0x67, 0x41, 0x77, 0x49, 0x42, 0x41, 0x67, 0x49, 0x4a, 0x41, 0x49, 0x62, 0x2f, 0x39, 0x59, 0x34, 0x51, 0x33, 0x72, 0x6a, 0x37, 0x4d, 0x41, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62, 0x33, 0x44, 0x51, 0x45, 0x42, 0x43, 0x77, 0x55, 0x41, 0x4d, 0x49, 0x47, 0x55, 0x4d, 0x51, 0x73, 0x77, 0x43, 0x51, 0x59, 0x44, 0xa, 0x56, 0x51, 0x51, 0x47, 0x45, 0x77, 0x4a, 0x56, 0x55, 0x7a, 0x45, 0x51, 0x4d, 0x41, 0x34, 0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x41, 0x77, 0x48, 0x54, 0x57, 0x39, 0x75, 0x64, 0x47, 0x46, 0x75, 0x59, 0x54, 0x45, 0x51, 0x4d, 0x41, 0x34, 0x47, 0x41, 0x31, 0x55, 0x45, 0x42, 0x77, 0x77, 0x48, 0x51, 0x6d, 0x39, 0x36, 0x5a, 0x57, 0x31, 0x68, 0x62, 0x6a, 0x45, 0x52, 0x4d, 0x41, 0x38, 0x47, 0xa, 0x41, 0x31, 0x55, 0x45, 0x43, 0x67, 0x77, 0x49, 0x55, 0x32, 0x46, 0x33, 0x64, 0x47, 0x39, 0x76, 0x64, 0x47, 0x67, 0x78, 0x45, 0x7a, 0x41, 0x52, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x73, 0x4d, 0x43, 0x6b, 0x4e, 0x76, 0x62, 0x6e, 0x4e, 0x31, 0x62, 0x48, 0x52, 0x70, 0x62, 0x6d, 0x63, 0x78, 0x47, 0x44, 0x41, 0x57, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x4d, 0x4d, 0x44, 0x33, 0x64, 0x33, 0xa, 0x64, 0x79, 0x35, 0x33, 0x62, 0x32, 0x78, 0x6d, 0x63, 0x33, 0x4e, 0x73, 0x4c, 0x6d, 0x4e, 0x76, 0x62, 0x54, 0x45, 0x66, 0x4d, 0x42, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62, 0x33, 0x44, 0x51, 0x45, 0x4a, 0x41, 0x52, 0x59, 0x51, 0x61, 0x57, 0x35, 0x6d, 0x62, 0x30, 0x42, 0x33, 0x62, 0x32, 0x78, 0x6d, 0x63, 0x33, 0x4e, 0x73, 0x4c, 0x6d, 0x4e, 0x76, 0x62, 0x54, 0x41, 0x65, 0xa, 0x46, 0x77, 0x30, 0x78, 0x4f, 0x44, 0x41, 0x30, 0x4d, 0x54, 0x4d, 0x78, 0x4e, 0x54, 0x49, 0x7a, 0x4d, 0x44, 0x6c, 0x61, 0x46, 0x77, 0x30, 0x79, 0x4d, 0x54, 0x41, 0x78, 0x4d, 0x44, 0x63, 0x78, 0x4e, 0x54, 0x49, 0x7a, 0x4d, 0x44, 0x6c, 0x61, 0x4d, 0x49, 0x47, 0x55, 0x4d, 0x51, 0x73, 0x77, 0x43, 0x51, 0x59, 0x44, 0x56, 0x51, 0x51, 0x47, 0x45, 0x77, 0x4a, 0x56, 0x55, 0x7a, 0x45, 0x51, 0xa, 0x4d, 0x41, 0x34, 0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x41, 0x77, 0x48, 0x54, 0x57, 0x39, 0x75, 0x64, 0x47, 0x46, 0x75, 0x59, 0x54, 0x45, 0x51, 0x4d, 0x41, 0x34, 0x47, 0x41, 0x31, 0x55, 0x45, 0x42, 0x77, 0x77, 0x48, 0x51, 0x6d, 0x39, 0x36, 0x5a, 0x57, 0x31, 0x68, 0x62, 0x6a, 0x45, 0x52, 0x4d, 0x41, 0x38, 0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x67, 0x77, 0x49, 0x55, 0x32, 0x46, 0x33, 0xa, 0x64, 0x47, 0x39, 0x76, 0x64, 0x47, 0x67, 0x78, 0x45, 0x7a, 0x41, 0x52, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x73, 0x4d, 0x43, 0x6b, 0x4e, 0x76, 0x62, 0x6e, 0x4e, 0x31, 0x62, 0x48, 0x52, 0x70, 0x62, 0x6d, 0x63, 0x78, 0x47, 0x44, 0x41, 0x57, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x4d, 0x4d, 0x44, 0x33, 0x64, 0x33, 0x64, 0x79, 0x35, 0x33, 0x62, 0x32, 0x78, 0x6d, 0x63, 0x33, 0x4e, 0x73, 0xa, 0x4c, 0x6d, 0x4e, 0x76, 0x62, 0x54, 0x45, 0x66, 0x4d, 0x42, 0x30, 0x47, 0x43, 0x53, 0x71, 0x47, 0x53, 0x49, 0x62, 0x33, 0x44, 0x51, 0x45, 0x4a, 0x41, 0x52, 0x59, 0x51, 0x61, 0x57, 0x35, 0x6d, 0x62, 0x30, 0x42, 0x33, 0x62, 0x32, 0x78, 0x6d, 0x63, 0x33, 0x4e, 0x73, 0x4c, 0x6d, 0x4e, 0x76, 0x62, 0x54, 0x43, 0x43, 0x41, 0x53, 0x49, 0x77, 0x44, 0x51, 0x59, 0x4a, 0x4b, 0x6f, 0x5a, 0x49, 0xa, 0x68, 0x76, 0x63, 0x4e, 0x41, 0x51, 0x45, 0x42, 0x42, 0x51, 0x41, 0x44, 0x67, 0x67, 0x45, 0x50, 0x41, 0x44, 0x43, 0x43, 0x41, 0x51, 0x6f, 0x43, 0x67, 0x67, 0x45, 0x42, 0x41, 0x4c, 0x38, 0x4d, 0x79, 0x69, 0x30, 0x55, 0x73, 0x68, 0x36, 0x45, 0x51, 0x6c, 0x76, 0x4e, 0x4f, 0x42, 0x39, 0x4b, 0x38, 0x6b, 0x31, 0x31, 0x45, 0x50, 0x47, 0x32, 0x4e, 0x5a, 0x2f, 0x66, 0x79, 0x6e, 0x30, 0x44, 0xa, 0x6d, 0x4e, 0x4f, 0x73, 0x33, 0x67, 0x4e, 0x6d, 0x37, 0x69, 0x72, 0x78, 0x32, 0x4c, 0x42, 0x39, 0x62, 0x67, 0x64, 0x55, 0x43, 0x78, 0x43, 0x59, 0x49, 0x55, 0x32, 0x41, 0x79, 0x78, 0x49, 0x67, 0x35, 0x38, 0x78, 0x50, 0x33, 0x6b, 0x56, 0x39, 0x79, 0x58, 0x4a, 0x33, 0x4d, 0x75, 0x72, 0x4b, 0x6b, 0x4c, 0x74, 0x70, 0x55, 0x68, 0x41, 0x44, 0x4c, 0x36, 0x6a, 0x7a, 0x6c, 0x63, 0x58, 0x78, 0xa, 0x69, 0x32, 0x4a, 0x57, 0x47, 0x2b, 0x39, 0x6e, 0x62, 0x36, 0x51, 0x51, 0x51, 0x5a, 0x57, 0x74, 0x43, 0x70, 0x76, 0x6a, 0x70, 0x63, 0x43, 0x77, 0x30, 0x6e, 0x42, 0x32, 0x55, 0x44, 0x42, 0x62, 0x71, 0x4f, 0x67, 0x49, 0x4c, 0x48, 0x7a, 0x74, 0x70, 0x36, 0x4a, 0x36, 0x6a, 0x54, 0x67, 0x70, 0x48, 0x4b, 0x7a, 0x48, 0x37, 0x66, 0x4a, 0x38, 0x6c, 0x62, 0x43, 0x56, 0x67, 0x6e, 0x31, 0x4a, 0xa, 0x58, 0x44, 0x6a, 0x4e, 0x64, 0x79, 0x58, 0x76, 0x76, 0x59, 0x42, 0x31, 0x55, 0x35, 0x51, 0x38, 0x50, 0x63, 0x70, 0x6a, 0x57, 0x35, 0x38, 0x56, 0x74, 0x64, 0x4d, 0x64, 0x45, 0x79, 0x38, 0x5a, 0x30, 0x54, 0x7a, 0x62, 0x64, 0x6a, 0x72, 0x4d, 0x75, 0x48, 0x33, 0x4a, 0x35, 0x63, 0x4c, 0x58, 0x32, 0x6b, 0x42, 0x76, 0x32, 0x43, 0x48, 0x63, 0x63, 0x78, 0x74, 0x43, 0x4c, 0x56, 0x4f, 0x63, 0xa, 0x2f, 0x68, 0x72, 0x38, 0x66, 0x61, 0x74, 0x36, 0x4e, 0x6a, 0x2b, 0x59, 0x33, 0x6f, 0x52, 0x38, 0x42, 0x57, 0x66, 0x4f, 0x61, 0x68, 0x51, 0x34, 0x68, 0x36, 0x6e, 0x78, 0x6a, 0x4c, 0x56, 0x6f, 0x79, 0x32, 0x68, 0x2f, 0x63, 0x53, 0x41, 0x72, 0x39, 0x61, 0x42, 0x6a, 0x39, 0x56, 0x59, 0x76, 0x6f, 0x79, 0x62, 0x53, 0x74, 0x32, 0x2b, 0x78, 0x57, 0x68, 0x66, 0x58, 0x4f, 0x4a, 0x6b, 0x49, 0xa, 0x2f, 0x70, 0x4e, 0x59, 0x62, 0x2f, 0x37, 0x44, 0x45, 0x30, 0x6b, 0x49, 0x46, 0x67, 0x75, 0x6e, 0x54, 0x57, 0x63, 0x41, 0x55, 0x6a, 0x46, 0x6e, 0x49, 0x30, 0x36, 0x59, 0x37, 0x56, 0x46, 0x46, 0x48, 0x62, 0x6b, 0x45, 0x32, 0x51, 0x76, 0x73, 0x32, 0x43, 0x69, 0x7a, 0x53, 0x37, 0x33, 0x74, 0x4e, 0x6e, 0x6b, 0x43, 0x41, 0x77, 0x45, 0x41, 0x41, 0x61, 0x4f, 0x42, 0x2f, 0x44, 0x43, 0x42, 0xa, 0x2b, 0x54, 0x41, 0x64, 0x42, 0x67, 0x4e, 0x56, 0x48, 0x51, 0x34, 0x45, 0x46, 0x67, 0x51, 0x55, 0x4a, 0x34, 0x35, 0x6e, 0x45, 0x58, 0x54, 0x44, 0x4a, 0x68, 0x30, 0x2f, 0x37, 0x54, 0x4e, 0x6a, 0x73, 0x36, 0x54, 0x59, 0x48, 0x54, 0x44, 0x6c, 0x36, 0x4e, 0x55, 0x77, 0x67, 0x63, 0x6b, 0x47, 0x41, 0x31, 0x55, 0x64, 0x49, 0x77, 0x53, 0x42, 0x77, 0x54, 0x43, 0x42, 0x76, 0x6f, 0x41, 0x55, 0xa, 0x4a, 0x34, 0x35, 0x6e, 0x45, 0x58, 0x54, 0x44, 0x4a, 0x68, 0x30, 0x2f, 0x37, 0x54, 0x4e, 0x6a, 0x73, 0x36, 0x54, 0x59, 0x48, 0x54, 0x44, 0x6c, 0x36, 0x4e, 0x57, 0x68, 0x67, 0x5a, 0x71, 0x6b, 0x67, 0x5a, 0x63, 0x77, 0x67, 0x5a, 0x51, 0x78, 0x43, 0x7a, 0x41, 0x4a, 0x42, 0x67, 0x4e, 0x56, 0x42, 0x41, 0x59, 0x54, 0x41, 0x6c, 0x56, 0x54, 0x4d, 0x52, 0x41, 0x77, 0x44, 0x67, 0x59, 0x44, 0xa, 0x56, 0x51, 0x51, 0x49, 0x44, 0x41, 0x64, 0x4e, 0x62, 0x32, 0x35, 0x30, 0x59, 0x57, 0x35, 0x68, 0x4d, 0x52, 0x41, 0x77, 0x44, 0x67, 0x59, 0x44, 0x56, 0x51, 0x51, 0x48, 0x44, 0x41, 0x64, 0x43, 0x62, 0x33, 0x70, 0x6c, 0x62, 0x57, 0x46, 0x75, 0x4d, 0x52, 0x45, 0x77, 0x44, 0x77, 0x59, 0x44, 0x56, 0x51, 0x51, 0x4b, 0x44, 0x41, 0x68, 0x54, 0x59, 0x58, 0x64, 0x30, 0x62, 0x32, 0x39, 0x30, 0xa, 0x61, 0x44, 0x45, 0x54, 0x4d, 0x42, 0x45, 0x47, 0x41, 0x31, 0x55, 0x45, 0x43, 0x77, 0x77, 0x4b, 0x51, 0x32, 0x39, 0x75, 0x63, 0x33, 0x56, 0x73, 0x64, 0x47, 0x6c, 0x75, 0x5a, 0x7a, 0x45, 0x59, 0x4d, 0x42, 0x59, 0x47, 0x41, 0x31, 0x55, 0x45, 0x41, 0x77, 0x77, 0x50, 0x64, 0x33, 0x64, 0x33, 0x4c, 0x6e, 0x64, 0x76, 0x62, 0x47, 0x5a, 0x7a, 0x63, 0x32, 0x77, 0x75, 0x59, 0x32, 0x39, 0x74, 0xa, 0x4d, 0x52, 0x38, 0x77, 0x48, 0x51, 0x59, 0x4a, 0x4b, 0x6f, 0x5a, 0x49, 0x68, 0x76, 0x63, 0x4e, 0x41, 0x51, 0x6b, 0x42, 0x46, 0x68, 0x42, 0x70, 0x62, 0x6d, 0x5a, 0x76, 0x51, 0x48, 0x64, 0x76, 0x62, 0x47, 0x5a, 0x7a, 0x63, 0x32, 0x77, 0x75, 0x59, 0x32, 0x39, 0x74, 0x67, 0x67, 0x6b, 0x41, 0x68, 0x76, 0x2f, 0x31, 0x6a, 0x68, 0x44, 0x65, 0x75, 0x50, 0x73, 0x77, 0x44, 0x41, 0x59, 0x44, 0xa, 0x56, 0x52, 0x30, 0x54, 0x42, 0x41, 0x55, 0x77, 0x41, 0x77, 0x45, 0x42, 0x2f, 0x7a, 0x41, 0x4e, 0x42, 0x67, 0x6b, 0x71, 0x68, 0x6b, 0x69, 0x47, 0x39, 0x77, 0x30, 0x42, 0x41, 0x51, 0x73, 0x46, 0x41, 0x41, 0x4f, 0x43, 0x41, 0x51, 0x45, 0x41, 0x6e, 0x69, 0x69, 0x49, 0x63, 0x67, 0x44, 0x4b, 0x35, 0x75, 0x65, 0x58, 0x79, 0x73, 0x48, 0x78, 0x48, 0x35, 0x34, 0x53, 0x73, 0x72, 0x6a, 0x48, 0xa, 0x55, 0x65, 0x6f, 0x6f, 0x34, 0x54, 0x61, 0x31, 0x4c, 0x65, 0x59, 0x76, 0x43, 0x43, 0x50, 0x4c, 0x71, 0x55, 0x71, 0x48, 0x4a, 0x63, 0x5a, 0x64, 0x69, 0x55, 0x58, 0x71, 0x39, 0x51, 0x43, 0x59, 0x72, 0x48, 0x62, 0x37, 0x47, 0x36, 0x2f, 0x77, 0x7a, 0x6d, 0x53, 0x65, 0x32, 0x67, 0x69, 0x2f, 0x74, 0x75, 0x75, 0x30, 0x74, 0x51, 0x79, 0x67, 0x35, 0x2f, 0x5a, 0x48, 0x57, 0x52, 0x78, 0x68, 0xa, 0x7a, 0x79, 0x34, 0x4f, 0x57, 0x4b, 0x53, 0x43, 0x72, 0x41, 0x38, 0x2f, 0x37, 0x4d, 0x53, 0x75, 0x67, 0x50, 0x65, 0x77, 0x69, 0x68, 0x36, 0x46, 0x51, 0x65, 0x6a, 0x2f, 0x2f, 0x76, 0x35, 0x50, 0x47, 0x69, 0x54, 0x56, 0x53, 0x66, 0x72, 0x37, 0x2f, 0x6c, 0x37, 0x6c, 0x30, 0x35, 0x45, 0x4f, 0x54, 0x30, 0x34, 0x4d, 0x49, 0x56, 0x46, 0x78, 0x67, 0x77, 0x52, 0x72, 0x59, 0x6e, 0x74, 0x50, 0xa, 0x57, 0x58, 0x5a, 0x49, 0x67, 0x52, 0x36, 0x30, 0x39, 0x77, 0x52, 0x48, 0x69, 0x70, 0x46, 0x58, 0x6f, 0x78, 0x47, 0x70, 0x38, 0x69, 0x43, 0x30, 0x65, 0x44, 0x4e, 0x69, 0x50, 0x62, 0x42, 0x65, 0x44, 0x66, 0x6d, 0x47, 0x4f, 0x49, 0x4c, 0x61, 0x6f, 0x5a, 0x69, 0x4e, 0x47, 0x51, 0x61, 0x48, 0x49, 0x54, 0x6d, 0x33, 0x41, 0x76, 0x66, 0x61, 0x66, 0x56, 0x69, 0x36, 0x55, 0x68, 0x58, 0x59, 0xa, 0x4f, 0x38, 0x6c, 0x37, 0x57, 0x44, 0x53, 0x67, 0x78, 0x2b, 0x4a, 0x38, 0x71, 0x59, 0x4d, 0x54, 0x34, 0x62, 0x62, 0x73, 0x41, 0x62, 0x39, 0x53, 0x4d, 0x77, 0x76, 0x45, 0x2f, 0x6b, 0x50, 0x54, 0x78, 0x71, 0x53, 0x4f, 0x4c, 0x34, 0x64, 0x2f, 0x65, 0x6b, 0x54, 0x71, 0x79, 0x6c, 0x4e, 0x73, 0x68, 0x65, 0x31, 0x6c, 0x64, 0x6e, 0x4d, 0x78, 0x41, 0x30, 0x37, 0x71, 0x76, 0x54, 0x56, 0x55, 0xa, 0x45, 0x2f, 0x4e, 0x6b, 0x68, 0x32, 0x76, 0x66, 0x4e, 0x4e, 0x30, 0x30, 0x6f, 0x59, 0x67, 0x37, 0x32, 0x30, 0x32, 0x76, 0x47, 0x32, 0x53, 0x51, 0x6b, 0x6e, 0x45, 0x77, 0x6a, 0x73, 0x6a, 0x4d, 0x35, 0x57, 0x41, 0x6b, 0x72, 0x7a, 0x45, 0x57, 0x4f, 0x54, 0x4f, 0x52, 0x55, 0x50, 0x6d, 0x72, 0x61, 0x45, 0x4a, 0x30, 0x65, 0x6a, 0x58, 0x5a, 0x33, 0x63, 0x6a, 0x45, 0x55, 0x67, 0x3d, 0x3d, 0xa, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x43, 0x45, 0x52, 0x54, 0x49, 0x46, 0x49, 0x43, 0x41, 0x54, 0x45, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0xa };
    std::array<u8, 0x68f>  sample_server_key_pem  = { 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x52, 0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0xa, 0x4d, 0x49, 0x49, 0x45, 0x70, 0x51, 0x49, 0x42, 0x41, 0x41, 0x4b, 0x43, 0x41, 0x51, 0x45, 0x41, 0x77, 0x4a, 0x55, 0x49, 0x34, 0x56, 0x64, 0x42, 0x38, 0x6e, 0x46, 0x74, 0x74, 0x39, 0x4a, 0x46, 0x51, 0x53, 0x63, 0x42, 0x5a, 0x63, 0x5a, 0x46, 0x72, 0x76, 0x4b, 0x38, 0x4a, 0x44, 0x43, 0x34, 0x6c, 0x63, 0x34, 0x76, 0x54, 0x74, 0x62, 0x32, 0x48, 0x49, 0x69, 0x38, 0x66, 0x4a, 0x2f, 0x37, 0xa, 0x71, 0x47, 0x64, 0x2f, 0x2f, 0x6c, 0x79, 0x63, 0x55, 0x58, 0x58, 0x33, 0x69, 0x73, 0x6f, 0x48, 0x35, 0x7a, 0x55, 0x76, 0x6a, 0x2b, 0x47, 0x39, 0x65, 0x38, 0x41, 0x76, 0x66, 0x4b, 0x74, 0x6b, 0x71, 0x42, 0x66, 0x38, 0x79, 0x6c, 0x31, 0x37, 0x75, 0x75, 0x41, 0x68, 0x35, 0x58, 0x49, 0x75, 0x62, 0x79, 0x36, 0x47, 0x32, 0x4a, 0x56, 0x7a, 0x32, 0x71, 0x77, 0x62, 0x55, 0x37, 0x6c, 0x66, 0xa, 0x50, 0x39, 0x63, 0x5a, 0x44, 0x53, 0x56, 0x50, 0x34, 0x57, 0x4e, 0x6a, 0x55, 0x59, 0x73, 0x4c, 0x5a, 0x44, 0x2b, 0x74, 0x51, 0x37, 0x69, 0x6c, 0x48, 0x46, 0x77, 0x30, 0x73, 0x36, 0x34, 0x41, 0x6f, 0x47, 0x50, 0x46, 0x39, 0x6e, 0x38, 0x4c, 0x57, 0x57, 0x68, 0x34, 0x63, 0x36, 0x61, 0x4d, 0x47, 0x4b, 0x6b, 0x43, 0x62, 0x61, 0x2f, 0x44, 0x47, 0x51, 0x45, 0x75, 0x75, 0x42, 0x44, 0x6a, 0xa, 0x78, 0x73, 0x78, 0x41, 0x74, 0x47, 0x6d, 0x6a, 0x52, 0x6a, 0x4e, 0x70, 0x68, 0x32, 0x37, 0x45, 0x75, 0x78, 0x65, 0x6d, 0x38, 0x2b, 0x6a, 0x64, 0x72, 0x58, 0x4f, 0x38, 0x65, 0x79, 0x38, 0x68, 0x74, 0x66, 0x31, 0x6d, 0x55, 0x51, 0x79, 0x39, 0x56, 0x4c, 0x50, 0x68, 0x62, 0x56, 0x38, 0x63, 0x76, 0x43, 0x4e, 0x7a, 0x30, 0x51, 0x6b, 0x44, 0x69, 0x52, 0x54, 0x53, 0x45, 0x4c, 0x6c, 0x6b, 0xa, 0x77, 0x79, 0x72, 0x51, 0x6f, 0x5a, 0x5a, 0x4b, 0x76, 0x4f, 0x48, 0x55, 0x47, 0x6c, 0x76, 0x48, 0x6f, 0x4d, 0x44, 0x42, 0x59, 0x33, 0x67, 0x50, 0x52, 0x44, 0x63, 0x77, 0x4d, 0x70, 0x61, 0x41, 0x4d, 0x69, 0x4f, 0x56, 0x6f, 0x58, 0x65, 0x36, 0x45, 0x39, 0x4b, 0x58, 0x63, 0x2b, 0x4a, 0x64, 0x4a, 0x63, 0x6c, 0x71, 0x44, 0x63, 0x4d, 0x35, 0x59, 0x4b, 0x53, 0x30, 0x73, 0x47, 0x6c, 0x43, 0xa, 0x51, 0x67, 0x6e, 0x70, 0x32, 0x41, 0x69, 0x38, 0x4d, 0x79, 0x43, 0x7a, 0x57, 0x43, 0x4b, 0x6e, 0x71, 0x75, 0x76, 0x45, 0x34, 0x65, 0x5a, 0x68, 0x67, 0x38, 0x58, 0x53, 0x6c, 0x74, 0x2f, 0x5a, 0x30, 0x45, 0x2b, 0x74, 0x31, 0x77, 0x49, 0x44, 0x41, 0x51, 0x41, 0x42, 0x41, 0x6f, 0x49, 0x42, 0x41, 0x51, 0x43, 0x61, 0x30, 0x44, 0x51, 0x50, 0x55, 0x6d, 0x49, 0x46, 0x55, 0x41, 0x48, 0x76, 0xa, 0x6e, 0x2b, 0x31, 0x6b, 0x62, 0x73, 0x4c, 0x45, 0x32, 0x68, 0x72, 0x79, 0x68, 0x4e, 0x65, 0x53, 0x45, 0x45, 0x69, 0x53, 0x78, 0x4f, 0x6c, 0x71, 0x36, 0x34, 0x74, 0x31, 0x62, 0x4d, 0x5a, 0x35, 0x4f, 0x50, 0x4c, 0x4a, 0x63, 0x6b, 0x71, 0x47, 0x5a, 0x46, 0x53, 0x56, 0x64, 0x38, 0x76, 0x44, 0x6d, 0x70, 0x32, 0x33, 0x31, 0x42, 0x32, 0x6b, 0x41, 0x4d, 0x69, 0x65, 0x54, 0x75, 0x54, 0x64, 0xa, 0x78, 0x37, 0x70, 0x6e, 0x46, 0x73, 0x46, 0x30, 0x76, 0x4b, 0x6e, 0x57, 0x6c, 0x49, 0x38, 0x72, 0x4d, 0x42, 0x72, 0x37, 0x37, 0x64, 0x38, 0x68, 0x42, 0x53, 0x50, 0x5a, 0x53, 0x6a, 0x6d, 0x39, 0x6d, 0x47, 0x74, 0x6c, 0x6d, 0x72, 0x6a, 0x63, 0x78, 0x48, 0x33, 0x75, 0x70, 0x6b, 0x4d, 0x56, 0x4c, 0x6a, 0x32, 0x2b, 0x48, 0x53, 0x4a, 0x67, 0x4b, 0x6e, 0x4d, 0x77, 0x31, 0x54, 0x37, 0x59, 0xa, 0x6f, 0x71, 0x79, 0x47, 0x51, 0x79, 0x37, 0x45, 0x39, 0x57, 0x52, 0x65, 0x50, 0x34, 0x6c, 0x31, 0x44, 0x78, 0x48, 0x59, 0x55, 0x53, 0x56, 0x4f, 0x6e, 0x39, 0x69, 0x71, 0x6f, 0x38, 0x35, 0x67, 0x73, 0x2b, 0x4b, 0x4b, 0x32, 0x58, 0x34, 0x62, 0x38, 0x47, 0x54, 0x4b, 0x6d, 0x6c, 0x73, 0x46, 0x43, 0x31, 0x75, 0x71, 0x79, 0x2b, 0x58, 0x6a, 0x50, 0x32, 0x34, 0x79, 0x49, 0x67, 0x58, 0x7a, 0xa, 0x30, 0x50, 0x72, 0x76, 0x64, 0x46, 0x4b, 0x42, 0x34, 0x6c, 0x39, 0x30, 0x30, 0x37, 0x33, 0x2f, 0x4d, 0x59, 0x4e, 0x46, 0x64, 0x66, 0x70, 0x6a, 0x65, 0x70, 0x63, 0x75, 0x31, 0x72, 0x59, 0x5a, 0x78, 0x70, 0x49, 0x6d, 0x35, 0x43, 0x67, 0x47, 0x55, 0x46, 0x41, 0x4f, 0x65, 0x43, 0x36, 0x70, 0x65, 0x41, 0x30, 0x55, 0x6c, 0x37, 0x51, 0x53, 0x32, 0x44, 0x46, 0x41, 0x71, 0x36, 0x45, 0x42, 0xa, 0x51, 0x63, 0x49, 0x77, 0x2b, 0x41, 0x64, 0x66, 0x46, 0x75, 0x52, 0x68, 0x64, 0x39, 0x4a, 0x67, 0x38, 0x70, 0x2b, 0x4e, 0x36, 0x50, 0x53, 0x36, 0x36, 0x32, 0x50, 0x65, 0x4b, 0x70, 0x65, 0x42, 0x37, 0x30, 0x78, 0x73, 0x35, 0x6c, 0x55, 0x30, 0x55, 0x53, 0x73, 0x6f, 0x4e, 0x50, 0x52, 0x54, 0x48, 0x4d, 0x52, 0x59, 0x43, 0x6a, 0x2b, 0x37, 0x72, 0x37, 0x58, 0x33, 0x53, 0x6f, 0x56, 0x44, 0xa, 0x4c, 0x54, 0x7a, 0x78, 0x57, 0x46, 0x69, 0x42, 0x41, 0x6f, 0x47, 0x42, 0x41, 0x50, 0x49, 0x73, 0x56, 0x48, 0x59, 0x35, 0x49, 0x32, 0x50, 0x4a, 0x45, 0x44, 0x4b, 0x33, 0x6b, 0x36, 0x32, 0x76, 0x76, 0x68, 0x6c, 0x31, 0x6c, 0x6f, 0x46, 0x6b, 0x35, 0x72, 0x57, 0x34, 0x69, 0x55, 0x4a, 0x42, 0x30, 0x57, 0x33, 0x51, 0x48, 0x42, 0x76, 0x34, 0x47, 0x36, 0x78, 0x70, 0x79, 0x7a, 0x59, 0x38, 0xa, 0x5a, 0x48, 0x33, 0x63, 0x39, 0x42, 0x6d, 0x34, 0x77, 0x32, 0x43, 0x78, 0x56, 0x30, 0x68, 0x66, 0x55, 0x6b, 0x39, 0x5a, 0x4f, 0x6c, 0x56, 0x2f, 0x4d, 0x73, 0x41, 0x5a, 0x51, 0x31, 0x41, 0x2f, 0x72, 0x73, 0x35, 0x76, 0x46, 0x2f, 0x4d, 0x4f, 0x6e, 0x30, 0x44, 0x4b, 0x54, 0x71, 0x30, 0x56, 0x4f, 0x38, 0x6c, 0x35, 0x36, 0x63, 0x42, 0x5a, 0x4f, 0x48, 0x4e, 0x77, 0x6e, 0x41, 0x70, 0x38, 0xa, 0x79, 0x54, 0x70, 0x49, 0x4d, 0x71, 0x66, 0x59, 0x53, 0x58, 0x55, 0x4b, 0x68, 0x63, 0x4c, 0x43, 0x2f, 0x52, 0x56, 0x7a, 0x32, 0x70, 0x6b, 0x4a, 0x4b, 0x6d, 0x6d, 0x61, 0x6e, 0x77, 0x70, 0x78, 0x76, 0x37, 0x41, 0x45, 0x70, 0x6f, 0x78, 0x36, 0x57, 0x6d, 0x39, 0x49, 0x57, 0x6c, 0x51, 0x37, 0x78, 0x72, 0x46, 0x54, 0x46, 0x39, 0x2f, 0x6e, 0x41, 0x6f, 0x47, 0x42, 0x41, 0x4d, 0x75, 0x54, 0xa, 0x33, 0x6e, 0x63, 0x56, 0x58, 0x62, 0x64, 0x63, 0x58, 0x48, 0x7a, 0x59, 0x6b, 0x4b, 0x6d, 0x59, 0x4c, 0x64, 0x5a, 0x70, 0x44, 0x6d, 0x4f, 0x7a, 0x6f, 0x39, 0x79, 0x6d, 0x7a, 0x49, 0x74, 0x71, 0x70, 0x4b, 0x49, 0x53, 0x6a, 0x49, 0x35, 0x37, 0x53, 0x43, 0x79, 0x79, 0x53, 0x7a, 0x66, 0x63, 0x42, 0x68, 0x68, 0x39, 0x36, 0x76, 0x35, 0x32, 0x6f, 0x64, 0x53, 0x68, 0x36, 0x54, 0x38, 0x4e, 0xa, 0x7a, 0x52, 0x74, 0x66, 0x72, 0x31, 0x2b, 0x65, 0x6c, 0x6c, 0x74, 0x62, 0x44, 0x36, 0x46, 0x38, 0x72, 0x37, 0x4f, 0x62, 0x6b, 0x4e, 0x74, 0x58, 0x63, 0x7a, 0x72, 0x74, 0x73, 0x43, 0x4e, 0x45, 0x72, 0x6b, 0x46, 0x50, 0x48, 0x77, 0x64, 0x43, 0x45, 0x79, 0x4e, 0x4d, 0x79, 0x2f, 0x72, 0x30, 0x46, 0x4b, 0x54, 0x56, 0x39, 0x35, 0x34, 0x32, 0x66, 0x46, 0x75, 0x66, 0x71, 0x44, 0x7a, 0x42, 0xa, 0x68, 0x56, 0x39, 0x30, 0x30, 0x6a, 0x6b, 0x74, 0x2f, 0x39, 0x43, 0x45, 0x33, 0x2f, 0x75, 0x7a, 0x49, 0x48, 0x6f, 0x75, 0x6d, 0x78, 0x65, 0x75, 0x35, 0x72, 0x6f, 0x4c, 0x72, 0x6c, 0x39, 0x54, 0x70, 0x46, 0x4c, 0x74, 0x47, 0x38, 0x53, 0x52, 0x41, 0x6f, 0x47, 0x42, 0x41, 0x4f, 0x79, 0x59, 0x32, 0x72, 0x76, 0x56, 0x2f, 0x76, 0x6c, 0x53, 0x53, 0x6e, 0x30, 0x43, 0x56, 0x55, 0x6c, 0x76, 0xa, 0x56, 0x57, 0x35, 0x53, 0x4c, 0x34, 0x53, 0x6a, 0x4b, 0x37, 0x4f, 0x47, 0x59, 0x72, 0x4e, 0x55, 0x30, 0x6d, 0x4e, 0x53, 0x32, 0x75, 0x4f, 0x49, 0x64, 0x71, 0x44, 0x76, 0x69, 0x78, 0x57, 0x6c, 0x30, 0x78, 0x67, 0x55, 0x63, 0x6e, 0x64, 0x65, 0x78, 0x36, 0x4d, 0x45, 0x48, 0x35, 0x34, 0x5a, 0x59, 0x72, 0x55, 0x62, 0x47, 0x35, 0x37, 0x44, 0x38, 0x72, 0x55, 0x79, 0x2b, 0x55, 0x7a, 0x42, 0xa, 0x71, 0x75, 0x73, 0x4d, 0x4a, 0x6e, 0x33, 0x55, 0x58, 0x30, 0x70, 0x52, 0x58, 0x4b, 0x52, 0x46, 0x42, 0x6e, 0x42, 0x45, 0x70, 0x31, 0x62, 0x41, 0x31, 0x43, 0x49, 0x55, 0x64, 0x70, 0x37, 0x59, 0x59, 0x31, 0x43, 0x4a, 0x6b, 0x4e, 0x50, 0x69, 0x76, 0x34, 0x47, 0x56, 0x6b, 0x6a, 0x46, 0x42, 0x68, 0x7a, 0x6b, 0x61, 0x51, 0x77, 0x73, 0x59, 0x70, 0x56, 0x4d, 0x66, 0x4f, 0x52, 0x70, 0x66, 0xa, 0x48, 0x30, 0x4f, 0x38, 0x68, 0x32, 0x72, 0x66, 0x62, 0x74, 0x4d, 0x69, 0x41, 0x50, 0x34, 0x69, 0x6d, 0x48, 0x42, 0x4f, 0x47, 0x68, 0x6b, 0x70, 0x41, 0x6f, 0x47, 0x42, 0x41, 0x49, 0x70, 0x42, 0x56, 0x69, 0x68, 0x52, 0x6e, 0x6c, 0x2f, 0x55, 0x6e, 0x67, 0x73, 0x37, 0x6d, 0x4b, 0x4e, 0x55, 0x38, 0x6d, 0x78, 0x57, 0x31, 0x4b, 0x72, 0x70, 0x61, 0x54, 0x4f, 0x46, 0x4a, 0x41, 0x7a, 0x61, 0xa, 0x31, 0x41, 0x77, 0x74, 0x78, 0x4c, 0x39, 0x50, 0x41, 0x6d, 0x6b, 0x34, 0x66, 0x4e, 0x54, 0x6d, 0x33, 0x45, 0x7a, 0x74, 0x31, 0x78, 0x59, 0x52, 0x77, 0x7a, 0x34, 0x41, 0x35, 0x38, 0x4d, 0x6d, 0x77, 0x46, 0x45, 0x43, 0x33, 0x72, 0x74, 0x31, 0x6e, 0x47, 0x39, 0x57, 0x6e, 0x48, 0x72, 0x7a, 0x6a, 0x75, 0x2f, 0x50, 0x69, 0x73, 0x55, 0x72, 0x30, 0x74, 0x6f, 0x47, 0x61, 0x6b, 0x54, 0x4a, 0xa, 0x63, 0x2f, 0x35, 0x75, 0x6d, 0x59, 0x66, 0x34, 0x57, 0x37, 0x37, 0x78, 0x66, 0x4f, 0x5a, 0x6c, 0x74, 0x55, 0x39, 0x73, 0x38, 0x4d, 0x6e, 0x46, 0x2f, 0x78, 0x62, 0x4b, 0x69, 0x78, 0x73, 0x58, 0x34, 0x6c, 0x67, 0x39, 0x6f, 0x6a, 0x65, 0x72, 0x41, 0x62, 0x79, 0x2f, 0x51, 0x4d, 0x35, 0x54, 0x6a, 0x49, 0x37, 0x74, 0x37, 0x2b, 0x35, 0x5a, 0x6e, 0x65, 0x42, 0x6a, 0x35, 0x6e, 0x78, 0x65, 0xa, 0x39, 0x59, 0x35, 0x4c, 0x38, 0x54, 0x76, 0x42, 0x41, 0x6f, 0x47, 0x41, 0x54, 0x55, 0x58, 0x35, 0x51, 0x49, 0x7a, 0x46, 0x57, 0x2f, 0x51, 0x71, 0x47, 0x6f, 0x71, 0x30, 0x38, 0x68, 0x79, 0x73, 0x61, 0x2b, 0x6b, 0x4d, 0x56, 0x6a, 0x61, 0x33, 0x54, 0x6e, 0x4b, 0x57, 0x31, 0x65, 0x57, 0x4b, 0x30, 0x75, 0x4c, 0x2f, 0x38, 0x66, 0x45, 0x59, 0x45, 0x7a, 0x32, 0x47, 0x43, 0x62, 0x6a, 0x59, 0xa, 0x64, 0x71, 0x66, 0x4a, 0x48, 0x48, 0x46, 0x53, 0x6c, 0x44, 0x42, 0x44, 0x34, 0x50, 0x46, 0x34, 0x64, 0x50, 0x31, 0x68, 0x47, 0x30, 0x77, 0x4a, 0x7a, 0x4f, 0x5a, 0x6f, 0x4b, 0x6e, 0x47, 0x74, 0x48, 0x4e, 0x39, 0x44, 0x76, 0x46, 0x62, 0x62, 0x70, 0x61, 0x53, 0x2b, 0x4e, 0x58, 0x43, 0x6b, 0x58, 0x73, 0x39, 0x50, 0x2f, 0x41, 0x42, 0x56, 0x6d, 0x54, 0x6f, 0x39, 0x49, 0x38, 0x39, 0x6e, 0xa, 0x57, 0x76, 0x55, 0x69, 0x2b, 0x4c, 0x55, 0x70, 0x30, 0x45, 0x51, 0x52, 0x36, 0x7a, 0x55, 0x75, 0x52, 0x72, 0x37, 0x39, 0x6a, 0x68, 0x69, 0x79, 0x58, 0x36, 0x69, 0x2f, 0x47, 0x54, 0x4b, 0x68, 0x39, 0x64, 0x77, 0x44, 0x35, 0x6e, 0x79, 0x61, 0x48, 0x77, 0x78, 0x38, 0x71, 0x62, 0x41, 0x4f, 0x49, 0x54, 0x63, 0x37, 0x38, 0x62, 0x41, 0x3d, 0xa, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x52, 0x53, 0x41, 0x20, 0x50, 0x52, 0x49, 0x56, 0x41, 0x54, 0x45, 0x20, 0x4b, 0x45, 0x59, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0xa };
    std::array<u8, 0x594>  sample_dh2048_pem      = { 0x44, 0x69, 0x66, 0x66, 0x69, 0x65, 0x2d, 0x48, 0x65, 0x6c, 0x6c, 0x6d, 0x61, 0x6e, 0x2d, 0x50, 0x61, 0x72, 0x61, 0x6d, 0x65, 0x74, 0x65, 0x72, 0x73, 0x3a, 0x20, 0x28, 0x32, 0x30, 0x34, 0x38, 0x20, 0x62, 0x69, 0x74, 0x29, 0xa, 0x20, 0x20, 0x20, 0x20, 0x70, 0x72, 0x69, 0x6d, 0x65, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x30, 0x30, 0x3a, 0x62, 0x30, 0x3a, 0x61, 0x31, 0x3a, 0x30, 0x38, 0x3a, 0x30, 0x36, 0x3a, 0x39, 0x63, 0x3a, 0x30, 0x38, 0x3a, 0x31, 0x33, 0x3a, 0x62, 0x61, 0x3a, 0x35, 0x39, 0x3a, 0x30, 0x36, 0x3a, 0x33, 0x63, 0x3a, 0x62, 0x63, 0x3a, 0x33, 0x30, 0x3a, 0x64, 0x35, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x35, 0x3a, 0x30, 0x30, 0x3a, 0x63, 0x31, 0x3a, 0x34, 0x66, 0x3a, 0x34, 0x34, 0x3a, 0x61, 0x37, 0x3a, 0x64, 0x36, 0x3a, 0x65, 0x66, 0x3a, 0x34, 0x61, 0x3a, 0x63, 0x36, 0x3a, 0x32, 0x35, 0x3a, 0x32, 0x37, 0x3a, 0x31, 0x63, 0x3a, 0x65, 0x38, 0x3a, 0x64, 0x32, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x39, 0x36, 0x3a, 0x35, 0x33, 0x3a, 0x30, 0x61, 0x3a, 0x35, 0x63, 0x3a, 0x39, 0x31, 0x3a, 0x64, 0x64, 0x3a, 0x61, 0x32, 0x3a, 0x63, 0x32, 0x3a, 0x39, 0x34, 0x3a, 0x38, 0x34, 0x3a, 0x62, 0x66, 0x3a, 0x37, 0x64, 0x3a, 0x62, 0x32, 0x3a, 0x34, 0x34, 0x3a, 0x39, 0x66, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x39, 0x62, 0x3a, 0x64, 0x32, 0x3a, 0x63, 0x31, 0x3a, 0x38, 0x61, 0x3a, 0x63, 0x35, 0x3a, 0x62, 0x65, 0x3a, 0x37, 0x32, 0x3a, 0x35, 0x63, 0x3a, 0x61, 0x37, 0x3a, 0x65, 0x37, 0x3a, 0x39, 0x31, 0x3a, 0x65, 0x36, 0x3a, 0x64, 0x34, 0x3a, 0x39, 0x66, 0x3a, 0x37, 0x33, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x30, 0x37, 0x3a, 0x38, 0x35, 0x3a, 0x35, 0x62, 0x3a, 0x36, 0x36, 0x3a, 0x34, 0x38, 0x3a, 0x63, 0x37, 0x3a, 0x37, 0x30, 0x3a, 0x66, 0x61, 0x3a, 0x62, 0x34, 0x3a, 0x65, 0x65, 0x3a, 0x30, 0x32, 0x3a, 0x63, 0x39, 0x3a, 0x33, 0x64, 0x3a, 0x39, 0x61, 0x3a, 0x34, 0x61, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x61, 0x3a, 0x33, 0x64, 0x3a, 0x63, 0x31, 0x3a, 0x34, 0x36, 0x3a, 0x33, 0x65, 0x3a, 0x31, 0x39, 0x3a, 0x36, 0x39, 0x3a, 0x64, 0x31, 0x3a, 0x31, 0x37, 0x3a, 0x34, 0x36, 0x3a, 0x30, 0x37, 0x3a, 0x61, 0x33, 0x3a, 0x34, 0x64, 0x3a, 0x39, 0x66, 0x3a, 0x32, 0x62, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x39, 0x36, 0x3a, 0x31, 0x37, 0x3a, 0x33, 0x39, 0x3a, 0x36, 0x64, 0x3a, 0x33, 0x30, 0x3a, 0x38, 0x64, 0x3a, 0x32, 0x61, 0x3a, 0x66, 0x33, 0x3a, 0x39, 0x34, 0x3a, 0x64, 0x33, 0x3a, 0x37, 0x35, 0x3a, 0x63, 0x66, 0x3a, 0x61, 0x30, 0x3a, 0x37, 0x35, 0x3a, 0x65, 0x36, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x32, 0x3a, 0x39, 0x32, 0x3a, 0x31, 0x66, 0x3a, 0x31, 0x61, 0x3a, 0x37, 0x30, 0x3a, 0x30, 0x35, 0x3a, 0x61, 0x61, 0x3a, 0x30, 0x34, 0x3a, 0x38, 0x33, 0x3a, 0x35, 0x37, 0x3a, 0x33, 0x30, 0x3a, 0x66, 0x62, 0x3a, 0x64, 0x61, 0x3a, 0x37, 0x36, 0x3a, 0x39, 0x33, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x38, 0x3a, 0x35, 0x30, 0x3a, 0x65, 0x38, 0x3a, 0x32, 0x37, 0x3a, 0x66, 0x64, 0x3a, 0x36, 0x33, 0x3a, 0x65, 0x65, 0x3a, 0x33, 0x63, 0x3a, 0x65, 0x35, 0x3a, 0x62, 0x37, 0x3a, 0x63, 0x38, 0x3a, 0x30, 0x39, 0x3a, 0x61, 0x65, 0x3a, 0x36, 0x66, 0x3a, 0x35, 0x30, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x35, 0x3a, 0x38, 0x65, 0x3a, 0x38, 0x34, 0x3a, 0x63, 0x65, 0x3a, 0x34, 0x61, 0x3a, 0x30, 0x30, 0x3a, 0x65, 0x39, 0x3a, 0x31, 0x32, 0x3a, 0x37, 0x65, 0x3a, 0x35, 0x61, 0x3a, 0x33, 0x31, 0x3a, 0x64, 0x37, 0x3a, 0x33, 0x33, 0x3a, 0x66, 0x63, 0x3a, 0x32, 0x31, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x31, 0x33, 0x3a, 0x37, 0x36, 0x3a, 0x63, 0x63, 0x3a, 0x31, 0x36, 0x3a, 0x33, 0x30, 0x3a, 0x64, 0x62, 0x3a, 0x30, 0x63, 0x3a, 0x66, 0x63, 0x3a, 0x63, 0x35, 0x3a, 0x36, 0x32, 0x3a, 0x61, 0x37, 0x3a, 0x33, 0x35, 0x3a, 0x62, 0x38, 0x3a, 0x65, 0x66, 0x3a, 0x62, 0x37, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x62, 0x30, 0x3a, 0x61, 0x63, 0x3a, 0x63, 0x30, 0x3a, 0x33, 0x36, 0x3a, 0x66, 0x36, 0x3a, 0x64, 0x39, 0x3a, 0x63, 0x39, 0x3a, 0x34, 0x36, 0x3a, 0x34, 0x38, 0x3a, 0x66, 0x39, 0x3a, 0x34, 0x30, 0x3a, 0x39, 0x30, 0x3a, 0x30, 0x30, 0x3a, 0x32, 0x62, 0x3a, 0x31, 0x62, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x61, 0x61, 0x3a, 0x36, 0x63, 0x3a, 0x65, 0x33, 0x3a, 0x31, 0x61, 0x3a, 0x63, 0x33, 0x3a, 0x30, 0x62, 0x3a, 0x30, 0x33, 0x3a, 0x39, 0x65, 0x3a, 0x31, 0x62, 0x3a, 0x63, 0x32, 0x3a, 0x34, 0x36, 0x3a, 0x65, 0x34, 0x3a, 0x34, 0x38, 0x3a, 0x34, 0x65, 0x3a, 0x32, 0x32, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x37, 0x33, 0x3a, 0x36, 0x66, 0x3a, 0x63, 0x33, 0x3a, 0x35, 0x66, 0x3a, 0x64, 0x34, 0x3a, 0x39, 0x61, 0x3a, 0x64, 0x36, 0x3a, 0x33, 0x30, 0x3a, 0x30, 0x37, 0x3a, 0x34, 0x38, 0x3a, 0x64, 0x36, 0x3a, 0x38, 0x63, 0x3a, 0x39, 0x30, 0x3a, 0x61, 0x62, 0x3a, 0x64, 0x34, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x36, 0x3a, 0x66, 0x31, 0x3a, 0x65, 0x33, 0x3a, 0x34, 0x38, 0x3a, 0x64, 0x33, 0x3a, 0x35, 0x38, 0x3a, 0x34, 0x62, 0x3a, 0x61, 0x36, 0x3a, 0x62, 0x39, 0x3a, 0x63, 0x64, 0x3a, 0x32, 0x39, 0x3a, 0x62, 0x66, 0x3a, 0x36, 0x38, 0x3a, 0x31, 0x66, 0x3a, 0x30, 0x38, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x34, 0x62, 0x3a, 0x36, 0x33, 0x3a, 0x38, 0x36, 0x3a, 0x32, 0x66, 0x3a, 0x35, 0x63, 0x3a, 0x36, 0x62, 0x3a, 0x64, 0x36, 0x3a, 0x62, 0x36, 0x3a, 0x30, 0x36, 0x3a, 0x36, 0x35, 0x3a, 0x66, 0x37, 0x3a, 0x61, 0x36, 0x3a, 0x64, 0x63, 0x3a, 0x30, 0x30, 0x3a, 0x36, 0x37, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x36, 0x62, 0x3a, 0x62, 0x62, 0x3a, 0x63, 0x33, 0x3a, 0x61, 0x39, 0x3a, 0x34, 0x31, 0x3a, 0x38, 0x33, 0x3a, 0x66, 0x62, 0x3a, 0x63, 0x37, 0x3a, 0x66, 0x61, 0x3a, 0x63, 0x38, 0x3a, 0x65, 0x32, 0x3a, 0x31, 0x65, 0x3a, 0x37, 0x65, 0x3a, 0x61, 0x66, 0x3a, 0x30, 0x30, 0x3a, 0xa, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x33, 0x66, 0x3a, 0x39, 0x33, 0xa, 0x20, 0x20, 0x20, 0x20, 0x67, 0x65, 0x6e, 0x65, 0x72, 0x61, 0x74, 0x6f, 0x72, 0x3a, 0x20, 0x32, 0x20, 0x28, 0x30, 0x78, 0x32, 0x29, 0xa, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x42, 0x45, 0x47, 0x49, 0x4e, 0x20, 0x44, 0x48, 0x20, 0x50, 0x41, 0x52, 0x41, 0x4d, 0x45, 0x54, 0x45, 0x52, 0x53, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0xa, 0x4d, 0x49, 0x49, 0x42, 0x43, 0x41, 0x4b, 0x43, 0x41, 0x51, 0x45, 0x41, 0x73, 0x4b, 0x45, 0x49, 0x42, 0x70, 0x77, 0x49, 0x45, 0x37, 0x70, 0x5a, 0x42, 0x6a, 0x79, 0x38, 0x4d, 0x4e, 0x58, 0x31, 0x41, 0x4d, 0x46, 0x50, 0x52, 0x4b, 0x66, 0x57, 0x37, 0x30, 0x72, 0x47, 0x4a, 0x53, 0x63, 0x63, 0x36, 0x4e, 0x4b, 0x57, 0x55, 0x77, 0x70, 0x63, 0x6b, 0x64, 0x32, 0x69, 0x77, 0x70, 0x53, 0x45, 0xa, 0x76, 0x33, 0x32, 0x79, 0x52, 0x4a, 0x2b, 0x62, 0x30, 0x73, 0x47, 0x4b, 0x78, 0x62, 0x35, 0x79, 0x58, 0x4b, 0x66, 0x6e, 0x6b, 0x65, 0x62, 0x55, 0x6e, 0x33, 0x4d, 0x48, 0x68, 0x56, 0x74, 0x6d, 0x53, 0x4d, 0x64, 0x77, 0x2b, 0x72, 0x54, 0x75, 0x41, 0x73, 0x6b, 0x39, 0x6d, 0x6b, 0x72, 0x61, 0x50, 0x63, 0x46, 0x47, 0x50, 0x68, 0x6c, 0x70, 0x30, 0x52, 0x64, 0x47, 0x42, 0x36, 0x4e, 0x4e, 0xa, 0x6e, 0x79, 0x75, 0x57, 0x46, 0x7a, 0x6c, 0x74, 0x4d, 0x49, 0x30, 0x71, 0x38, 0x35, 0x54, 0x54, 0x64, 0x63, 0x2b, 0x67, 0x64, 0x65, 0x62, 0x79, 0x6b, 0x68, 0x38, 0x61, 0x63, 0x41, 0x57, 0x71, 0x42, 0x49, 0x4e, 0x58, 0x4d, 0x50, 0x76, 0x61, 0x64, 0x70, 0x4d, 0x34, 0x55, 0x4f, 0x67, 0x6e, 0x2f, 0x57, 0x50, 0x75, 0x50, 0x4f, 0x57, 0x33, 0x79, 0x41, 0x6d, 0x75, 0x62, 0x31, 0x41, 0x31, 0xa, 0x6a, 0x6f, 0x54, 0x4f, 0x53, 0x67, 0x44, 0x70, 0x45, 0x6e, 0x35, 0x61, 0x4d, 0x64, 0x63, 0x7a, 0x2f, 0x43, 0x45, 0x54, 0x64, 0x73, 0x77, 0x57, 0x4d, 0x4e, 0x73, 0x4d, 0x2f, 0x4d, 0x56, 0x69, 0x70, 0x7a, 0x57, 0x34, 0x37, 0x37, 0x65, 0x77, 0x72, 0x4d, 0x41, 0x32, 0x39, 0x74, 0x6e, 0x4a, 0x52, 0x6b, 0x6a, 0x35, 0x51, 0x4a, 0x41, 0x41, 0x4b, 0x78, 0x75, 0x71, 0x62, 0x4f, 0x4d, 0x61, 0xa, 0x77, 0x77, 0x73, 0x44, 0x6e, 0x68, 0x76, 0x43, 0x52, 0x75, 0x52, 0x49, 0x54, 0x69, 0x4a, 0x7a, 0x62, 0x38, 0x4e, 0x66, 0x31, 0x4a, 0x72, 0x57, 0x4d, 0x41, 0x64, 0x49, 0x31, 0x6f, 0x79, 0x51, 0x71, 0x39, 0x54, 0x32, 0x38, 0x65, 0x4e, 0x49, 0x30, 0x31, 0x68, 0x4c, 0x70, 0x72, 0x6e, 0x4e, 0x4b, 0x62, 0x39, 0x6f, 0x48, 0x77, 0x68, 0x4c, 0x59, 0x34, 0x59, 0x76, 0x58, 0x47, 0x76, 0x57, 0xa, 0x74, 0x67, 0x5a, 0x6c, 0x39, 0x36, 0x62, 0x63, 0x41, 0x47, 0x64, 0x72, 0x75, 0x38, 0x4f, 0x70, 0x51, 0x59, 0x50, 0x37, 0x78, 0x2f, 0x72, 0x49, 0x34, 0x68, 0x35, 0x2b, 0x72, 0x77, 0x41, 0x2f, 0x6b, 0x77, 0x49, 0x42, 0x41, 0x67, 0x3d, 0x3d, 0xa, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0x45, 0x4e, 0x44, 0x20, 0x44, 0x48, 0x20, 0x50, 0x41, 0x52, 0x41, 0x4d, 0x45, 0x54, 0x45, 0x52, 0x53, 0x2d, 0x2d, 0x2d, 0x2d, 0x2d, 0xa };
}

#endif