# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['django_generic_tasks', 'django_generic_tasks.migrations']

package_data = \
{'': ['*']}

install_requires = \
['CacheControl>=0.12.11,<0.13.0',
 'Django>=4.0.4,<5.0.0',
 'django-ninja>=0.17.0,<0.18.0',
 'google-auth>=2.6.6,<3.0.0',
 'google-cloud-tasks>=2.8.1,<3.0.0',
 'pydantic>=1.9.0,<2.0.0',
 'requests>=2.27.1,<3.0.0']

setup_kwargs = {
    'name': 'django-generic-tasks',
    'version': '0.3.1',
    'description': '',
    'long_description': '# django-generic-tasks\n\nActive Job for Django\n\n## Example usage\n\n```\nfrom django.core.mail import send_mail\nfrom pydantic import BaseModel\n\nimport django_generic_tasks as tasks\n\n\n# define task params as a Pydantic BaseModel\nclass EmailNotificationParams(BaseModel):\n    subject: str\n    content: str\n    recipients: list[str]\n\n\n# subclass Task and specify params type as a generic type argument and implement the run method\nclass EmailNotificationTask(tasks.Task[EmailNotificationParams]):\n    def run(self):\n        send_mail(\n            subject=self.params.subject,\n            message=self.params.content,\n            from_email=None,\n            recipient_list=self.params.recipients,\n        )\n\n\nif __name__ == "__main__":\n    params = EmailNotificationParams(\n        subject="Hello",\n        content="Have a nice day",\n        recipients=["alice@example.com", "bob@example.com"],\n    )\n    task = EmailNotificationTask(params)\n\n    # run a task synchronously\n    task.run()\n\n    # run a task asynchronously using settings.TASKS_BACKEND\n    task.start()\n```\n\n## Registering tasks\n\nSimilar to [signals](https://docs.djangoproject.com/en/4.0/topics/signals/#connecting-receiver-functions-1), tasks have\nto be implicitly registered by ensuring they are imported during application startup. This can be done in the `ready`\nmethod in an application\'s `AppConfig`.\n\n[my_app/apps.py](django_generic_tasks_example/my_app/apps.py)\n\n```\nfrom django.apps import AppConfig\n\n\nclass MyAppConfig(AppConfig):\n    default_auto_field = "django.db.models.BigAutoField"\n    name = "my_app"\n\n    def ready(self):\n        from . import tasks  # noqa: F401\n```\n\n## HTTP endpoints\n\n`django-generic-tasks` uses `django-ninja` to automatically expose API endpoints for running tasks. Each defined task\ngets its own API\nendpoint and uses the specified Pydantic `BaseModel` for parameter verification.\n\n![autogenerated OpenAPI docs](https://user-images.githubusercontent.com/11734309/165979039-df4d2bfe-4c38-4798-af2e-fd0792303608.png)\n\n### Example configuration\n\nCreate your own `NinjaAPI` instance and mount `django_generic_tasks.api.router`.\n\n[django_generic_tasks_example/api.py](django_generic_tasks_example/django_generic_tasks_example/api.py)\n\n```\nfrom ninja import NinjaAPI\n\nimport django_generic_tasks.api\n\napi = NinjaAPI()\napi.add_router("/tasks/", django_generic_tasks.api.router)\n```\n\nAdd your `NinjaAPI` instance in `urls.py`:\n\n[django_generic_tasks_example/urls.py](django_generic_tasks_example/django_generic_tasks_example/urls.py)\n\n```\nfrom django.contrib import admin\nfrom django.urls import path\n\nfrom .api import api\n\nurlpatterns = [\n    path("admin/", admin.site.urls),\n    path("api/", api.urls),\n]\n```\n\n## Supported backends\n\n### ThreadingBackend\n\nRuns tasks in a new `threading.Thread`.\n\n### CloudTasksBackend\n\nRuns tasks using [Cloud Tasks](https://cloud.google.com/tasks) HTTP Target tasks.\n\n## Configuration\n\n### `TASKS_API_AUTHENTICATION`\n\nSpecifies what authentication should be required to run tasks via HTTP.\n\nType: `Optional[str]`\n\nSupported values:\n\n* `"oidc"` - Require API requests to contain a valid Google-issued OIDC token\n* `None` (default) - No authentication\n\n### `TASKS_BACKEND`\n\nThe default backend to use to run tasks asynchronously.\n\nType: any class which implements the `django_generic_tasks.backends.Backend` protocol\n',
    'author': 'Jiayu Yi',
    'author_email': 'yijiayu@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
