import { MainAreaWidget } from '@jupyterlab/apputils';
import { Panel, Widget } from '@lumino/widgets';
import { PromiseDelegate } from '@lumino/coreutils';
import { generateNode, requestAPI } from '../../utils';
import { Discussion } from '../discussion/Discussion';
/**
 * DescriptionWidget component
 */
export class DescriptionWidget extends MainAreaWidget {
    constructor(props) {
        const content = new Panel();
        const isLoaded = new PromiseDelegate();
        super({
            content,
            reveal: isLoaded.promise
        });
        this.pullRequestId = props.pullRequest.id;
        this.renderMime = props.renderMime;
        content.addClass('jp-PullRequestTab');
        const container = new Panel();
        container.addClass('jp-PullRequestDescriptionTab');
        container.addWidget(DescriptionWidget.createHeader(props.pullRequest.title, props.pullRequest.link));
        const markdownRenderer = props.renderMime.createRenderer('text/markdown');
        container.addWidget(markdownRenderer);
        Promise.all([
            markdownRenderer.renderModel({
                data: {
                    'text/markdown': props.pullRequest.body
                },
                trusted: false,
                metadata: {},
                setData: () => null
            }),
            this.loadComments(container, props.pullRequest.id, props.renderMime)
        ])
            .then(() => {
            isLoaded.resolve();
            this.content.addWidget(container);
        })
            .catch(reason => {
            isLoaded.reject(reason);
        });
    }
    /**
     * Load the discussion of the pull request
     *
     * @param container Discussion widgets container
     * @param pullRequestId Pull request id
     * @param renderMime Rendermime registry
     */
    async loadComments(container, pullRequestId, renderMime) {
        return await requestAPI(`pullrequests/files/comments?id=${encodeURIComponent(pullRequestId)}`, 'GET')
            .then(threads => {
            this.threads = threads;
            threads.forEach(thread => {
                const widget = new Discussion({
                    renderMime,
                    thread,
                    handleRemove: () => null
                });
                container.addWidget(widget);
            });
            container.addWidget(this.createNewThreadButton(container));
            return Promise.resolve();
        })
            .catch(reason => {
            container.addWidget(this.createNewThreadButton(container));
            return Promise.reject(reason);
        });
    }
    static createHeader(title, link) {
        const node = generateNode('div', {
            class: 'jp-PullRequestDescriptionHeader'
        });
        node.appendChild(generateNode('h1', {}, title));
        node.appendChild(generateNode('button', { class: 'jp-Button-flat jp-mod-styled jp-mod-accept' }, 'View Details', {
            click: () => {
                window.open(link, '_blank');
            }
        }));
        return new Widget({
            node
        });
    }
    createNewThreadButton(container) {
        const node = generateNode('div', { class: 'jp-PullRequestThread' });
        node
            .appendChild(generateNode('div', { class: 'jp-PullRequestCommentItem' }))
            .appendChild(generateNode('div', { class: 'jp-PullRequestCommentItemContent' }))
            .appendChild(generateNode('button', { class: 'jp-PullRequestReplyButton jp-PullRequestGrayedText' }, 'Start a new discussion', {
            click: () => {
                var _a;
                // Append an empty thread to start a new discussion
                const hasNewThread = ((_a = this.threads[this.threads.length - 1]) === null || _a === void 0 ? void 0 : _a.comments.length) === 0;
                if (!hasNewThread) {
                    const thread = {
                        comments: new Array(),
                        pullRequestId: this.pullRequestId
                    };
                    this.threads.push(thread);
                    const widget = new Discussion({
                        thread,
                        renderMime: this.renderMime,
                        handleRemove: () => {
                            const threadIndex = this.threads.findIndex(thread_ => thread.id === thread_.id);
                            this.threads.splice(threadIndex, 1);
                            widget.parent = null;
                            widget.dispose();
                        }
                    });
                    container.insertWidget(container.widgets.length - 1, widget);
                }
            }
        }));
        return new Widget({ node });
    }
}
//# sourceMappingURL=DescriptionWidget.js.map