"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.compactFilterPreset = exports.nonExtraneousFilterPreset = exports.focusFilter = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const core_1 = require("../core");
const filters_1 = require("./filters");
/**
 * Focus the graph on a specific node.
 * @throws Error if store is not filterable
 * @destructive
 */
function focusFilter(store, plan) {
    filters_1.verifyFilterable(store);
    if (plan.focus == null)
        return; // noop
    let focusedNode;
    let hoist = true;
    if (typeof plan.focus === "function") {
        focusedNode = plan.focus(store);
    }
    else if (plan.focus instanceof core_1.Graph.Node) {
        focusedNode = plan.focus;
    }
    else {
        const { node: _node, noHoist: _noHoist } = plan.focus;
        if (typeof _node === "function") {
            focusedNode = _node(store);
        }
        else {
            focusedNode = _node;
        }
        hoist = !_noHoist;
    }
    if (focusedNode === store.root)
        return; // noop
    const ancestors = focusedNode.scopes.slice();
    // remove the actual store.root from ancestors (we can't destroy that)
    const rootAncestor = ancestors.shift();
    if (rootAncestor !== store.root) {
        throw new Error(`${focusedNode} is not within the store root graph: it has root of ${rootAncestor}`);
    }
    if (hoist) {
        // Move focused node as direct child of store root and prune all other ancestors
        focusedNode.mutateHoist(store.root);
        // Only need to destroy to first non-root ancestor to prune the ancestral tree
        if (ancestors.length) {
            ancestors[0].mutateDestroy();
        }
        // prune all other root children (unless preserved)
        store.root.children.forEach((child) => {
            if (child !== focusedNode) {
                child.mutateDestroy();
            }
        });
    }
    else {
        // keep the focused node in place, but prune non-direct ancestor chain nodes
        // the direct ancestor chain is only the nodes scopes
        ancestors.reverse().forEach((ancestor) => {
            ancestor.siblings.forEach((ancestorSibling) => {
                ancestorSibling.mutateDestroy();
            });
        });
        // prune all planRoot siblings
        focusedNode.siblings.forEach((sibling) => {
            sibling.mutateDestroy();
        });
    }
}
exports.focusFilter = focusFilter;
/**
 * Preset that performs {@link Filters.pruneExtraneous} filter on store.
 * @throws Error if store is not filterable
 * @destructive
 */
function nonExtraneousFilterPreset(store) {
    return filters_1.Filters.pruneExtraneous()(store);
}
exports.nonExtraneousFilterPreset = nonExtraneousFilterPreset;
/**
 * Preset that performs {@link Filters.compact} filter on store.
 * @throws Error if store is not filterable
 * @destructive
 */
function compactFilterPreset(store) {
    filters_1.Filters.compact()(store);
}
exports.compactFilterPreset = compactFilterPreset;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJlc2V0cy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9maWx0ZXJpbmcvcHJlc2V0cy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQTtzQ0FDc0M7QUFDdEMsa0NBQWdDO0FBQ2hDLHVDQUFzRDtBQUd0RDs7OztHQUlHO0FBQ0gsU0FBZ0IsV0FBVyxDQUFDLEtBQWtCLEVBQUUsSUFBc0I7SUFDcEUsMEJBQWdCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFeEIsSUFBSSxJQUFJLENBQUMsS0FBSyxJQUFJLElBQUk7UUFBRSxPQUFPLENBQUMsT0FBTztJQUV2QyxJQUFJLFdBQXVCLENBQUM7SUFDNUIsSUFBSSxLQUFLLEdBQVksSUFBSSxDQUFDO0lBQzFCLElBQUksT0FBTyxJQUFJLENBQUMsS0FBSyxLQUFLLFVBQVUsRUFBRTtRQUNwQyxXQUFXLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztLQUNqQztTQUFNLElBQUksSUFBSSxDQUFDLEtBQUssWUFBWSxZQUFLLENBQUMsSUFBSSxFQUFFO1FBQzNDLFdBQVcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDO0tBQzFCO1NBQU07UUFDTCxNQUFNLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFLEdBQ3RDLElBQUksQ0FBQyxLQUFvQyxDQUFDO1FBQzVDLElBQUksT0FBTyxLQUFLLEtBQUssVUFBVSxFQUFFO1lBQy9CLFdBQVcsR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDNUI7YUFBTTtZQUNMLFdBQVcsR0FBRyxLQUFLLENBQUM7U0FDckI7UUFDRCxLQUFLLEdBQUcsQ0FBQyxRQUFRLENBQUM7S0FDbkI7SUFFRCxJQUFJLFdBQVcsS0FBSyxLQUFLLENBQUMsSUFBSTtRQUFFLE9BQU8sQ0FBQyxPQUFPO0lBRS9DLE1BQU0sU0FBUyxHQUFHLFdBQVcsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDN0Msc0VBQXNFO0lBQ3RFLE1BQU0sWUFBWSxHQUFHLFNBQVMsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUV2QyxJQUFJLFlBQVksS0FBSyxLQUFLLENBQUMsSUFBSSxFQUFFO1FBQy9CLE1BQU0sSUFBSSxLQUFLLENBQ2IsR0FBRyxXQUFXLHVEQUF1RCxZQUFZLEVBQUUsQ0FDcEYsQ0FBQztLQUNIO0lBRUQsSUFBSSxLQUFLLEVBQUU7UUFDVCxnRkFBZ0Y7UUFDaEYsV0FBVyxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFcEMsOEVBQThFO1FBQzlFLElBQUksU0FBUyxDQUFDLE1BQU0sRUFBRTtZQUNwQixTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYSxFQUFFLENBQUM7U0FDOUI7UUFDRCxtREFBbUQ7UUFDbkQsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUU7WUFDcEMsSUFBSSxLQUFLLEtBQUssV0FBVyxFQUFFO2dCQUN6QixLQUFLLENBQUMsYUFBYSxFQUFFLENBQUM7YUFDdkI7UUFDSCxDQUFDLENBQUMsQ0FBQztLQUNKO1NBQU07UUFDTCw0RUFBNEU7UUFDNUUscURBQXFEO1FBQ3JELFNBQVMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxRQUFRLEVBQUUsRUFBRTtZQUN2QyxRQUFRLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLGVBQWUsRUFBRSxFQUFFO2dCQUM1QyxlQUFlLENBQUMsYUFBYSxFQUFFLENBQUM7WUFDbEMsQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILDhCQUE4QjtRQUM5QixXQUFXLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLE9BQU8sRUFBRSxFQUFFO1lBQ3ZDLE9BQU8sQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUMxQixDQUFDLENBQUMsQ0FBQztLQUNKO0FBQ0gsQ0FBQztBQTlERCxrQ0E4REM7QUFFRDs7OztHQUlHO0FBQ0gsU0FBZ0IseUJBQXlCLENBQUMsS0FBa0I7SUFDMUQsT0FBTyxpQkFBTyxDQUFDLGVBQWUsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQzFDLENBQUM7QUFGRCw4REFFQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFnQixtQkFBbUIsQ0FBQyxLQUFrQjtJQUNwRCxpQkFBTyxDQUFDLE9BQU8sRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQzNCLENBQUM7QUFGRCxrREFFQyIsInNvdXJjZXNDb250ZW50IjpbIi8qISBDb3B5cmlnaHQgW0FtYXpvbi5jb21dKGh0dHA6Ly9hbWF6b24uY29tLyksIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMCAqL1xuaW1wb3J0IHsgR3JhcGggfSBmcm9tIFwiLi4vY29yZVwiO1xuaW1wb3J0IHsgRmlsdGVycywgdmVyaWZ5RmlsdGVyYWJsZSB9IGZyb20gXCIuL2ZpbHRlcnNcIjtcbmltcG9ydCB7IElHcmFwaEZpbHRlclBsYW4sIElHcmFwaEZpbHRlclBsYW5Gb2N1c0NvbmZpZyB9IGZyb20gXCIuL3R5cGVzXCI7XG5cbi8qKlxuICogRm9jdXMgdGhlIGdyYXBoIG9uIGEgc3BlY2lmaWMgbm9kZS5cbiAqIEB0aHJvd3MgRXJyb3IgaWYgc3RvcmUgaXMgbm90IGZpbHRlcmFibGVcbiAqIEBkZXN0cnVjdGl2ZVxuICovXG5leHBvcnQgZnVuY3Rpb24gZm9jdXNGaWx0ZXIoc3RvcmU6IEdyYXBoLlN0b3JlLCBwbGFuOiBJR3JhcGhGaWx0ZXJQbGFuKTogdm9pZCB7XG4gIHZlcmlmeUZpbHRlcmFibGUoc3RvcmUpO1xuXG4gIGlmIChwbGFuLmZvY3VzID09IG51bGwpIHJldHVybjsgLy8gbm9vcFxuXG4gIGxldCBmb2N1c2VkTm9kZTogR3JhcGguTm9kZTtcbiAgbGV0IGhvaXN0OiBib29sZWFuID0gdHJ1ZTtcbiAgaWYgKHR5cGVvZiBwbGFuLmZvY3VzID09PSBcImZ1bmN0aW9uXCIpIHtcbiAgICBmb2N1c2VkTm9kZSA9IHBsYW4uZm9jdXMoc3RvcmUpO1xuICB9IGVsc2UgaWYgKHBsYW4uZm9jdXMgaW5zdGFuY2VvZiBHcmFwaC5Ob2RlKSB7XG4gICAgZm9jdXNlZE5vZGUgPSBwbGFuLmZvY3VzO1xuICB9IGVsc2Uge1xuICAgIGNvbnN0IHsgbm9kZTogX25vZGUsIG5vSG9pc3Q6IF9ub0hvaXN0IH0gPVxuICAgICAgcGxhbi5mb2N1cyBhcyBJR3JhcGhGaWx0ZXJQbGFuRm9jdXNDb25maWc7XG4gICAgaWYgKHR5cGVvZiBfbm9kZSA9PT0gXCJmdW5jdGlvblwiKSB7XG4gICAgICBmb2N1c2VkTm9kZSA9IF9ub2RlKHN0b3JlKTtcbiAgICB9IGVsc2Uge1xuICAgICAgZm9jdXNlZE5vZGUgPSBfbm9kZTtcbiAgICB9XG4gICAgaG9pc3QgPSAhX25vSG9pc3Q7XG4gIH1cblxuICBpZiAoZm9jdXNlZE5vZGUgPT09IHN0b3JlLnJvb3QpIHJldHVybjsgLy8gbm9vcFxuXG4gIGNvbnN0IGFuY2VzdG9ycyA9IGZvY3VzZWROb2RlLnNjb3Blcy5zbGljZSgpO1xuICAvLyByZW1vdmUgdGhlIGFjdHVhbCBzdG9yZS5yb290IGZyb20gYW5jZXN0b3JzICh3ZSBjYW4ndCBkZXN0cm95IHRoYXQpXG4gIGNvbnN0IHJvb3RBbmNlc3RvciA9IGFuY2VzdG9ycy5zaGlmdCgpO1xuXG4gIGlmIChyb290QW5jZXN0b3IgIT09IHN0b3JlLnJvb3QpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICBgJHtmb2N1c2VkTm9kZX0gaXMgbm90IHdpdGhpbiB0aGUgc3RvcmUgcm9vdCBncmFwaDogaXQgaGFzIHJvb3Qgb2YgJHtyb290QW5jZXN0b3J9YFxuICAgICk7XG4gIH1cblxuICBpZiAoaG9pc3QpIHtcbiAgICAvLyBNb3ZlIGZvY3VzZWQgbm9kZSBhcyBkaXJlY3QgY2hpbGQgb2Ygc3RvcmUgcm9vdCBhbmQgcHJ1bmUgYWxsIG90aGVyIGFuY2VzdG9yc1xuICAgIGZvY3VzZWROb2RlLm11dGF0ZUhvaXN0KHN0b3JlLnJvb3QpO1xuXG4gICAgLy8gT25seSBuZWVkIHRvIGRlc3Ryb3kgdG8gZmlyc3Qgbm9uLXJvb3QgYW5jZXN0b3IgdG8gcHJ1bmUgdGhlIGFuY2VzdHJhbCB0cmVlXG4gICAgaWYgKGFuY2VzdG9ycy5sZW5ndGgpIHtcbiAgICAgIGFuY2VzdG9yc1swXS5tdXRhdGVEZXN0cm95KCk7XG4gICAgfVxuICAgIC8vIHBydW5lIGFsbCBvdGhlciByb290IGNoaWxkcmVuICh1bmxlc3MgcHJlc2VydmVkKVxuICAgIHN0b3JlLnJvb3QuY2hpbGRyZW4uZm9yRWFjaCgoY2hpbGQpID0+IHtcbiAgICAgIGlmIChjaGlsZCAhPT0gZm9jdXNlZE5vZGUpIHtcbiAgICAgICAgY2hpbGQubXV0YXRlRGVzdHJveSgpO1xuICAgICAgfVxuICAgIH0pO1xuICB9IGVsc2Uge1xuICAgIC8vIGtlZXAgdGhlIGZvY3VzZWQgbm9kZSBpbiBwbGFjZSwgYnV0IHBydW5lIG5vbi1kaXJlY3QgYW5jZXN0b3IgY2hhaW4gbm9kZXNcbiAgICAvLyB0aGUgZGlyZWN0IGFuY2VzdG9yIGNoYWluIGlzIG9ubHkgdGhlIG5vZGVzIHNjb3Blc1xuICAgIGFuY2VzdG9ycy5yZXZlcnNlKCkuZm9yRWFjaCgoYW5jZXN0b3IpID0+IHtcbiAgICAgIGFuY2VzdG9yLnNpYmxpbmdzLmZvckVhY2goKGFuY2VzdG9yU2libGluZykgPT4ge1xuICAgICAgICBhbmNlc3RvclNpYmxpbmcubXV0YXRlRGVzdHJveSgpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICAvLyBwcnVuZSBhbGwgcGxhblJvb3Qgc2libGluZ3NcbiAgICBmb2N1c2VkTm9kZS5zaWJsaW5ncy5mb3JFYWNoKChzaWJsaW5nKSA9PiB7XG4gICAgICBzaWJsaW5nLm11dGF0ZURlc3Ryb3koKTtcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIFByZXNldCB0aGF0IHBlcmZvcm1zIHtAbGluayBGaWx0ZXJzLnBydW5lRXh0cmFuZW91c30gZmlsdGVyIG9uIHN0b3JlLlxuICogQHRocm93cyBFcnJvciBpZiBzdG9yZSBpcyBub3QgZmlsdGVyYWJsZVxuICogQGRlc3RydWN0aXZlXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBub25FeHRyYW5lb3VzRmlsdGVyUHJlc2V0KHN0b3JlOiBHcmFwaC5TdG9yZSk6IHZvaWQge1xuICByZXR1cm4gRmlsdGVycy5wcnVuZUV4dHJhbmVvdXMoKShzdG9yZSk7XG59XG5cbi8qKlxuICogUHJlc2V0IHRoYXQgcGVyZm9ybXMge0BsaW5rIEZpbHRlcnMuY29tcGFjdH0gZmlsdGVyIG9uIHN0b3JlLlxuICogQHRocm93cyBFcnJvciBpZiBzdG9yZSBpcyBub3QgZmlsdGVyYWJsZVxuICogQGRlc3RydWN0aXZlXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBjb21wYWN0RmlsdGVyUHJlc2V0KHN0b3JlOiBHcmFwaC5TdG9yZSk6IHZvaWQge1xuICBGaWx0ZXJzLmNvbXBhY3QoKShzdG9yZSk7XG59XG4iXX0=