"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deserializeStore = exports.Graph = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const console_1 = require("console");
const constructs_1 = require("constructs");
const cloneDeep = require("lodash.clonedeep"); // eslint-disable-line @typescript-eslint/no-require-imports
const isEmpty = require("lodash.isempty"); // eslint-disable-line @typescript-eslint/no-require-imports
const omit = require("lodash.omit"); // eslint-disable-line @typescript-eslint/no-require-imports
const uniq = require("lodash.uniq"); // eslint-disable-line @typescript-eslint/no-require-imports
const counter_1 = require("./counter");
const types_1 = require("./types");
/** Public cdk-graph interface */
var Graph;
(function (Graph) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s;
    /** Store class provides the in-memory database-like interface for managing all entities in the graph */
    class Store {
        constructor(allowDestructiveMutations = false) {
            /** Current SemVer version of the store */
            this.version = "0.0.0";
            /** @internal */
            this._edges = new Map();
            /** @internal */
            this._nodes = new Map();
            /** @internal */
            this._stacks = new Map();
            /** @internal */
            this._stages = new Map();
            /** @internal */
            this._logicalIdLookup = new Map();
            /** @internal */
            this._counters = {
                cfnResources: new counter_1.Counter(),
                nodeTypes: new counter_1.Counter(),
                edgeTypes: new counter_1.Counter(),
            };
            this._root = new RootNode(this);
            this.allowDestructiveMutations = allowDestructiveMutations;
        }
        /** Builds store from serialized store data */
        static fromSerializedStore(serializedStore) {
            return deserializeStore(serializedStore);
        }
        /**
         * Root node in the store. The **root** node is not the computed root, but the graph root
         * which is auto-generated and can not be mutated.
         */
        get root() {
            return this._root;
        }
        /**
         * Gets all stored **edges**
         * @type ReadonlyArray<Edge>
         */
        get edges() {
            return Array.from(this._edges.values());
        }
        /**
         * Gets all stored **nodes**
         * @type ReadonlyArray<Node>
         */
        get nodes() {
            return Array.from(this._nodes.values());
        }
        /**
         * Gets all stored **stack** nodes
         * @type ReadonlyArray<StackNode>
         */
        get stacks() {
            return Array.from(this._stacks.values());
        }
        /**
         * Gets all stored **stage** nodes
         * @type ReadonlyArray<StageNode>
         */
        get stages() {
            return Array.from(this._stages.values());
        }
        /**
         * Gets all stored **root stack** nodes
         * @type ReadonlyArray<StackNode>
         */
        get rootStacks() {
            return this.stacks.filter((stack) => StackNode.isStackNode(stack));
        }
        /** Get record of all store counters */
        get counts() {
            return {
                nodes: this._nodes.size,
                edges: this._edges.size,
                stacks: this._stacks.size,
                stages: this._stages.size,
                nodeTypes: this._counters.nodeTypes.counts,
                edgeTypes: this._counters.edgeTypes.counts,
                cfnResources: this._counters.cfnResources.counts,
            };
        }
        /** Add **edge** to the store */
        addEdge(edge) {
            this._edges.set(edge.uuid, edge);
            this._counters.edgeTypes.add(edge.edgeType);
        }
        /** Get stored **edge** by UUID */
        getEdge(uuid) {
            const edge = this._edges.get(uuid);
            if (edge != null) {
                return edge;
            }
            throw new Error(`Edge ${uuid} is not defined`);
        }
        /** Add **node** to the store */
        addNode(node) {
            // Do not store root node
            if (RootNode.isRootNode(node) === true) {
                return;
            }
            this._nodes.set(node.uuid, node);
            this._counters.nodeTypes.add(node.nodeType);
            if (CfnResourceNode.isCfnResourceNode(node) && node.cfnType) {
                this._counters.cfnResources.add(node.cfnType);
            }
        }
        /** Get stored **node** by UUID */
        getNode(uuid) {
            // Root node is not stored in "nodes" map
            if (uuid === RootNode.UUID && this.root) {
                return this.root;
            }
            const node = this._nodes.get(uuid);
            if (node != null) {
                return node;
            }
            throw new Error(`Node ${uuid} is not defined`);
        }
        /** Add **stack** node to the store */
        addStack(stack) {
            this._stacks.set(stack.uuid, stack);
        }
        /** Get stored **stack** node by UUID */
        getStack(uuid) {
            const stack = this._stacks.get(uuid);
            if (stack != null) {
                return stack;
            }
            throw new Error(`Stack ${uuid} is not defined`);
        }
        /** Add **stage** to the store */
        addStage(stage) {
            this._stages.set(stage.uuid, stage);
        }
        /** Get stored **stage** node by UUID */
        getStage(uuid) {
            const stage = this._stages.get(uuid);
            if (stage != null) {
                return stage;
            }
            throw new Error(`Stage ${uuid} is not defined`);
        }
        /**
         * Compute **universal** *logicalId* based on parent stack and construct *logicalId* (`<stack>:<logicalId>`).
         *
         * Construct *logicalIds are only unique within their containing stack, so to use *logicalId*
         * lookups universally (like resolving references) we need a universal key.
         */
        computeLogicalUniversalId(stack, logicalId) {
            return `${stack.uuid}:${logicalId}`;
        }
        /** Find node by **universal** *logicalId* (`<stack>:<logicalId>`) */
        findNodeByLogicalUniversalId(uid) {
            const [stackUUID, logicalId] = uid.split(":");
            const stack = this.getStack(stackUUID);
            return this.findNodeByLogicalId(stack, logicalId);
        }
        /** Find node within given **stack** with given *logicalId* */
        findNodeByLogicalId(stack, logicalId) {
            const uid = this.computeLogicalUniversalId(stack, logicalId);
            const nodeUUID = this._logicalIdLookup.get(uid);
            if (nodeUUID == null) {
                if (stack instanceof NestedStackNode && stack.parentStack) {
                    return this.findNodeByLogicalId(stack.parentStack, logicalId);
                }
                throw new Error(`Failed to find node by logicalId: ${uid}`);
            }
            const node = this._nodes.get(nodeUUID);
            if (node != null) {
                return node;
            }
            throw new Error(`Unable to find node mapped to logical id ${logicalId}`);
        }
        /** Record a **universal** *logicalId* to node mapping in the store */
        recordLogicalId(stack, logicalId, resource) {
            const uid = this.computeLogicalUniversalId(stack, logicalId);
            this._logicalIdLookup.set(uid, resource.uuid);
        }
        /** Serialize the store */
        serialize() {
            return {
                version: this.version,
                tree: this.root._serialize(),
                edges: Array.from(this.edges).map((edge) => edge._serialize()),
            };
        }
        /**
         * Clone the store to allow destructive mutations.
         * @param allowDestructiveMutations Indicates if destructive mutations are allowed; defaults to `true`
         * @returns {Store} Returns a clone of the store that allows destructive mutations
         */
        clone(allowDestructiveMutations = true) {
            return deserializeStore(this.serialize(), allowDestructiveMutations);
        }
        /**
         * Verifies that the store allows destructive mutations.
         * @throws Error is store does **not** allow mutations
         */
        verifyDestructiveMutationAllowed() {
            if (!this.allowDestructiveMutations) {
                throw new Error("GraphStore must be a clone to perform destructive mutations");
            }
        }
        /**
         * Remove **edge** from the store
         * @destructive
         */
        mutateRemoveEdge(edge) {
            const deleted = this._edges.delete(edge.uuid);
            if (deleted) {
                this._counters.edgeTypes.subtract(edge.edgeType);
            }
            return deleted;
        }
        /**
         * Remove **node** from the store
         * @destructive
         */
        mutateRemoveNode(node) {
            // Root node can not be removed
            if (RootNode.isRootNode(node) === true) {
                throw new Error("Root not can not be removed");
            }
            if (node.logicalId && node.stack) {
                this._logicalIdLookup.delete(this.computeLogicalUniversalId(node.stack, node.logicalId));
            }
            if (StackNode.isStackNode(node)) {
                this._stacks.delete(node.uuid);
            }
            const deleted = this._nodes.delete(node.uuid);
            if (deleted) {
                this._counters.nodeTypes.subtract(node.nodeType);
                if (CfnResourceNode.isCfnResourceNode(node) && node.cfnType) {
                    this._counters.cfnResources.subtract(node.cfnType);
                }
            }
            return deleted;
        }
    }
    _a = JSII_RTTI_SYMBOL_1;
    Store[_a] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.Store", version: "0.12.25" };
    Graph.Store = Store;
    /** Base class for all store entities (Node and Edges) */
    class BaseEntity {
        constructor(props) {
            /** @internal */
            this._destroyed = false;
            this.store = props.store;
            this.uuid = props.uuid;
            this._attributes = props.attributes || {};
            this._metadata = props.metadata || [];
            this._tags = new Map(Object.entries(props.tags || {}));
            this._flags = new Set(props.flags);
        }
        /**
         * Get *readonly* record of all attributes
         * @type Readonly<SerializedGraph.Attributes>
         */
        get attributes() {
            return cloneDeep(this._attributes);
        }
        /**
         * Get *readonly* list of all metadata entries
         * @type Readonly<SerializedGraph.Metadata>
         */
        get metadata() {
            return cloneDeep(this._metadata);
        }
        /**
         * Get *readonly* record of all tags
         * @type Readonly<SerializedGraph.Tags>
         */
        get tags() {
            return Object.fromEntries(this._tags);
        }
        /**
         * Get *readonly* list of all flags
         * @type ReadonlyArray<FlagEnum>
         */
        get flags() {
            return Array.from(this._flags);
        }
        /** Indicates if the entity has been destroyed (eg: removed from store) */
        get isDestroyed() {
            return this._destroyed;
        }
        /** Indicates if the entity has had destructive mutations applied */
        get isMutated() {
            return this.hasFlag(types_1.FlagEnum.MUTATED);
        }
        /** Indicates if entity has a given attribute defined, and optionally with a specific value */
        hasAttribute(key, value) {
            if (key in this._attributes) {
                if (value !== undefined) {
                    return this._attributes[key] === value;
                }
                return true;
            }
            return false;
        }
        /**
         * Add attribute.
         *
         * @throws Error if attribute for key already exists
         */
        addAttribute(key, value) {
            if (this.hasAttribute(key)) {
                throw new Error(`Entity ${String(this)} already has attribute ${key}; use setAttribute to override`);
            }
            this.setAttribute(key, value);
        }
        /** Set attribute. This will overwrite existing attribute. */
        setAttribute(key, value) {
            // @ts-ignore
            this._attributes[key] = value;
        }
        /** Get attribute by key */
        getAttribute(key) {
            return this._attributes[key];
        }
        /** Add metadata entry */
        addMetadata(metadataType, data) {
            this._metadata.push({
                type: metadataType,
                data,
            });
        }
        /** Indicates if entity has matching metadata entry */
        hasMetadata(metadataType, data) {
            return !!this._metadata.find((metadata) => {
                if (metadata.type !== metadataType)
                    return false;
                if (metadata.data !== data)
                    return false;
                return true;
            });
        }
        /**
         * Retrieves all metadata entries of a given type
         * @type Readonly<SerializedGraph.Metadata>
         */
        findMetadata(metadataType) {
            return this._metadata.filter((entry) => entry.type === metadataType);
        }
        /**
         * Add tag.
         * @throws Throws Error is tag for key already exists
         */
        addTag(key, value) {
            if (this.hasTag(key)) {
                throw new Error(`Entity ${String(this)} already has tag ${key}; use setTag to override`);
            }
            this.setTag(key, value);
        }
        /** Set tag. Will overwrite existing tag. */
        setTag(key, value) {
            this._tags.set(key, value);
        }
        /** Indicates if entity has tag, optionally verifying tag value */
        hasTag(key, value) {
            if (!this._tags.has(key))
                return false;
            if (value !== undefined && this._tags.get(key) !== value)
                return false;
            return true;
        }
        /** Get tag by key */
        getTag(key) {
            return this._tags.get(key);
        }
        /** Add flag */
        addFlag(flag) {
            this._flags.add(flag);
        }
        /** Indicates if entity has a given flag */
        hasFlag(flag) {
            return this._flags.has(flag);
        }
        /**
         * Applies data (attributes, metadata, tags, flag) to entity.
         *
         * Generally used only for mutations such as collapse and consume to retain data.
         * @param data - The data to apply
         * @param {boolean} [applyFlags=false] - Indicates if data is overwritten
         * @param {boolean} [applyFlags=false] - Indicates if flags should be applied
         */
        applyData(data, overwrite = false, applyFlags = false) {
            if (data.attributes) {
                Object.entries(data.attributes).forEach(([key, value]) => {
                    if (overwrite || !this.hasAttribute(key)) {
                        this.setAttribute(key, value);
                    }
                });
            }
            if (data.metadata) {
                data.metadata.forEach((v) => {
                    if (!this.hasMetadata(v.type, v.data)) {
                        this.addMetadata(v.type, v.data);
                    }
                });
            }
            if (data.tags) {
                Object.entries(data.tags).forEach(([key, value]) => {
                    if (overwrite || !this.hasTag(key)) {
                        this.setTag(key, value);
                    }
                });
            }
            if (applyFlags && data.flags) {
                data.flags.forEach((flag) => {
                    this.addFlag(flag);
                });
            }
        }
        /**
         * Performs pre-mutate operations on entity and store
         * @internal
         */
        _preMutate() {
            this.store.verifyDestructiveMutationAllowed();
            this.addFlag(types_1.FlagEnum.MUTATED);
        }
        /**
         * Serialize entity
         * @internal
         */
        _serialize() {
            return {
                uuid: this.uuid,
                attributes: isEmpty(this._attributes) ? undefined : this._attributes,
                metadata: isEmpty(this._metadata) ? undefined : this._metadata,
                tags: this._tags.size ? Object.fromEntries(this._tags) : undefined,
                flags: this._flags.size ? Array.from(this._flags) : undefined,
            };
        }
    }
    _b = JSII_RTTI_SYMBOL_1;
    BaseEntity[_b] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.BaseEntity", version: "0.12.25" };
    Graph.BaseEntity = BaseEntity;
    /** Edge class defines a link (relationship) between nodes, as in standard [graph theory](https://en.wikipedia.org/wiki/Graph_theory) */
    class Edge extends BaseEntity {
        constructor(props) {
            super(props);
            this.edgeType = props.edgeType;
            this._direction = props.direction;
            this._source = props.source;
            this._target = props.target;
            // Do not change original closed edge flag from a mutation.
            if (this._target === this._source && this.hasFlag(types_1.FlagEnum.MUTATED)) {
                this.addFlag(types_1.FlagEnum.CLOSED_EDGE);
            }
            // wire up links
            this._source.addLink(this);
            this._target.addReverseLink(this);
            this.store.addEdge(this);
        }
        /** Find first edge matching predicate within an EdgeChain */
        static findInChain(chain, predicate) {
            for (const entry of chain) {
                if (Array.isArray(entry)) {
                    const edge = Edge.findInChain(entry, predicate);
                    if (edge)
                        return edge;
                }
                else {
                    if (predicate(entry))
                        return entry;
                }
            }
            return undefined;
        }
        /** Find all matching edges based on predicate within an EdgeChain */
        static findAllInChain(chain, predicate) {
            const edges = [];
            for (const entry of chain) {
                if (Array.isArray(entry)) {
                    const edge = Edge.findInChain(entry, predicate);
                    if (edge) {
                        edges.push(edge);
                    }
                }
                else {
                    if (predicate(entry)) {
                        edges.push(entry);
                    }
                }
            }
            return edges;
        }
        /** Edge **source** is the node that defines the edge (tail) */
        get source() {
            return this._source;
        }
        /** Edge **target** is the node being referenced by the **source** (head) */
        get target() {
            return this._target;
        }
        /** Indicates the direction in which the edge is directed */
        get direction() {
            return this._direction;
        }
        /** Indicates if **source** and **target** nodes reside in different *root* stacks */
        get isCrossStack() {
            return this._source.rootStack !== this._target.rootStack;
        }
        /**
         * Indicates if the Edge's **source** and **target** are the same, or were the same
         * when it was created (prior to mutations).
         *
         * To check whether it was originally closed, use `hasFlag(FlagEnum.CLOSED_EDGE)` instead.
         */
        get isClosed() {
            return (this._source === this._target || this.hasFlag(types_1.FlagEnum.CLOSED_EDGE));
        }
        /**
         * Indicates if edge is extraneous which is determined by explicitly having *EXTRANEOUS* flag
         * added and/or being a closed loop (source===target).
         */
        get isExtraneous() {
            return this.hasFlag(types_1.FlagEnum.EXTRANEOUS) || this.isClosed;
        }
        /**
         * Indicates if this edge is equivalent to another edge.
         *
         * Edges are considered equivalent if they share same type, source, and target.
         */
        isEquivalent(edge) {
            if (edge.edgeType !== this.edgeType)
                return false;
            if (edge.source !== this.source)
                return false;
            if (edge.target !== this.target)
                return false;
            return true;
        }
        /** Indicates if edge allows destructive mutations */
        get allowDestructiveMutations() {
            return this.store.allowDestructiveMutations;
        }
        /**
         * Change the edge **direction**
         * @destructive
         */
        mutateDirection(direction) {
            this._preMutate();
            this._direction = direction;
        }
        /**
         * Change the edge **source**
         * @destructive
         */
        mutateSource(node) {
            this._preMutate();
            this._source.mutateRemoveLink(this);
            this._source = node;
            this._source.addLink(this);
        }
        /**
         * Change the edge **target**
         * @destructive
         */
        mutateTarget(node) {
            this._preMutate();
            this._target.mutateRemoveReverseLink(this);
            this._target = node;
            this._target.addReverseLink(this);
        }
        /**
         * Destroy the edge. Remove all references and remove from store.
         * @destructive
         */
        mutateDestroy(_strict = false) {
            this._preMutate();
            this.source.mutateRemoveLink(this);
            this.target.mutateRemoveReverseLink(this);
            this.store.mutateRemoveEdge(this);
            this._destroyed = true;
        }
        /**
         * Merge an equivalent edge's data into this edge and destroy the other edge.
         *
         * Used during filtering operations to consolidate equivalent edges.
         * @param edge - The edge to consume
         * @throws Error is edge is not *equivalent*
         * @destructive
         */
        mutateConsume(edge) {
            this._preMutate();
            if (!this.isEquivalent(edge)) {
                throw new Error(`Only equivalent edges can be consumed: ${edge} > ${this}`);
            }
            // propagate edge data
            this.applyData(edge);
            // destroy the consumed edge
            edge.mutateDestroy();
        }
        /** Get string representation of this edge */
        toString() {
            return `Edge:${this.edgeType}::${this.uuid}::${this.direction}(${this.source}->${this.target})`;
        }
        /** @internal */
        _serialize() {
            return {
                ...super._serialize(),
                edgeType: this.edgeType,
                direction: this.direction,
                source: this.source.uuid,
                target: this.target.uuid,
            };
        }
    }
    _c = JSII_RTTI_SYMBOL_1;
    Edge[_c] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.Edge", version: "0.12.25" };
    Graph.Edge = Edge;
    /** Dependency edge class defines CloudFormation dependency between resources */
    class Dependency extends Edge {
        constructor(props) {
            super({
                ...props,
                edgeType: types_1.EdgeTypeEnum.DEPENDENCY,
                direction: types_1.EdgeDirectionEnum.FORWARD,
            });
            this.addFlag(types_1.FlagEnum.EXTRANEOUS);
        }
        /** Indicates if given edge is a {@link Dependency} edge */
        static isDependency(edge) {
            return edge.edgeType === types_1.EdgeTypeEnum.DEPENDENCY;
        }
    }
    _d = JSII_RTTI_SYMBOL_1;
    Dependency[_d] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.Dependency", version: "0.12.25" };
    /** Edge prefix to denote dependency edge  */
    Dependency.PREFIX = "DEP:";
    Graph.Dependency = Dependency;
    /** Reference edge class defines a directed relationship between nodes  */
    class Reference extends Edge {
        constructor(props) {
            super({
                edgeType: types_1.EdgeTypeEnum.REFERENCE,
                direction: types_1.EdgeDirectionEnum.FORWARD,
                ...props,
            });
            this.setAttribute(Reference.ATT_TYPE, props.referenceType || types_1.ReferenceTypeEnum.REF);
        }
        /** Indicates if edge is a {@link Reference} */
        static isReference(edge) {
            return edge.edgeType === types_1.EdgeTypeEnum.REFERENCE;
        }
        /** Indicates if edge is a **Ref** based {@link Reference} edge */
        static isRef(edge) {
            return edge.referenceType === types_1.ReferenceTypeEnum.REF;
        }
        /** Get type of reference */
        get referenceType() {
            return this.getAttribute(Reference.ATT_TYPE);
        }
        /** Resolve reference chain */
        resolveChain() {
            if (OutputNode.isOutputNode(this.target)) {
                function _resolveChain(_ref) {
                    if (OutputNode.isOutputNode(_ref.target)) {
                        return [
                            _ref,
                            ..._ref.target.referenceLinks.map(_resolveChain),
                        ];
                    }
                    return [_ref];
                }
                return [
                    this,
                    ...this.target.referenceLinks.map(_resolveChain),
                ];
            }
            return [this];
        }
        /**
         * Resolve targets by following potential edge chain.
         *
         * @see {@link EdgeChain}
         */
        resolveTargets() {
            if (OutputNode.isOutputNode(this.target)) {
                function resolveOutputTarget(_target) {
                    if (OutputNode.isOutputNode(_target))
                        return resolveOutputTarget(_target);
                    return [_target];
                }
                return this.target.referenceLinks.flatMap((ref) => resolveOutputTarget(ref.target));
            }
            return [this.target];
        }
    }
    _e = JSII_RTTI_SYMBOL_1;
    Reference[_e] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.Reference", version: "0.12.25" };
    /** Edge prefix to denote **Ref** type reference edge  */
    Reference.PREFIX = "REF:";
    /** Attribute defining the type of reference */
    Reference.ATT_TYPE = "graph:reference:type";
    Graph.Reference = Reference;
    /** Attribute type reference edge */
    class AttributeReference extends Reference {
        constructor(props) {
            super({
                ...props,
                referenceType: types_1.ReferenceTypeEnum.ATTRIBUTE,
            });
            this.setAttribute(AttributeReference.ATT_VALUE, props.value);
        }
        /** Indicates if edge in an **Fn::GetAtt** {@link Reference} */
        static isAtt(edge) {
            return edge.referenceType === types_1.ReferenceTypeEnum.ATTRIBUTE;
        }
        /** Get the resolved attribute value */
        get value() {
            return this.getAttribute(AttributeReference.ATT_VALUE);
        }
    }
    _f = JSII_RTTI_SYMBOL_1;
    AttributeReference[_f] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.AttributeReference", version: "0.12.25" };
    /** Edge prefix to denote **Fn::GetAtt** type reference edge  */
    AttributeReference.PREFIX = "ATT:";
    /** Attribute key for resolved value of attribute reference */
    AttributeReference.ATT_VALUE = "graph:reference:attribute:value";
    Graph.AttributeReference = AttributeReference;
    /** Import reference defines **Fn::ImportValue** type reference edge. */
    class ImportReference extends Reference {
        constructor(props) {
            super({
                ...props,
                referenceType: types_1.ReferenceTypeEnum.IMPORT,
            });
        }
        /** Indicates if edge is **Fn::ImportValue** based {@link Reference} */
        static isImport(edge) {
            return edge.referenceType === types_1.ReferenceTypeEnum.IMPORT;
        }
    }
    _g = JSII_RTTI_SYMBOL_1;
    ImportReference[_g] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.ImportReference", version: "0.12.25" };
    /** Edge prefix to denote **Fn::ImportValue** type reference edge */
    ImportReference.PREFIX = "IMP:";
    Graph.ImportReference = ImportReference;
    /** Node class is the base definition of **node** entities in the graph, as in standard [graph theory](https://en.wikipedia.org/wiki/Graph_theory) */
    class Node extends BaseEntity {
        constructor(props) {
            super(props);
            /** @internal */
            this._children = new Map();
            /** @internal */
            this._links = new Map();
            /** @internal */
            this._reverseLinks = new Map();
            this.nodeType = props.nodeType;
            this.id = props.id;
            this.path = props.path;
            this.constructInfo = props.constructInfo;
            this._cfnType = props.cfnType;
            this._parent = props.parent;
            this.depth = this.parent ? this.parent.depth + 1 : 0;
            this._stack =
                props.stack || (this instanceof StackNode ? this : undefined);
            this.logicalId = props.logicalId;
            if (this.logicalId) {
                if (this.stack == null) {
                    throw new Error(`LogicalId defined outside of stack: ${this.logicalId} - ${String(this)}`);
                }
                this.store.recordLogicalId(this.stack, this.logicalId, this);
            }
            if (this.parent) {
                this.parent.addChild(this);
            }
            this.store.addNode(this);
        }
        /** Stack the node is contained in */
        get stack() {
            return this._stack;
        }
        /** Parent node. Only the root node should not have parent. */
        get parent() {
            return this._parent;
        }
        /** Gets descending ordered list of ancestors from the root */
        get scopes() {
            if (this.parent) {
                return [...this.parent.scopes, this.parent];
            }
            return [];
        }
        /** Indicates if node is direct child of the graph root node */
        get isTopLevel() {
            return this.parent === this.store.root;
        }
        /** Get **root** stack */
        get rootStack() {
            if (StackNode.isStackNode(this))
                return this;
            return this.scopes.find((scope) => StackNode.isStackNode(scope));
        }
        /** Get all direct child nodes */
        get children() {
            return Array.from(this._children.values());
        }
        /** Indicates if this node is a *leaf* node, which means it does not have children */
        get isLeaf() {
            return this._children.size === 0;
        }
        /** Gets all links (edges) in which this node is the **source** */
        get links() {
            return Array.from(this._links.values());
        }
        /** Gets all links (edges) in which this node is the **target** */
        get reverseLinks() {
            return Array.from(this._reverseLinks.values());
        }
        /** Synthesized construct information defining jii resolution data */
        get constructInfoFqn() {
            return this.constructInfo?.fqn;
        }
        /** Indicates if node is a *Custom Resource* */
        get isCustomResource() {
            return types_1.ConstructInfoFqnEnum.CUSTOM_RESOURCE === this.constructInfoFqn;
        }
        /** Gets CloudFormation properties for this node */
        get cfnProps() {
            return this.attributes[types_1.CfnAttributesEnum.PROPS];
        }
        /** Get the CloudFormation resource type for this node */
        get cfnType() {
            return this._cfnType;
        }
        /** Gets list of {@link Dependency} links (edges) where this node is the **source** */
        get dependencyLinks() {
            return Array.from(this._links.values()).filter((link) => {
                return link.edgeType === types_1.EdgeTypeEnum.DEPENDENCY;
            });
        }
        /** Gets list of {@link Dependency} links (edges) where this node is the **target** */
        get reverseDependencyLinks() {
            return Array.from(this._links.values()).filter((link) => {
                return link.edgeType === types_1.EdgeTypeEnum.DEPENDENCY;
            });
        }
        /** Gets list of {@link Reference} links (edges) where this node is the **source** */
        get referenceLinks() {
            return Array.from(this._links.values()).filter((link) => {
                return link.edgeType === types_1.EdgeTypeEnum.REFERENCE;
            });
        }
        /** Gets list of {@link Reference} links (edges) where this node is the **target** */
        get reverseReferenceLinks() {
            return Array.from(this._links.values()).filter((link) => {
                return link.edgeType === types_1.EdgeTypeEnum.REFERENCE;
            });
        }
        /**
         * Get list of **Nodes** that *this node references*
         * @see {@link Node.referenceLinks}
         */
        get references() {
            return uniq(this.referenceLinks.flatMap((link) => link.resolveTargets()));
        }
        /**
         * Get list of **Nodes** that *reference this node*
         * @see {@link Node.reverseReferenceLinks}
         */
        get referencedBy() {
            return uniq(this.reverseReferenceLinks.flatMap((link) => link.resolveTargets()));
        }
        /**
         * Get list of **Nodes** that *this node depends on*
         * @see {@link Node.dependencyLinks}
         */
        get dependencies() {
            return uniq(this.dependencyLinks.flatMap((link) => link.target));
        }
        /**
         * Get list of **Nodes** that *depend on this node*
         * @see {@link Node.reverseDependencyLinks}
         */
        get dependedOnBy() {
            return uniq(this.reverseDependencyLinks.flatMap((link) => link.target));
        }
        /** Indicates if this node is considered a {@link FlagEnum.GRAPH_CONTAINER} */
        get isGraphContainer() {
            return this.hasFlag(types_1.FlagEnum.GRAPH_CONTAINER);
        }
        /** Indicates if this node is considered a {@link FlagEnum.CLUSTER} */
        get isCluster() {
            return this.hasFlag(types_1.FlagEnum.CLUSTER);
        }
        /**
         * Indicates if this node is considered a {@link FlagEnum.EXTRANEOUS} node
         * or determined to be extraneous:
         * - Clusters that contain no children
         */
        get isExtraneous() {
            return (this.hasFlag(types_1.FlagEnum.EXTRANEOUS) || (this.isCluster && this.isLeaf));
        }
        /** Indicates if this node is considered a {@link FlagEnum.RESOURCE_WRAPPER} */
        get isResourceWrapper() {
            return this.hasFlag(types_1.FlagEnum.RESOURCE_WRAPPER);
        }
        /** Indicates if this node is considered a {@link FlagEnum.ASSET} */
        get isAsset() {
            return this.hasFlag(types_1.FlagEnum.ASSET);
        }
        /** Get list of *siblings* of this node. */
        get siblings() {
            if (this.parent) {
                return this.parent.children.filter((child) => child !== this);
            }
            return [];
        }
        /** Get specific CloudFormation property */
        getCfnProp(key) {
            return this.cfnProps && this.cfnProps[key];
        }
        /** Add *link* to another node */
        addLink(edge) {
            this._links.set(edge.uuid, edge);
        }
        /** Add *link* from another node */
        addReverseLink(edge) {
            this._reverseLinks.set(edge.uuid, edge);
        }
        /** Add *child* node */
        addChild(node) {
            this._children.set(node.id, node);
        }
        /** Indicates if specific *node* is a *child* of *this node* */
        isChild(node) {
            for (const child of this._children.values()) {
                if (child === node)
                    return true;
            }
            return false;
        }
        /** Indicates if a specific *node* is an *ancestor* of *this node* */
        isAncestor(ancestor) {
            return this.scopes.includes(ancestor);
        }
        /**
         * Find nearest *ancestor* of *this node* matching given predicate.
         * @param predicate - Predicate to match ancestor
         * @max {number} [max] - Optional maximum levels to ascend
         */
        findAncestor(predicate, max) {
            let ancestors = this.scopes.slice().reverse();
            if (max) {
                ancestors = ancestors.slice(0, max);
            }
            return ancestors.find(predicate);
        }
        /**
         * Gets the nearest **common** *ancestor* shared between *this node* and another *node*.
         * @throws Error if *node* does not share a **common** *ancestor*
         */
        getNearestAncestor(node) {
            if (node === this)
                throw new Error("Node is the current node");
            const aScopes = this.scopes;
            const bScopes = node.scopes;
            for (const aScope of aScopes) {
                for (const bScope of bScopes) {
                    if (aScope === bScope)
                        return aScope;
                }
            }
            throw new Error(`Nodes do not share common ancestor: ${String(this)} ^ ${String(node)}`);
        }
        /**
         * Return this construct and all of its sub-nodes in the given order.
         *
         * Optionally filter nodes based on predicate.
         */
        findAll(options) {
            const { predicate, order = constructs_1.ConstructOrder.PREORDER } = options || {};
            const all = new Array();
            function visit(c) {
                if (order === constructs_1.ConstructOrder.PREORDER && !RootNode.isRootNode(c)) {
                    all.push(c);
                }
                for (const child of c.children) {
                    visit(child);
                }
                if (order === constructs_1.ConstructOrder.POSTORDER && !RootNode.isRootNode(c)) {
                    all.push(c);
                }
            }
            visit(this);
            if (predicate) {
                return all.filter(predicate);
            }
            return all;
        }
        /** Recursively find the nearest sub-node matching predicate */
        find(predicate) {
            if (predicate(this))
                return this;
            for (const child of this.children) {
                const node = child.find(predicate);
                if (node != null)
                    return node;
            }
            return undefined;
        }
        /**
         * Get *child* node with given *id*.
         *
         * @throws Error if no child with given id
         */
        getChild(id) {
            const child = this._children.get(id);
            if (child == null) {
                throw new Error(`${String(this)} does not have child with id "${id}"`);
            }
            return child;
        }
        /** Find child with given *id*. Similar to `find` but does not throw error if no child found. */
        findChild(id) {
            return this._children.get(id);
        }
        /**
         * Return all direct links of this node and that of all sub-nodes.
         *
         * Optionally filter links based on predicate.
         */
        findAllLinks(options) {
            const { predicate, order = constructs_1.ConstructOrder.PREORDER, reverse, } = options || {};
            const all = new Array();
            visit(this);
            if (predicate) {
                return all.filter(predicate);
            }
            return all;
            function visit(c) {
                if (order === constructs_1.ConstructOrder.PREORDER) {
                    all.push(...c[reverse ? "reverseLinks" : "links"]);
                }
                for (const child of c.children) {
                    visit(child);
                }
                if (order === constructs_1.ConstructOrder.POSTORDER) {
                    all.push(...c[reverse ? "reverseLinks" : "links"]);
                }
            }
        }
        /**
         * Resolve all link chains
         * @see {@link EdgeChain}
         */
        getLinkChains(reverse = false) {
            let links = this[reverse ? "reverseLinks" : "links"];
            return links.map((link) => {
                if (Reference.isReference(link)) {
                    return link.resolveChain();
                }
                return [link];
            });
        }
        /**
         * Find link of this node based on predicate. By default this will follow link
         * chains to evaluate the predicate against and return the matching direct link
         * of this node.
         *
         * @param predicate Edge predicate function to match edge
         * @param reverse Indicates if links are search in reverse order
         * @param follow Indicates if link chain is followed
         * @param direct Indicates that only *direct* links should be searched
         * @returns
         */
        findLink(predicate, reverse = false, follow = true, direct = true) {
            if (follow) {
                const chains = this.getLinkChains(reverse);
                for (const chain of chains) {
                    const edge = Edge.findInChain(chain, predicate);
                    if (edge) {
                        if (direct)
                            return chain[0];
                        return edge;
                    }
                }
                return undefined;
            }
            return this[reverse ? "reverseLinks" : "links"].find(predicate);
        }
        /**
         * Find all links of this node based on predicate. By default this will follow link
         * chains to evaluate the predicate against and return the matching direct links
         * of this node.
         *
         * @param predicate Edge predicate function to match edge
         * @param reverse Indicates if links are search in reverse order
         * @param follow Indicates if link chain is followed
         * @param direct Indicates that only *direct* links should be searched
         * @returns
         */
        findLinks(predicate, reverse = false, follow = true, direct = true) {
            if (follow) {
                return this.getLinkChains(reverse).flatMap((chain) => {
                    const edges = Edge.findAllInChain(chain, predicate);
                    if (direct) {
                        return edges.length ? [chain[0]] : [];
                    }
                    return edges;
                });
            }
            return this[reverse ? "reverseLinks" : "links"].filter(predicate);
        }
        /** Indicates if *this node* references *another node* */
        doesReference(node) {
            return this.references.includes(node);
        }
        /** Indicates if *this node* depends on *another node* */
        doesDependOn(node) {
            return this.dependencies.includes(node);
        }
        /**
         * Indicates if this node allows destructive mutations
         * @see {@link Store.allowDestructiveMutations}
         */
        get allowDestructiveMutations() {
            return this.store.allowDestructiveMutations;
        }
        /**
         * Collapses all sub-nodes of *this node* into *this node*.
         * @destructive
         */
        mutateCollapse() {
            this._preMutate();
            this.children.forEach((child) => child.mutateCollapseToParent());
            this._mutateReconcileLinks();
        }
        /**
         * Collapses *this node* into *it's parent node*
         * @destructive
         */
        mutateCollapseToParent() {
            this._preMutate();
            if (this.parent == null) {
                throw new Error(`${this} does not have parent to collapse to.`);
            }
            return this.mutateCollapseTo(this.parent);
        }
        /**
         * Collapses *this node* into *an ancestor*
         * @destructive
         */
        mutateCollapseTo(ancestor) {
            this._preMutate();
            if (!this.isAncestor(ancestor)) {
                throw new Error(`${ancestor} is not an ancestor of ${this}`);
            }
            // TODO: should we retain the child attributes somewhere?
            this.children.forEach((child) => {
                if (child.isDestroyed)
                    return;
                child.mutateCollapseToParent();
            });
            // redirect all links to parent
            // while also deleting links to parent
            this.links.forEach((link) => {
                if (link.isDestroyed)
                    return;
                if (link.target === ancestor) {
                    link.mutateDestroy();
                }
                else {
                    link.mutateSource(ancestor);
                }
            });
            // redirect all "reverse" links to parent
            // while also deleting links from parent
            this.reverseLinks.forEach((link) => {
                if (link.isDestroyed)
                    return;
                if (link.source === ancestor) {
                    link.mutateDestroy();
                }
                else {
                    link.mutateTarget(ancestor);
                }
            });
            this.mutateDestroy(true);
            ancestor._mutateReconcileLinks();
            return ancestor;
        }
        /**
         * Destroys this node by removing all references and removing this node from the store.
         * @param {boolean} [strict=false] - Indicates that this node must not have references
         * @destructive
         */
        mutateDestroy(strict = false) {
            this._preMutate();
            if (strict) {
                if (this.children.length) {
                    throw new Error(`[strict] ${this} can not destroys because it has children`);
                }
                if (this.links.length || this.reverseLinks.length) {
                    throw new Error(`[strict] ${this} can not destroys because there are links referencing it`);
                }
            }
            if (strict && (this.links.length || this.reverseLinks.length)) {
                throw new Error(`[strict] ${this} can not destroys because there are links referencing it`);
            }
            this.children.forEach((child) => {
                child.mutateDestroy();
            });
            this.links.forEach((link) => {
                link.mutateDestroy();
            });
            this.reverseLinks.forEach((link) => {
                link.mutateDestroy();
            });
            if (this.parent) {
                this.parent.mutateRemoveChild(this);
            }
            this._parent = undefined;
            this._stack = undefined;
            this.store.mutateRemoveNode(this);
            this._destroyed = true;
        }
        /**
         * Reconciles links defined by this node. During mutations, multiple *equivalent* links may exist and should be
         * consolidated into a single link. This operation should be called after collapsing children to remove duplicates.
         * @internal
         * @destructive
         */
        _mutateReconcileLinks() {
            this._preMutate();
            const links = this.links;
            for (const a of links) {
                if (a.isDestroyed)
                    continue;
                for (const b of links) {
                    if (a === b || b.isDestroyed)
                        continue;
                    if (a.isEquivalent(b)) {
                        a.mutateConsume(b);
                    }
                }
            }
            const reverseLinks = this.reverseLinks;
            for (const a of reverseLinks) {
                if (a.isDestroyed)
                    continue;
                for (const b of reverseLinks) {
                    if (a === b || b.isDestroyed)
                        continue;
                    if (a.isEquivalent(b)) {
                        a.mutateConsume(b);
                    }
                }
            }
        }
        /**
         * Remove a *child* node from *this node*
         * @destructive
         */
        mutateRemoveChild(node) {
            this._preMutate();
            if (!this.isChild(node)) {
                throw new Error(`${node} is not a child of ${this}`);
            }
            // NB: children are stored by "id" not "uuid"
            return this._children.delete(node.id);
        }
        /**
         * Remove a *link* from *this node*
         * @destructive
         */
        mutateRemoveLink(link) {
            this._preMutate();
            return this._links.delete(link.uuid);
        }
        /**
         * Remove a *link* to *this node*
         * @destructive
         */
        mutateRemoveReverseLink(link) {
            this._preMutate();
            return this._reverseLinks.delete(link.uuid);
        }
        /**
         * Hoist *this node* to an *ancestor* by removing it from its current parent node and
         * in turn moving it to the ancestor.
         * @destructive
         */
        mutateHoist(newParent) {
            this._preMutate();
            if (!this.isAncestor(newParent)) {
                throw new Error(`${newParent} is not an ancestor of ${this}`);
            }
            if (this.parent) {
                this.parent.mutateRemoveChild(this);
            }
            this._parent = newParent;
            newParent.addChild(this);
            if (this.stack &&
                this.stack !== this &&
                !this.isAncestor(this.stack)) {
                this._stack = this.findAncestor((node) => StackNode.isStackNode(node) ||
                    NestedStackNode.isNestedStackNode(node));
            }
        }
        /**
         * Hoist all children to parent and collapse node to parent.
         * @destructive
         */
        mutateUncluster() {
            this._preMutate();
            if (this.parent && !this.isLeaf) {
                for (const child of this.children) {
                    child.mutateHoist(this.parent);
                }
                this.mutateCollapseToParent();
            }
        }
        /** Get string representation of this node */
        toString() {
            return `Node:${this.nodeType}::${this.uuid}`;
        }
        /**
         * Serialize this node
         * @internal
         */
        _serialize() {
            return {
                ...super._serialize(),
                nodeType: this.nodeType,
                stack: this.stack?.uuid,
                parent: this.parent?.uuid,
                id: this.id,
                path: this.path,
                constructInfo: this.constructInfo,
                logicalId: this.logicalId,
                cfnType: this.cfnType,
                edges: this._links.size
                    ? Array.from(this._links.values()).map(({ uuid }) => uuid)
                    : undefined,
                children: this._children.size
                    ? Object.fromEntries(Array.from(this._children.entries()).map(([key, node]) => [
                        key,
                        node._serialize(),
                    ]))
                    : undefined,
            };
        }
    }
    _h = JSII_RTTI_SYMBOL_1;
    Node[_h] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.Node", version: "0.12.25" };
    Graph.Node = Node;
    /** ResourceNode class defines a L2 cdk resource construct */
    class ResourceNode extends Node {
        constructor(props) {
            super({
                nodeType: types_1.NodeTypeEnum.RESOURCE,
                ...props,
            });
            if (props.cdkOwned) {
                this.addFlag(types_1.FlagEnum.CDK_OWNED);
            }
        }
        /** Indicates if node is a {@link ResourceNode} */
        static isResourceNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.RESOURCE;
        }
        /** Get the CloudFormation resource type for this L2 resource or for the L1 resource is wraps. */
        get cfnType() {
            return (super.cfnType ||
                this.getAttribute(ResourceNode.ATT_WRAPPED_CFN_TYPE) ||
                this.cfnResource?.cfnType);
        }
        /** Indicates if this resource is owned by cdk (defined in cdk library) */
        get isCdkOwned() {
            return this.hasFlag(types_1.FlagEnum.CDK_OWNED);
        }
        /** Get the L1 cdk resource that this L2 resource wraps */
        get cfnResource() {
            if (this._cfnResource !== undefined) {
                if (this._cfnResource && this._cfnResource.isDestroyed)
                    return undefined;
                return this._cfnResource || undefined;
            }
            const resourceNode = this.findChild(types_1.CdkConstructIds.RESOURCE);
            if (resourceNode) {
                this._cfnResource = resourceNode;
                return resourceNode;
            }
            const defaultNode = this.findChild(types_1.CdkConstructIds.DEFAULT);
            if (defaultNode) {
                this._cfnResource = defaultNode;
                return defaultNode;
            }
            const childCfnResources = this.children.filter((node) => {
                return (CfnResourceNode.isCfnResourceNode(node) && node.isEquivalentFqn(this));
            });
            if (childCfnResources.length === 1) {
                this._cfnResource = childCfnResources[0];
                return childCfnResources[0];
            }
            // prevent looking up again by setting to `null`
            this._cfnResource = null;
            return undefined;
        }
        /** Get the cfn properties from the L1 resource that this L2 resource wraps */
        get cfnProps() {
            if (this.cfnResource) {
                return this.cfnResource.cfnProps;
            }
            return this.getAttribute(ResourceNode.ATT_WRAPPED_CFN_PROPS);
        }
        /**
         * Modifies the L1 resource wrapped by this L2 resource
         * @param cfnResource
         * @destructive
         */
        mutateCfnResource(cfnResource) {
            this._preMutate();
            this._cfnResource = cfnResource || null;
        }
        /** @inheritdoc */
        mutateRemoveChild(node) {
            if (this._cfnResource === node) {
                this.mutateCfnResource(undefined);
            }
            return super.mutateRemoveChild(node);
        }
    }
    _j = JSII_RTTI_SYMBOL_1;
    ResourceNode[_j] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.ResourceNode", version: "0.12.25" };
    /** Attribute key for cfn resource type */
    ResourceNode.ATT_WRAPPED_CFN_TYPE = "graph:resource:cfn-type";
    /** Attribute key for cfn properties */
    ResourceNode.ATT_WRAPPED_CFN_PROPS = "graph:resource:cfn-props";
    Graph.ResourceNode = ResourceNode;
    /** CfnResourceNode defines an L1 cdk resource */
    class CfnResourceNode extends Node {
        constructor(props) {
            super({
                nodeType: types_1.NodeTypeEnum.CFN_RESOURCE,
                ...props,
            });
            if (this.cfnType == null) {
                throw new Error("CfnResourceNode requires `cfnType` property");
            }
        }
        /** Indicates if a node is a {@link CfnResourceNode} */
        static isCfnResourceNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.CFN_RESOURCE;
        }
        /**
         * Evaluates if CfnResourceNode fqn is equivalent to ResourceNode fqn.
         * @example `aws-cdk-lib.aws_lambda.Function` => `aws-cdk-lib.aws_lambda.CfnFunction`
         * @param resource - {@link Graph.ResourceNode} to compare
         * @returns Returns `true` if equivalent, otherwise `false`
         */
        isEquivalentFqn(resource) {
            const resourceFqnStub = resource.constructInfoFqn
                ?.split(".")
                .pop()
                ?.toLowerCase();
            const cfnResourceFqnStub = this.constructInfoFqn
                ?.split(".")
                .pop()
                ?.toLowerCase();
            if (!resourceFqnStub || !cfnResourceFqnStub) {
                return false;
            }
            return `cfn${resourceFqnStub}` === cfnResourceFqnStub;
        }
        /**
         * Finds the near *ancestor* that is a {@link ResourceNode}
         */
        findNearestResource() {
            return this.scopes
                .slice()
                .reverse()
                .find((scope) => ResourceNode.isResourceNode(scope));
        }
        /**
         * @inheritdoc
         */
        mutateDestroy(strict) {
            const resource = this.findNearestResource();
            if (resource?.cfnResource === this) {
                resource.setAttribute(ResourceNode.ATT_WRAPPED_CFN_TYPE, this.cfnType);
                resource.setAttribute(ResourceNode.ATT_WRAPPED_CFN_PROPS, this.cfnProps);
                resource.mutateCfnResource(undefined);
            }
            super.mutateDestroy(strict);
        }
    }
    _k = JSII_RTTI_SYMBOL_1;
    CfnResourceNode[_k] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.CfnResourceNode", version: "0.12.25" };
    Graph.CfnResourceNode = CfnResourceNode;
    /** OutputNode defines a cdk CfnOutput resources */
    class OutputNode extends Node {
        constructor(props) {
            super({
                ...props,
                nodeType: types_1.NodeTypeEnum.OUTPUT,
            });
            /** Indicates if {@link OutputNode} is **exported** */
            this.isExport = false;
            if (this.stack == null) {
                throw new Error(`OutputNode instantiated outside of stack: ${this}`);
            }
            this.addFlag(types_1.FlagEnum.EXTRANEOUS);
            this.setAttribute(OutputNode.ATTR_VALUE, props.value);
            if (props.exportName) {
                this.isExport = true;
                this.setAttribute(OutputNode.ATTR_EXPORT_NAME, props.exportName);
            }
            props.description && this.setAttribute("description", props.description);
            this.stack.addOutput(this);
        }
        /** Indicates if node is an {@link OutputNode} */
        static isOutputNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.OUTPUT;
        }
        /** Get the *value** attribute */
        get value() {
            return this.getAttribute(OutputNode.ATTR_VALUE);
        }
        /** Get the export name attribute */
        get exportName() {
            return this.getAttribute(OutputNode.ATTR_EXPORT_NAME);
        }
        /** @inheritdoc */
        mutateDestroy(strict) {
            super.mutateDestroy(strict);
            this.stack?.mutateRemoveOutput(this);
        }
    }
    _l = JSII_RTTI_SYMBOL_1;
    OutputNode[_l] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.OutputNode", version: "0.12.25" };
    /** Attribute key where output value is stored */
    OutputNode.ATTR_VALUE = "graph:output:value";
    /** Attribute key where output export name is stored */
    OutputNode.ATTR_EXPORT_NAME = "graph:output:export-name";
    Graph.OutputNode = OutputNode;
    /** ParameterNode defines a CfnParameter node */
    class ParameterNode extends Node {
        constructor(props) {
            super({
                ...props,
                nodeType: types_1.NodeTypeEnum.PARAMETER,
            });
            if (this.stack == null) {
                throw new Error(`ParameterNode instantiated outside of stack: ${this}`);
            }
            this.addFlag(types_1.FlagEnum.EXTRANEOUS);
            this.setAttribute(ParameterNode.ATTR_VALUE, props.value);
            this.setAttribute(ParameterNode.ATTR_TYPE, props.parameterType);
            props.description && this.setAttribute("description", props.description);
            this.isStackReference = this.id.startsWith("reference-to-");
            this.stack.addParameter(this);
        }
        /** Indicates if node is a {@link ParameterNode} */
        static isParameterNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.PARAMETER;
        }
        /** Get the value attribute */
        get value() {
            return this.getAttribute(ParameterNode.ATTR_VALUE);
        }
        /** Get the parameter type attribute */
        get parameterType() {
            return this.getAttribute(ParameterNode.ATTR_TYPE);
        }
        /** @inheritdoc */
        mutateDestroy(strict) {
            super.mutateDestroy(strict);
            this.stack?.mutateRemoveParameter(this);
        }
    }
    _m = JSII_RTTI_SYMBOL_1;
    ParameterNode[_m] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.ParameterNode", version: "0.12.25" };
    /** Attribute key where parameter value is store */
    ParameterNode.ATTR_VALUE = "graph:parameter:value";
    /** Attribute key where parameter type is stored */
    ParameterNode.ATTR_TYPE = "graph:parameter:type";
    Graph.ParameterNode = ParameterNode;
    /** StackNode defines a cdk Stack */
    class StackNode extends Node {
        constructor(props) {
            super({
                nodeType: types_1.NodeTypeEnum.STACK,
                ...props,
            });
            /** @internal */
            this._outputs = new Set();
            /** @internal */
            this._parameters = new Set();
            if (this.stack !== this) {
                throw new Error(`Stack.stack is not self: ${this.uuid}`);
            }
            this.addFlag(types_1.FlagEnum.CLUSTER);
            this.store.addStack(this);
            const stage = this.findAncestor(StageNode.isStageNode);
            if (stage) {
                this._stage = stage;
                stage.addStack(this);
            }
        }
        /** Indicates if node is a {@link StackNode} */
        static isStackNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.STACK;
        }
        /**
         * Gets the {@link StackNode} containing a given resource
         * @throws Error is node is not contained in a stack
         */
        static of(node) {
            const stack = node.stack;
            if (stack == null) {
                throw new Error(`${String(node)} is not within StackNode`);
            }
            return stack;
        }
        /** Get {@link StageNode} containing this stack */
        get stage() {
            return this._stage;
        }
        /** Get all {@link OutputNode}s defined by this stack */
        get outputs() {
            return Array.from(this._outputs);
        }
        /** Get all {@link ParameterNode}s defined by this stack */
        get parameters() {
            return Array.from(this._parameters);
        }
        /** Get all **exported** {@link OutputNode}s defined by this stack */
        get exports() {
            return this.outputs.filter((node) => node.isExport);
        }
        /** Associate {@link OutputNode} with this stack */
        addOutput(node) {
            this._outputs.add(node);
        }
        /**
         * Find {@link OutputNode} with *logicalId* defined by this stack
         * @throws Error is no output found matching *logicalId*
         */
        findOutput(logicalId) {
            const output = this.outputs.find((_output) => _output.logicalId === logicalId);
            if (output == null) {
                console.debug(`${this}.Outputs: [logicalId]`, this.outputs.map((n) => n.logicalId));
                throw new Error(`Output ${logicalId} does not exist in ${this}`);
            }
            return output;
        }
        /** Associate {@link ParameterNode} with this stack */
        addParameter(node) {
            this._parameters.add(node);
        }
        /**
         * Find {@link ParameterNode} with *parameterId* defined by this stack
         * @throws Error is no parameter found matching *parameterId*
         */
        findParameter(parameterId) {
            const parameter = this.parameters.find((_parameter) => _parameter.id === parameterId);
            if (parameter == null) {
                console.debug(`${this}.Parameters: [id]`, this.parameters.map((n) => n.id));
                throw new Error(`Parameter ${parameterId} does not exist in ${this}`);
            }
            return parameter;
        }
        /**
         * Disassociate {@link OutputNode} from this stack
         * @destructive
         */
        mutateRemoveOutput(node) {
            this._preMutate();
            return this._outputs.delete(node);
        }
        /**
         * Disassociate {@link ParameterNode} from this stack
         * @destructive
         */
        mutateRemoveParameter(node) {
            this._preMutate();
            return this._parameters.delete(node);
        }
        /** @inheritdoc */
        mutateDestroy(strict) {
            super.mutateDestroy(strict);
            this.stage?.mutateRemoveStack(this);
        }
        /** @inheritdoc */
        mutateHoist(newParent) {
            super.mutateHoist(newParent);
            if (this.stage && this.isAncestor(this.stage)) {
                this.stage.mutateRemoveStack(this);
                this._stage = this.findAncestor((node) => StageNode.isStageNode(node));
                if (this._stage) {
                    this._stage.addStack(this);
                }
            }
        }
    }
    _o = JSII_RTTI_SYMBOL_1;
    StackNode[_o] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.StackNode", version: "0.12.25" };
    Graph.StackNode = StackNode;
    /** NestedStackNode defines a cdk NestedStack */
    class NestedStackNode extends StackNode {
        constructor(props) {
            super({
                ...props,
                nodeType: types_1.NodeTypeEnum.NESTED_STACK,
            });
            this._parentStack = props.parentStack;
        }
        /** Indicates if node is a {@link NestedStackNode} */
        static isNestedStackNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.NESTED_STACK;
        }
        /** Get parent stack of this nested stack */
        get parentStack() {
            return this._parentStack;
        }
        /** @inheritdoc */
        mutateHoist(newParent) {
            super.mutateHoist(newParent);
            if (this.parentStack && this.isAncestor(this.parentStack)) {
                this._parentStack = this.findAncestor((node) => StackNode.isStackNode(node));
            }
        }
    }
    _p = JSII_RTTI_SYMBOL_1;
    NestedStackNode[_p] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.NestedStackNode", version: "0.12.25" };
    Graph.NestedStackNode = NestedStackNode;
    /** StageNode defines a cdk Stage */
    class StageNode extends Node {
        constructor(props) {
            super({
                ...props,
                nodeType: types_1.NodeTypeEnum.STAGE,
            });
            /** @internal */
            this._stacks = new Set();
            this.store.addStage(this);
            this.addFlag(types_1.FlagEnum.CLUSTER);
        }
        /** Indicates if node is a {@link StageNode} */
        static isStageNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.STAGE;
        }
        /**
         * Gets the {@link StageNode} containing a given resource
         * @throws Error is node is not contained in a stage
         */
        static of(node) {
            const stage = node.rootStack?.stage;
            if (stage == null) {
                throw new Error(`${node} is not within a stage`);
            }
            return stage;
        }
        /** Gets all stacks contained by this stage */
        get stacks() {
            return Array.from(this._stacks);
        }
        /** Associate a {@link StackNode} with this stage */
        addStack(stack) {
            this._stacks.add(stack);
        }
        /**
         * Disassociate {@link StackNode} from this stage
         * @destructive
         */
        mutateRemoveStack(stack) {
            this._preMutate();
            return this._stacks.delete(stack);
        }
    }
    _q = JSII_RTTI_SYMBOL_1;
    StageNode[_q] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.StageNode", version: "0.12.25" };
    Graph.StageNode = StageNode;
    /** AppNode defines a cdk App */
    class AppNode extends Node {
        constructor(props) {
            super({
                ...props,
                nodeType: types_1.NodeTypeEnum.APP,
                uuid: AppNode.UUID,
                id: AppNode.UUID,
                path: AppNode.PATH,
            });
            this.addFlag(types_1.FlagEnum.GRAPH_CONTAINER);
            this.addFlag(types_1.FlagEnum.CLUSTER);
        }
        /** Indicates if node is a {@link AppNode} */
        static isAppNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.APP;
        }
    }
    _r = JSII_RTTI_SYMBOL_1;
    AppNode[_r] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.AppNode", version: "0.12.25" };
    /** Fixed UUID for App node */
    AppNode.UUID = "App";
    /** Fixed path of the App  */
    AppNode.PATH = "/";
    Graph.AppNode = AppNode;
    /** RootNode represents the root of the store tree */
    class RootNode extends Node {
        constructor(store) {
            super({
                store,
                nodeType: types_1.NodeTypeEnum.ROOT,
                uuid: RootNode.UUID,
                id: RootNode.UUID,
                path: RootNode.PATH,
            });
            this.addFlag(types_1.FlagEnum.GRAPH_CONTAINER);
            this.addFlag(types_1.FlagEnum.CLUSTER);
        }
        /** Indicates if node is a {@link RootNode} */
        static isRootNode(node) {
            return node.nodeType === types_1.NodeTypeEnum.ROOT;
        }
        /**
         * @inheritdoc **The root not is excluded from list**
         */
        findAll(options) {
            return super.findAll(options);
        }
        /**
         * > {@link RootNode} does not support this mutation
         * @throws Error does not support
         * @inheritdoc
         */
        mutateCollapse() {
            throw new Error("Root node can not be collapsed");
        }
        /**
         * > {@link RootNode} does not support this mutation
         * @throws Error does not support
         * @inheritdoc
         */
        mutateCollapseToParent() {
            throw new Error("Root node can not be collapsed to parent");
        }
        /**
         * > {@link RootNode} does not support this mutation
         * @throws Error does not support
         * @inheritdoc
         */
        mutateCollapseTo(_ancestor) {
            throw new Error("Root node can not be collapsed");
        }
        /**
         * > {@link RootNode} does not support this mutation
         * @throws Error does not support
         * @inheritdoc
         */
        mutateDestroy(_strict = false) {
            throw new Error("Root node can not be destroyed");
        }
        /**
         * > {@link RootNode} does not support this mutation
         * @throws Error does not support
         * @inheritdoc
         */
        mutateHoist(_newParent) {
            throw new Error("Root node can not be hoisted");
        }
    }
    _s = JSII_RTTI_SYMBOL_1;
    RootNode[_s] = { fqn: "@aws-prototyping-sdk/cdk-graph.Graph.RootNode", version: "0.12.25" };
    /** Fixed UUID of root */
    RootNode.UUID = "Root";
    /** Fixed path of root */
    RootNode.PATH = "";
    Graph.RootNode = RootNode;
})(Graph = exports.Graph || (exports.Graph = {}));
/**
 * Deserializes a *serialized store object* into an *in-memory store instance**.
 * @param serializedStore - The serialized store to deserialize
 * @param allowDestructiveMutations - Indicates if the store instance allows destructive mutations.
 * @throws Error if a serialized node's parent does not match visitor parent
 * @throws Error if a serialized node type deserialization mapping is not defined
 * @throws Error if edge type deserialization mapping is not defined
 */
function deserializeStore(serializedStore, allowDestructiveMutations = false) {
    const store = new Graph.Store(allowDestructiveMutations);
    // TODO: ensure store versions are compatible
    function visit(sNode, parent) {
        const nodeProps = {
            ...omit(sNode, [
                "children",
                "parent",
                "stack",
                "nodeType",
            ]),
            parent: sNode.parent ? store.getNode(sNode.parent) : undefined,
            // resolve stack node, unless stack is itself
            stack: sNode.stack && sNode.stack !== sNode.uuid
                ? store.getStack(sNode.stack)
                : undefined,
            store,
        };
        if (nodeProps.parent !== parent) {
            throw new Error(`SerializedNode parent ${sNode.parent} does not match visitor parent ${parent.uuid}`);
        }
        let node = undefined;
        switch (sNode.nodeType) {
            case types_1.NodeTypeEnum.APP: {
                node = new Graph.AppNode({
                    ...nodeProps,
                    parent,
                });
                break;
            }
            case types_1.NodeTypeEnum.STAGE: {
                node = new Graph.StageNode(nodeProps);
                break;
            }
            case types_1.NodeTypeEnum.STACK: {
                node = new Graph.StackNode(nodeProps);
                break;
            }
            case types_1.NodeTypeEnum.NESTED_STACK: {
                node = new Graph.NestedStackNode({
                    ...nodeProps,
                    parentStack: Graph.StackNode.of(parent),
                });
                break;
            }
            case types_1.NodeTypeEnum.OUTPUT: {
                node = new Graph.OutputNode({
                    ...nodeProps,
                    value: nodeProps.attributes[Graph.OutputNode.ATTR_VALUE],
                    exportName: nodeProps.attributes[Graph.OutputNode.ATTR_EXPORT_NAME],
                    description: nodeProps.attributes.description,
                });
                break;
            }
            case types_1.NodeTypeEnum.PARAMETER: {
                node = new Graph.ParameterNode({
                    ...nodeProps,
                    value: nodeProps.attributes[Graph.ParameterNode.ATTR_VALUE],
                    parameterType: nodeProps.attributes[Graph.ParameterNode.ATTR_TYPE],
                    description: nodeProps.attributes.description,
                });
                break;
            }
            case types_1.NodeTypeEnum.CFN_RESOURCE: {
                node = new Graph.CfnResourceNode(nodeProps);
                break;
            }
            case types_1.NodeTypeEnum.RESOURCE: {
                node = new Graph.ResourceNode({
                    ...nodeProps,
                    cdkOwned: !!nodeProps.flags?.includes(types_1.FlagEnum.CDK_OWNED),
                });
                break;
            }
            case types_1.NodeTypeEnum.DEFAULT: {
                node = new Graph.Node({
                    ...nodeProps,
                    nodeType: types_1.NodeTypeEnum.DEFAULT,
                });
                break;
            }
        }
        if (node == null) {
            console.debug(sNode.nodeType, sNode);
            throw new Error(`NodeType ${sNode.nodeType} missing deserialization mapping`);
        }
        // ensure node is registered in store
        console_1.assert(store.getNode(sNode.uuid) === node, `Node ${sNode.uuid} did not register in store`);
        Object.values(sNode.children || {}).forEach((sChild) => {
            visit(sChild, node);
        });
    }
    Object.values(serializedStore.tree.children || {}).forEach((sNode) => {
        visit(sNode, store.root);
    });
    serializedStore.edges.forEach((sEdge) => {
        const edgeProps = {
            ...sEdge,
            store,
            source: store.getNode(sEdge.source),
            target: store.getNode(sEdge.target),
        };
        switch (sEdge.edgeType) {
            case types_1.EdgeTypeEnum.DEPENDENCY: {
                new Graph.Dependency({
                    ...edgeProps,
                });
                break;
            }
            case types_1.EdgeTypeEnum.REFERENCE: {
                const referenceType = sEdge.attributes[Graph.Reference.ATT_TYPE];
                if (referenceType === types_1.ReferenceTypeEnum.ATTRIBUTE) {
                    new Graph.AttributeReference({
                        ...edgeProps,
                        value: sEdge.attributes[Graph.AttributeReference.ATT_VALUE],
                    });
                }
                else if (referenceType === types_1.ReferenceTypeEnum.IMPORT) {
                    new Graph.ImportReference({
                        ...edgeProps,
                    });
                }
                else if (referenceType === types_1.ReferenceTypeEnum.REF) {
                    new Graph.Reference(edgeProps);
                }
                else {
                    throw new Error(`Unsupported reference type of ${referenceType}`);
                }
                break;
            }
            default: {
                // TODO: support custom edge types explicitly
                new Graph.Edge(edgeProps);
            }
        }
    });
    return store;
}
exports.deserializeStore = deserializeStore;
//# sourceMappingURL=data:application/json;base64,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