import { ISynthesisSession } from "aws-cdk-lib";
import { Construct, IConstruct } from "constructs";
import { CdkGraphConfig } from "./config";
import { Graph, Version } from "./core";
/** CdkGraph core artifacts */
export declare enum CdkGraphArtifacts {
    GRAPH_METADATA = "graph-metadata.json",
    GRAPH = "graph.json"
}
/**
 * CdkGraph artifact definition
 * @struct
 */
export interface CdkGraphArtifact {
    /** The unique type of the artifact */
    readonly id: string;
    /** Filename of the artifact */
    readonly filename: string;
    /** Full path where artifact is stored */
    readonly filepath: string;
    /** Description of artifact */
    readonly description?: string;
    /** The source of the artifact (such as plugin, or core system, etc) */
    readonly source: string;
}
/** Dictionary of graph artifacts by id */
export declare type CdkGraphArtifactDict = {
    [id: string]: CdkGraphArtifact;
};
/** CdkGraph context */
export declare class CdkGraphContext {
    readonly store: Graph.Store;
    readonly outdir: string;
    /** @internal */
    readonly _artifacts: CdkGraphArtifactDict;
    constructor(store: Graph.Store, outdir: string);
    /**
     * Get CdkGraph artifact by id
     * @throws Error is artifact does not exist
     */
    getArtifact(id: string): CdkGraphArtifact;
    /** Get CdkGraph core `graph.json` artifact */
    get graphJson(): CdkGraphArtifact;
    /** Indicates if context has an artifact with *filename* defined */
    hasArtifactFile(filename: string): boolean;
    /** Get record of all graph artifacts keyed by artifact id */
    get artifacts(): CdkGraphArtifactDict;
    /**
     * Logs an artifact entry. In general this should not be called directly, as `writeArtifact` should be utilized
     * to perform writing and logging artifacts. However some plugins utilize other tools that generate the artifacts,
     * in which case the plugin would call this method to log the entry.
     * @param source The source of the artifact, such as the name of plugin
     * @param id Unique id of the artifact
     * @param filepath Full path where the artifact is stored
     * @param description Description of the artifact
     * @returns
     * @throws Error is artifact id or filename already exists
     */
    logArtifact(source: CdkGraph | ICdkGraphPlugin, id: string, filepath: string, description?: string): CdkGraphArtifact;
    /**
     * Writes artifact data to outdir and logs the entry.
     * @param source The source of the artifact, such as the name of plugin
     * @param id Unique id of the artifact
     * @param filename Relative name of the file
     * @param description Description of the artifact
     * @returns
     */
    writeArtifact(source: CdkGraph | ICdkGraphPlugin, id: string, filename: string, data: string, description?: string): CdkGraphArtifact;
}
/** Callback signature for graph `Plugin.bind` operation */
export interface IGraphPluginBindCallback {
    (graph: CdkGraph): void;
}
/** Callback signature for graph `Plugin.inspect` operation */
export interface IGraphVisitorCallback {
    (construct: IConstruct): void;
}
/** Callback signature for graph `Plugin.synthesize` operation */
export interface IGraphSynthesizeCallback {
    (context: CdkGraphContext): void;
}
/** Callback signature for graph `Plugin.report` operation */
export interface IGraphReportCallback {
    (context: CdkGraphContext): Promise<void>;
}
/** CdkGraph **Plugin** interface */
export interface ICdkGraphPlugin {
    /** Unique identifier for this plugin */
    readonly id: string;
    /** Plugin version */
    readonly version: Version;
    /** List of plugins this plugin depends on, including optional semver version (eg: ["foo", "bar@1.2"]) */
    readonly dependencies?: string[];
    /**
     * Binds the plugin to the CdkGraph instance. Enables plugins to receive base configs.
     */
    bind: IGraphPluginBindCallback;
    /**
     * Node visitor callback for construct tree traversal. This follows IAspect.visit pattern, but the order
     * of visitor traversal in managed by the CdkGraph.
     */
    inspect?: IGraphVisitorCallback;
    /**
     * Called during CDK synthesize to generate synchronous artifacts based on the in-memory graph passed
     * to the plugin. This is called in fifo order of plugins.
     */
    synthesize?: IGraphSynthesizeCallback;
    /**
     * Generate asynchronous reports based on the graph. This is not automatically called when synthesizing CDK.
     * Developer must explicitly add `await graphInstance.report()` to the CDK bin or invoke this outside
     * of the CDK synth. In either case, the plugin receives the in-memory graph interface when invoked, as the
     * CdkGraph will deserialize the graph prior to invoking the plugin report.
     */
    report?: IGraphReportCallback;
}
/** {@link CdkGraph} props */
export interface ICdkGraphProps {
    /** List of plugins to extends the graph. Plugins are invoked at each phases in fifo order. */
    plugins?: ICdkGraphPlugin[];
}
/**
 * CdkGraph construct is the cdk-graph framework controller that is responsible for
 * computing the graph, storing serialized graph, and instrumenting plugins per the
 * plugin contract.
 */
export declare class CdkGraph extends Construct {
    readonly root: Construct;
    /** Fixed CdkGraph construct id */
    static readonly ID = "CdkGraph";
    /** Current CdkGraph semantic version */
    static readonly VERSION = "0.0.0";
    /** List of plugins registered with this instance */
    readonly plugins: ICdkGraphPlugin[];
    /** @internal */
    private _context?;
    /** Config */
    readonly config: CdkGraphConfig;
    /**
     * Get the context for the graph instance.
     *
     * This will be `undefined` before construct synthesis has initiated.
     */
    get graphContext(): CdkGraphContext | undefined;
    constructor(root: Construct, props?: ICdkGraphProps);
    /** @internal */
    protected _synthesize(session: ISynthesisSession): void;
    /**
     * Asynchronous report generation. This operation enables running expensive and non-synchronous
     * report generation by plugins post synthesis.
     *
     * If a given plugin requires performing asynchronous operations or is general expensive, it should
     * utilize `report` rather than `synthesize`.
     */
    report(): Promise<void>;
}
