"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
test('Symbol map is generated', () => {
    const result = (0, lib_1.compileJsiiForTest)({
        'index.ts': `
        export * from './some/nested/file';
        export class Foo {
          public bar(){}
        }
      `,
        'some/nested/file.ts': `
        export interface Bar {
          readonly x: string;
        }
        export enum Baz {
          ALPHA = 0,
          BETA = 1,
        }
        `,
    }, undefined /* callback */, { stripDeprecated: true });
    const types = result.assembly.types ?? {};
    expect(types['testpkg.Foo'].symbolId).toEqual('index:Foo');
    expect(types['testpkg.Bar'].symbolId).toEqual('some/nested/file:Bar');
    expect(types['testpkg.Baz'].symbolId).toEqual('some/nested/file:Baz');
});
test('Symbol id for single-value enum correctly identifies enum', () => {
    const result = (0, lib_1.compileJsiiForTest)({
        'index.ts': `
        export enum SomeEnum {
          SINGLETON_VALUE = 'value',
        }
      `,
    }, undefined /* callback */, { stripDeprecated: true });
    const types = result.assembly.types ?? {};
    expect(types['testpkg.SomeEnum'].symbolId).toEqual('index:SomeEnum');
});
test('Module declarations are included in symbolId', () => {
    const result = (0, lib_1.compileJsiiForTest)({
        'index.ts': `
        export class Foo {
          constructor() {
          }
        }
        export namespace Foo {
          export class Bar {
            public baz() {}
          }
        }
      `,
    }, undefined /* callback */, { stripDeprecated: true });
    const types = result.assembly.types ?? {};
    expect(types['testpkg.Foo.Bar'].symbolId).toEqual('index:Foo.Bar');
});
test('Submodules also have symbol identifiers', () => {
    const result = (0, lib_1.compileJsiiForTest)({
        'index.ts': `export * as submod from './submodule';`,
        'submodule.ts': `
        export class Foo {
          constructor() {
          }
        }
      `,
    }, undefined /* callback */, { stripDeprecated: true });
    expect(result.assembly.submodules?.['testpkg.submod']?.symbolId).toEqual('submodule:');
});
test('Submodules also have symbol identifiers', () => {
    const result = (0, lib_1.compileJsiiForTest)({
        'index.ts': `
        export namespace cookie {
          export class Foo {
            constructor() {
            }
          }
        }
      `,
    }, undefined /* callback */, { stripDeprecated: true });
    expect(result.assembly.submodules?.['testpkg.cookie']?.symbolId).toEqual('index:cookie');
});
describe(lib_1.normalizePath, () => {
    test('basic rootDir and outDir', () => {
        expect((0, lib_1.normalizePath)('out/filename.ts', 'root', 'out')).toEqual('root/filename.ts');
        expect((0, lib_1.normalizePath)('out/filename.ts', undefined, 'out')).toEqual('out/filename.ts');
        expect((0, lib_1.normalizePath)('out/filename.ts', 'root', undefined)).toEqual('out/filename.ts');
        expect((0, lib_1.normalizePath)('out/filename.ts', undefined, undefined)).toEqual('out/filename.ts');
    });
    test('extra slashes in directories', () => {
        expect((0, lib_1.normalizePath)('out/filename.ts', 'root/', 'out/')).toEqual('root/filename.ts');
        expect((0, lib_1.normalizePath)('out/filename.ts', 'root////', 'out////')).toEqual('root/filename.ts');
        // eslint-disable-next-line prettier/prettier
        expect((0, lib_1.normalizePath)('out/lib/filename.ts', 'root///', 'out//lib//')).toEqual('root/filename.ts');
    });
    test('additional paths in directories', () => {
        expect((0, lib_1.normalizePath)('out/filename.ts', './root', 'out')).toEqual('root/filename.ts');
        expect((0, lib_1.normalizePath)('out/filename.ts', 'root', './out')).toEqual('root/filename.ts');
        expect((0, lib_1.normalizePath)('out/filename.ts', 'root', './here/../out')).toEqual('root/filename.ts');
        expect((0, lib_1.normalizePath)('out/filename.ts', 'root/../root/..', '.')).toEqual('out/filename.ts');
    });
    test('empty paths', () => {
        expect((0, lib_1.normalizePath)('out/lib/filename.ts', '', 'out')).toEqual('lib/filename.ts');
        expect((0, lib_1.normalizePath)('out/lib/filename.ts', '.', 'out')).toEqual('lib/filename.ts');
        expect((0, lib_1.normalizePath)('lib/filename.ts', 'root', '')).toEqual('root/lib/filename.ts');
        expect((0, lib_1.normalizePath)('lib/filename.ts', 'root', '.')).toEqual('root/lib/filename.ts');
        // eslint-disable-next-line prettier/prettier
        expect((0, lib_1.normalizePath)('lib/filename.ts', '', '')).toEqual('lib/filename.ts');
        expect((0, lib_1.normalizePath)('lib/filename.ts', '.', '.')).toEqual('lib/filename.ts');
    });
    test('specify multiple directories', () => {
        expect((0, lib_1.normalizePath)('out/lib/filename.ts', 'root', 'out/lib')).toEqual('root/filename.ts');
        expect((0, lib_1.normalizePath)('out/lib/filename.ts', 'root/extra', 'out')).toEqual('root/extra/lib/filename.ts');
        expect((0, lib_1.normalizePath)('out/lib/filename.ts', '.', 'out/lib')).toEqual('filename.ts');
        expect((0, lib_1.normalizePath)('lib/filename.ts', 'root/extra', '.')).toEqual('root/extra/lib/filename.ts');
    });
});
//# sourceMappingURL=symbol-identifiers.test.js.map