"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const nag_rules_1 = require("../../nag-rules");
/**
 * CloudFront distributions use an origin access identity for S3 origins
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_cloudfront_1.CfnDistribution) {
        const distributionConfig = aws_cdk_lib_1.Stack.of(node).resolve(node.distributionConfig);
        if (distributionConfig.origins != undefined) {
            const origins = aws_cdk_lib_1.Stack.of(node).resolve(distributionConfig.origins);
            for (const origin of origins) {
                const resolvedOrigin = aws_cdk_lib_1.Stack.of(node).resolve(origin);
                const resolvedDomainName = aws_cdk_lib_1.Stack.of(node).resolve(resolvedOrigin.domainName);
                const s3Regex = /^.+\.s3(?:-website)?(?:\..+)?(?:(?:\.amazonaws\.com(?:\.cn)?)|(?:\.c2s\.ic\.gov)|(?:\.sc2s\.sgov\.gov))$/;
                if (s3Regex.test(resolvedDomainName)) {
                    if (resolvedOrigin.s3OriginConfig == undefined) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                    const resolvedConfig = aws_cdk_lib_1.Stack.of(node).resolve(resolvedOrigin.s3OriginConfig);
                    if (resolvedConfig.originAccessIdentity == undefined ||
                        resolvedConfig.originAccessIdentity.replace(/\s/g, '').length == 0) {
                        return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
                    }
                }
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else if (node instanceof aws_cloudfront_1.CfnStreamingDistribution) {
        const distributionConfig = aws_cdk_lib_1.Stack.of(node).resolve(node.streamingDistributionConfig);
        const resolvedOrigin = aws_cdk_lib_1.Stack.of(node).resolve(distributionConfig.s3Origin);
        if (resolvedOrigin.originAccessIdentity.replace(/\s/g, '').length == 0) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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