/*
 * Decompiled with CFR 0.152.
 */
package oracle.jdbc.util;

import java.sql.SQLException;
import java.util.Map;
import java.util.Properties;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import oracle.jdbc.OracleDriver;
import oracle.jdbc.driver.utils.ThrowingSupplier;
import oracle.jdbc.util.OracleConfigurationProviderNetworkError;
import oracle.jdbc.util.ThreadSafeCache;
import oracle.net.nt.TimeoutInterruptHandler;

public final class OracleConfigurationCache {
    private static final String CONFIG_TTL_JSON_OBJECT_NAME = "config_time_to_live";
    private static final long TEN_YEARS_IN_SECONDS = TimeUnit.DAYS.toSeconds(3650L);
    private final ExecutorService executor;
    private final ThreadSafeCache<String, CacheEntry> cache;
    private final Map<String, Object> tasks = new ConcurrentHashMap<String, Object>();

    private OracleConfigurationCache(int maxEntries, ExecutorService executor) {
        this.cache = new ThreadSafeCache(maxEntries);
        this.executor = executor;
    }

    public static OracleConfigurationCache create(int maxEntries) {
        try {
            ExecutorService executor = OracleDriver.getExecutorService();
            return new OracleConfigurationCache(maxEntries, executor);
        }
        catch (SQLException sqlException) {
            throw new IllegalStateException(sqlException);
        }
    }

    public void put(String location, Properties properties, ThrowingSupplier<Properties, OracleConfigurationProviderNetworkError> refreshSupplier, long msTimeout, long msRefreshInterval) {
        this.putCacheEntry(location, new CacheEntry(properties, refreshSupplier, msTimeout, msRefreshInterval), true);
    }

    public void put(String location, Properties properties, long configTimeToLive, ThrowingSupplier<Properties, OracleConfigurationProviderNetworkError> refreshSupplier, long msTimeout, long msRefreshInterval) {
        if (configTimeToLive > TEN_YEARS_IN_SECONDS) {
            this.putCacheEntry(location, new CacheEntry(properties, configTimeToLive, refreshSupplier, msTimeout, msRefreshInterval), false);
        } else {
            this.putCacheEntry(location, new CacheEntry(properties, configTimeToLive, refreshSupplier, msTimeout, msRefreshInterval), true);
        }
    }

    public Properties get(String location) {
        CacheEntry entry = this.cache.get(location);
        if (entry == null || entry.state() == CacheState.HARDLY_EXPIRED) {
            return null;
        }
        return entry.properties();
    }

    public Properties remove(String location) {
        CacheEntry entry = this.cache.remove(location);
        Object task = this.tasks.remove(location);
        if (task != null) {
            if (task instanceof TimerTask) {
                ((TimerTask)task).cancel();
            } else {
                ((Future)task).cancel(true);
            }
        }
        return entry == null ? null : entry.properties();
    }

    private void putCacheEntry(String location, CacheEntry cacheEntry, boolean startBackgroundThread) {
        this.cache.put(location, cacheEntry);
        if (startBackgroundThread) {
            TimerTask scheduledTask = this.scheduleUpdate(location, cacheEntry.expirationTimeMillis() - System.currentTimeMillis());
            this.tasks.put(location, scheduledTask);
        }
    }

    private TimerTask scheduleUpdate(String location, long msDelay) {
        return TimeoutInterruptHandler.scheduleTask(() -> {
            CacheEntry currentEntry = this.cache.get(location);
            if (currentEntry == null) {
                return;
            }
            CacheState state = currentEntry.state();
            switch (state) {
                case SOFTLY_EXPIRED: {
                    Future<?> future = this.requestUpdate(location, currentEntry);
                    this.tasks.put(location, future);
                    this.scheduleTimeout(future, currentEntry.msTimeout());
                    break;
                }
                case HARDLY_EXPIRED: {
                    this.cache.remove(location);
                    break;
                }
                default: {
                    throw new IllegalStateException("Unexpected value: " + String.valueOf((Object)state));
                }
            }
        }, msDelay);
    }

    private Future<?> requestUpdate(String location, CacheEntry entry) {
        ThrowingSupplier<Properties, OracleConfigurationProviderNetworkError> refreshSupplier = entry.refreshSupplier();
        long msTimeout = entry.msTimeout();
        long msRefreshInterval = entry.msRefreshInterval();
        return this.executor.submit(() -> {
            try {
                Properties refreshedConfig = (Properties)refreshSupplier.getOrThrow();
                String ttlString = refreshedConfig.getProperty(CONFIG_TTL_JSON_OBJECT_NAME);
                if (ttlString == null) {
                    this.put(location, refreshedConfig, refreshSupplier, msTimeout, msRefreshInterval);
                } else {
                    long refreshedTTL = Long.parseLong(ttlString);
                    refreshedConfig.remove(CONFIG_TTL_JSON_OBJECT_NAME);
                    this.put(location, refreshedConfig, refreshedTTL, refreshSupplier, msTimeout, msRefreshInterval);
                }
            }
            catch (OracleConfigurationProviderNetworkError networkError) {
                Logger.getLogger("oracle.jdbc.driver").log(Level.WARNING, "Failed to get the configuration from remote location. Task skipped.", networkError);
                TimerTask scheduledTask = this.scheduleUpdate(location, msRefreshInterval);
                this.tasks.put(location, scheduledTask);
            }
        });
    }

    private void scheduleTimeout(Future<?> future, long msTimeout) {
        TimeoutInterruptHandler.scheduleTask(() -> {
            if (!future.isDone()) {
                future.cancel(true);
                Logger.getLogger("oracle.jdbc.driver").log(Level.WARNING, "Call to the remote location did not complete within " + msTimeout + " ms. Thread interrupted.");
            }
        }, msTimeout);
    }

    private final class CacheEntry {
        private static final long DEFAULT_TTL = 86400000L;
        private static final long SOFT_EXPIRATION_TIME = 1800000L;
        private long expirationTimeMillis;
        private Properties properties;
        private ThrowingSupplier<Properties, OracleConfigurationProviderNetworkError> refreshSupplier;
        private long msTimeout;
        private long msRefreshInterval;

        public CacheEntry(Properties properties, ThrowingSupplier<Properties, OracleConfigurationProviderNetworkError> refreshSupplier, long msTimeout, long msRefreshInterval) {
            this(properties, 86400000L, refreshSupplier, msTimeout, msRefreshInterval);
        }

        public CacheEntry(Properties properties, long timeToLive, ThrowingSupplier<Properties, OracleConfigurationProviderNetworkError> refreshSupplier, long msTimeout, long msRefreshInterval) {
            this.properties = properties;
            this.refreshSupplier = refreshSupplier;
            this.msTimeout = msTimeout;
            this.msRefreshInterval = msRefreshInterval;
            if (timeToLive < 0L) {
                throw new IllegalArgumentException("time-to-live value cannot be lessthan 0");
            }
            this.expirationTimeMillis = System.currentTimeMillis() + timeToLive * 1000L;
        }

        public long expirationTimeMillis() {
            return this.expirationTimeMillis;
        }

        public Properties properties() {
            return this.properties;
        }

        public ThrowingSupplier<Properties, OracleConfigurationProviderNetworkError> refreshSupplier() {
            return this.refreshSupplier;
        }

        public long msTimeout() {
            return this.msTimeout;
        }

        public long msRefreshInterval() {
            return this.msRefreshInterval;
        }

        public CacheState state() {
            long currentMillis = System.currentTimeMillis();
            if (currentMillis < this.expirationTimeMillis) {
                return CacheState.ALIVE;
            }
            if (currentMillis >= this.expirationTimeMillis && currentMillis < this.expirationTimeMillis + 1800000L) {
                return CacheState.SOFTLY_EXPIRED;
            }
            return CacheState.HARDLY_EXPIRED;
        }
    }

    public static enum CacheState {
        ALIVE,
        SOFTLY_EXPIRED,
        HARDLY_EXPIRED;

    }
}

