"""
@author: Uwe Ziegenhagen, ziegenhagen@gmail.com
"""

import pandas as pd
import numpy as np
from .Transaction import Transaction
from .Classification import Classification
from .Category import Category
from .Account import Account

t = Transaction()
temp_classification = Classification()
temp_category = Category()
temp_account = Account()

class PyQifParser():
    """
    Parses a QIF File generated by (German) Quicken 2021.
    """

    def __init__(self, qiffile, encoding='cp1252'):
        self.transactions = pd.DataFrame(columns=('Date', 'Amount',
                                                  'Category', 'Reference',
                                                  'Payee', 'Description',
                                                  'Cleared'))
        self.classifications = pd.DataFrame(columns=('Label', 'Description'))
        self.categories = pd.DataFrame(columns=('Label', 'Description',
                                                'Parent', 'Type'))
        self.accounts = pd.DataFrame(columns=('Label', 'Description', 'Type'))
        self.__qiffile = qiffile
        self.__mode = None
        self.encoding = encoding
        self.dateformat = '%m.%d.%y'
        self.__autoswitch = None


    def get_transactions(self):
        """
            Returns the dataframe for the transactions, sets columns 'Month'
            and 'Year' based on the transaction date
        """
        # add columns for Month and Year
        try:
            self.transactions['Year'] = self.transactions['Date'].dt.year
            self.transactions['Month'] = self.transactions['Date'].dt.month
        except AttributeError:
            self.transactions['Year'] = np.NaN
            self.transactions['Month'] = np.NaN
        return self.transactions
    
    def df_memberpayments(self, outputfile):
        """
            calculates a Pivot of payments from the transactions for
            a specific year. This is specific for usage at Dingfabrik.de
        """
        t = self.get_transactions()
        t['Category'] = t['Category'].fillna('')
        t = t[t['Category'].str.startswith('Mitgliedsbeitrag_2110')]
        t['Mitglied'] = t['Category'].str.split("/",expand=True)[1].astype(int) #   + ' - ' + t['Payee'] 
        t.to_clipboard()
        table = pd.pivot_table(t, values='Amount', index=['Mitglied'], columns=['Year', 'Month'], aggfunc=np.sum)
        table.to_excel(outputfile)
    
    
    def transactions_to_pickle(self, path):
        """
            Exports the transactions into a pickle file
        """
        # add columns for Month and Year
        self.transactions['Year'] = self.transactions['Date'].dt.year
        self.transactions['Month'] = self.transactions['Date'].dt.month
        self.transactions.to_pickle(path)
    

    def to_excel(self, outputfile):
        """
            Exports the transactions, accounts, classifications and
            categories into an Excel-file
        """
        with pd.ExcelWriter(outputfile, engine='xlsxwriter') as writer:
            self.transactions.to_excel(writer, sheet_name='Transactions', index=False)
            self.accounts.to_excel(writer, sheet_name='Accounts', index=False)
            self.classifications.to_excel(writer, sheet_name='Classifications', index=False)
            self.categories.to_excel(writer, sheet_name='Categories', index=False)
            writer.save()
            

    def to_beancount(self, outputfile):
        """
            Exports the transactions, accounts, classifications and
            categories into an Excel-file
            @TODO: waits for detailed specifications, currency is still hardwired
        """
        self.transactions['bcdate'] = self.transactions['Date'].dt.strftime('%Y-%m-%d')
        with open(outputfile, "w", encoding="utf-8") as writer:
            for entry in self.transactions.index: 
                writer.write(self.transactions['bcdate'][entry] + ' * "' + str(self.transactions['Description'][entry])[:50]  + '"\n')
                writer.write('\t' + self.transactions['Category'][entry] + '\t'*10 +  str(self.transactions['Amount'][entry]) + ' EUR\n')
                writer.write('\t' + self.transactions['Category'][entry] + '\t'*10 +  str(-1 * self.transactions['Amount'][entry]) + ' EUR\n\n')
                
            

    def mode(self, mode):
        """
            The parser can be in one of several modes. The mode determines,
            how the individual lines are handled.
        """
        if mode not in ('classifications', 'categories', 'transactions',
                        'accounts', 'other', 'options'):
            raise ValueError('Unknown mode "' + mode + '"')
        else:
            self.__mode = mode

    def handle_option(self, line):
        """
            sets the date format and the autoswitch (used with list of accounts)
            MDY could be different for other languages
        """
        splits = line.split(':')
        if splits[1][:-1] == 'MDY':
            self.dateformat = '%m.%d.%y'
        elif splits[1][:-1] == 'AutoSwitch':
            self.__autoswitch = True # we are in the list of accounts

    def handle_other(self, line):
        """
            handles the individual lines of the file
            based on the current mode
        """
        if self.__mode == 'classifications':
            if line.startswith('N'):
                temp_classification.clear()
                temp_classification.label = line[1:-1]
            elif line.startswith('D'):
                temp_classification.description = line[1:-1]
            elif line.startswith('^'):
                self.classifications = self.classifications.append(
                    {'Label': temp_classification.label,
                     'Description': temp_classification.description},
                    ignore_index=True)

        elif self.__mode == 'categories':
            if line.startswith('N'):
                temp_category.clear()
                temp_category.label = line[1:-1]
            elif line[:-1].startswith('S'):
                temp_category.parent = line[1:-1]
            elif line[:-1].startswith('D'):
                temp_category.description = line[:-1]
            elif line[:-1] in ['I', 'E']:
                temp_category.type = line[:-1]
            elif line.startswith('^'):
                self.categories = self.categories.append(
                    {'Label': temp_category.label,
                     'Parent': temp_category.parent,
                     'Description': temp_category.description,
                     'Type':temp_category.type},
                    ignore_index=True)

        elif self.__mode == 'accounts':
            if line.startswith('N'):
                temp_account.clear()
                temp_account.label = line[1:-1]
            elif line.startswith('T'):
                temp_account.type = line[1:-1]
            elif line.startswith('D'):
                temp_account.description = line[1:-1]
            elif line.startswith('^'):
                self.accounts = self.accounts.append(
                    {'Label': temp_account.label,
                     'Type' : temp_account.type,
                     'Description': temp_account.description},
                    ignore_index=True)

        elif self.__mode == 'transactions':
            if line.startswith('D'):
                t.clear()
                t.date = line[1:-1]
            elif line.startswith('T'):
                t.amount = line[1:-1]
            elif line.startswith('C'):
                t.cleared = line[1:-1]
            elif line.startswith('N'):
                t.reference = line[1:-1]
            elif line.startswith('P'):
                t.payee = line[1:-1]
            elif line.startswith('M'):
                t.description = line[1:-1]
            elif line.startswith('L'):
                t.category = line[1:-1]
            elif line.startswith('^'):
                self.transactions = self.transactions.append(
                    {'Date': pd.to_datetime(t.date, format=self.dateformat),
                     'Amount': t.amount, 'Cleared': t.cleared,
                     'Category': t.category, 'Payee': t.payee,
                     'Reference': t.reference,
                     'Description': t.description},
                    ignore_index=True)


    def parse(self):
        """
            parses the QIF file and fills the internal data structures
        """
        with open(self.__qiffile, encoding=self.encoding) as inputfile:
            for line in inputfile:

                if line.startswith('!Option'):
                    self.mode('options')
                    self.handle_option(line)
                elif line.startswith('!Type:Class'):
                    self.mode('classifications')
                elif line.startswith('!Type:Cat'):
                    self.mode('categories')
                elif line.startswith('!Type:Bank') or line.startswith('!Type:Cash') or line.startswith('!Type:CCard'):
                    self.mode('transactions')
                elif line.startswith('!Account'):
                    if self.__autoswitch:
                        self.mode('accounts')
                elif line.startswith('!Clear:AutoSwitch'):
                    self.__autoswitch = False
                elif line.startswith('!'):
                    self.mode('other')
                else:
                    self.handle_other(line)
