"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findCfnTypeAssets = exports.parseCfnType = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const assets_1 = require("../../generated/assets");
const utils_1 = require("../../utils");
const helpers_1 = require("../resources/helpers");
const helpers_2 = require("../services/helpers");
/** Parse CloudFormation resource type string into parts */
function parseCfnType(cfnType) {
    return cfnType.match(/(?<provider>\w+)::(?<serviceName>\w+)::(?<resourceName>\w+)/).groups;
}
exports.parseCfnType = parseCfnType;
// VPC is nested in EC2 within CloudFormation but separate in other systems
const VpcAssetComparables = Object.fromEntries(Object.keys(assets_1.AwsAsset.Resources).reduce((entries, key) => {
    if (key.startsWith("vpc_")) {
        entries.push([utils_1.normalizeComparisonString(key.replace("vpc_", "")), key]);
    }
    return entries;
}, []));
/** Find asset type details for given CloudFormation type */
function findCfnTypeAssets(cfnType) {
    const cfn = parseCfnType(cfnType);
    let serviceName;
    let resourceName;
    // handle edge cases (eg: EC2::VPCxxx is vpc:xxx in assets)
    if (cfn.serviceName === "EC2") {
        if (cfn.resourceName.startsWith("VPC")) {
            serviceName = "vpc";
            cfn.resourceName = cfn.resourceName.replace(/^VPC/, "");
        }
        else if (utils_1.normalizeComparisonString(cfn.resourceName) in VpcAssetComparables) {
            serviceName = "vpc";
            cfn.resourceName = utils_1.normalizeComparisonString(cfn.resourceName);
        }
    }
    if (serviceName == null) {
        try {
            serviceName = helpers_2.resolveServiceName(cfn.serviceName);
        }
        catch (e) {
            console.warn(e.message, cfnType);
        }
    }
    if (resourceName == null) {
        // There are lots of low-level cfn resource definitions without mappings to other systems,
        // for this reason we just ignore unresolved resources without spamming the console or
        // bubbling the error as this is expected in large percent of cases.
        try {
            resourceName = helpers_1.resolveResourceName(cfn.resourceName, serviceName);
        }
        catch { }
    }
    const generalIcon = resolveGeneralIcon(cfn.resourceName);
    return {
        serviceName,
        resourceName,
        generalIcon,
    };
}
exports.findCfnTypeAssets = findCfnTypeAssets;
/** Resolve general icon for a given value.  */
function resolveGeneralIcon(value) {
    value = utils_1.normalizeComparisonString(value);
    if (value.match(/(configuration|config|setting)$/i)) {
        return "config";
    }
    if (value.match(/^(client)?(certificate)$/i)) {
        return "ssl_padlock";
    }
    if (value.match(/(user)$/i)) {
        return "user";
    }
    if (value.match(/(policy|permissions?)$/i)) {
        return "policy";
    }
    if (value.match(/(key)$/i)) {
        return "key";
    }
    if (value.match(/(role)$/i)) {
        return "role";
    }
    if (value.match(/(server)$/i)) {
        return "traditional_server";
    }
    if (value.match(/(database)$/i)) {
        return "generic_database";
    }
    if (value.match(/(log)s?$/i)) {
        return "logs";
    }
    if (value.match(/(alarm|alert)s?$/i)) {
        return "alarm";
    }
    if (value.match(/(event)s?$/i)) {
        return "event";
    }
    if (value.match(/(rule)s?$/i)) {
        return "rule";
    }
    if (value.match(/(branch)$/i)) {
        return "git_repository";
    }
    if (value.match(/^(app|application)$/i)) {
        return "generic_application";
    }
    if (value.match(/^(documentation)/i)) {
        return "document";
    }
    if (value.match(/^(model)$/i)) {
        return "document";
    }
    if (value.match(/(template)/i)) {
        return "template";
    }
    if (value.match(/(attachment)$/i)) {
        return "attachment";
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaGVscGVycy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9pbnRlcm5hbC9tYXBwaW5nL2hlbHBlcnMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUE7c0NBQ3NDO0FBQ3RDLG1EQUFrRDtBQUVsRCx1Q0FBd0Q7QUFDeEQsa0RBQTJEO0FBQzNELGlEQUF5RDtBQVN6RCwyREFBMkQ7QUFDM0QsU0FBZ0IsWUFBWSxDQUMxQixPQUE2QjtJQUU3QixPQUFPLE9BQU8sQ0FBQyxLQUFLLENBQ2xCLDZEQUE2RCxDQUM3RCxDQUFDLE1BQWEsQ0FBQztBQUNuQixDQUFDO0FBTkQsb0NBTUM7QUFTRCwyRUFBMkU7QUFDM0UsTUFBTSxtQkFBbUIsR0FBMkIsTUFBTSxDQUFDLFdBQVcsQ0FDcEUsTUFBTSxDQUFDLElBQUksQ0FBQyxpQkFBUSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLE9BQU8sRUFBRSxHQUFHLEVBQUUsRUFBRTtJQUN0RCxJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLEVBQUU7UUFDMUIsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLGlDQUF5QixDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztLQUN6RTtJQUNELE9BQU8sT0FBTyxDQUFDO0FBQ2pCLENBQUMsRUFBRSxFQUF3QixDQUFDLENBQzdCLENBQUM7QUFFRiw0REFBNEQ7QUFDNUQsU0FBZ0IsaUJBQWlCLENBQy9CLE9BQTZCO0lBRTdCLE1BQU0sR0FBRyxHQUFHLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUVsQyxJQUFJLFdBQXlDLENBQUM7SUFDOUMsSUFBSSxZQUEyQyxDQUFDO0lBRWhELDJEQUEyRDtJQUMzRCxJQUFJLEdBQUcsQ0FBQyxXQUFXLEtBQUssS0FBSyxFQUFFO1FBQzdCLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDdEMsV0FBVyxHQUFHLEtBQUssQ0FBQztZQUNwQixHQUFHLENBQUMsWUFBWSxHQUFHLEdBQUcsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztTQUN6RDthQUFNLElBQ0wsaUNBQXlCLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxJQUFJLG1CQUFtQixFQUNsRTtZQUNBLFdBQVcsR0FBRyxLQUFLLENBQUM7WUFDcEIsR0FBRyxDQUFDLFlBQVksR0FBRyxpQ0FBeUIsQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLENBQUM7U0FDaEU7S0FDRjtJQUVELElBQUksV0FBVyxJQUFJLElBQUksRUFBRTtRQUN2QixJQUFJO1lBQ0YsV0FBVyxHQUFHLDRCQUFrQixDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUNuRDtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsT0FBTyxDQUFDLElBQUksQ0FBRSxDQUFXLENBQUMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1NBQzdDO0tBQ0Y7SUFFRCxJQUFJLFlBQVksSUFBSSxJQUFJLEVBQUU7UUFDeEIsMEZBQTBGO1FBQzFGLHNGQUFzRjtRQUN0RixvRUFBb0U7UUFDcEUsSUFBSTtZQUNGLFlBQVksR0FBRyw2QkFBbUIsQ0FBQyxHQUFHLENBQUMsWUFBWSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1NBQ25FO1FBQUMsTUFBTSxHQUFFO0tBQ1g7SUFFRCxNQUFNLFdBQVcsR0FBRyxrQkFBa0IsQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLENBQUM7SUFFekQsT0FBTztRQUNMLFdBQVc7UUFDWCxZQUFZO1FBQ1osV0FBVztLQUNaLENBQUM7QUFDSixDQUFDO0FBN0NELDhDQTZDQztBQUVELCtDQUErQztBQUMvQyxTQUFTLGtCQUFrQixDQUFDLEtBQWE7SUFDdkMsS0FBSyxHQUFHLGlDQUF5QixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRXpDLElBQUksS0FBSyxDQUFDLEtBQUssQ0FBQyxrQ0FBa0MsQ0FBQyxFQUFFO1FBQ25ELE9BQU8sUUFBUSxDQUFDO0tBQ2pCO0lBQ0QsSUFBSSxLQUFLLENBQUMsS0FBSyxDQUFDLDJCQUEyQixDQUFDLEVBQUU7UUFDNUMsT0FBTyxhQUFhLENBQUM7S0FDdEI7SUFDRCxJQUFJLEtBQUssQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLEVBQUU7UUFDM0IsT0FBTyxNQUFNLENBQUM7S0FDZjtJQUNELElBQUksS0FBSyxDQUFDLEtBQUssQ0FBQyx5QkFBeUIsQ0FBQyxFQUFFO1FBQzFDLE9BQU8sUUFBUSxDQUFDO0tBQ2pCO0lBQ0QsSUFBSSxLQUFLLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxFQUFFO1FBQzFCLE9BQU8sS0FBSyxDQUFDO0tBQ2Q7SUFDRCxJQUFJLEtBQUssQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLEVBQUU7UUFDM0IsT0FBTyxNQUFNLENBQUM7S0FDZjtJQUNELElBQUksS0FBSyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsRUFBRTtRQUM3QixPQUFPLG9CQUFvQixDQUFDO0tBQzdCO0lBQ0QsSUFBSSxLQUFLLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxFQUFFO1FBQy9CLE9BQU8sa0JBQWtCLENBQUM7S0FDM0I7SUFDRCxJQUFJLEtBQUssQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLEVBQUU7UUFDNUIsT0FBTyxNQUFNLENBQUM7S0FDZjtJQUNELElBQUksS0FBSyxDQUFDLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFO1FBQ3BDLE9BQU8sT0FBTyxDQUFDO0tBQ2hCO0lBQ0QsSUFBSSxLQUFLLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxFQUFFO1FBQzlCLE9BQU8sT0FBTyxDQUFDO0tBQ2hCO0lBQ0QsSUFBSSxLQUFLLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxFQUFFO1FBQzdCLE9BQU8sTUFBTSxDQUFDO0tBQ2Y7SUFDRCxJQUFJLEtBQUssQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLEVBQUU7UUFDN0IsT0FBTyxnQkFBZ0IsQ0FBQztLQUN6QjtJQUNELElBQUksS0FBSyxDQUFDLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxFQUFFO1FBQ3ZDLE9BQU8scUJBQXFCLENBQUM7S0FDOUI7SUFDRCxJQUFJLEtBQUssQ0FBQyxLQUFLLENBQUMsbUJBQW1CLENBQUMsRUFBRTtRQUNwQyxPQUFPLFVBQVUsQ0FBQztLQUNuQjtJQUNELElBQUksS0FBSyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsRUFBRTtRQUM3QixPQUFPLFVBQVUsQ0FBQztLQUNuQjtJQUNELElBQUksS0FBSyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsRUFBRTtRQUM5QixPQUFPLFVBQVUsQ0FBQztLQUNuQjtJQUNELElBQUksS0FBSyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFO1FBQ2pDLE9BQU8sWUFBWSxDQUFDO0tBQ3JCO0lBRUQsT0FBTyxTQUFTLENBQUM7QUFDbkIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qISBDb3B5cmlnaHQgW0FtYXpvbi5jb21dKGh0dHA6Ly9hbWF6b24uY29tLyksIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5TUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMCAqL1xuaW1wb3J0IHsgQXdzQXNzZXQgfSBmcm9tIFwiLi4vLi4vZ2VuZXJhdGVkL2Fzc2V0c1wiO1xuaW1wb3J0IHsgQ2ZuU3BlYyB9IGZyb20gXCIuLi8uLi9nZW5lcmF0ZWQvY2Zuc3BlY1wiO1xuaW1wb3J0IHsgbm9ybWFsaXplQ29tcGFyaXNvblN0cmluZyB9IGZyb20gXCIuLi8uLi91dGlsc1wiO1xuaW1wb3J0IHsgcmVzb2x2ZVJlc291cmNlTmFtZSB9IGZyb20gXCIuLi9yZXNvdXJjZXMvaGVscGVyc1wiO1xuaW1wb3J0IHsgcmVzb2x2ZVNlcnZpY2VOYW1lIH0gZnJvbSBcIi4uL3NlcnZpY2VzL2hlbHBlcnNcIjtcblxuLyoqIFBhcnNlZCBDbG91ZEZvcm1hdGlvbiByZXNvdXJjZSB0eXBlIHN0cmluZyBwYXJ0cyAqL1xuZXhwb3J0IGludGVyZmFjZSBDZm5SZXNvdXJjZVR5cGVQYXJ0cyB7XG4gIHByb3ZpZGVyOiBzdHJpbmc7XG4gIHNlcnZpY2VOYW1lOiBDZm5TcGVjLlNlcnZpY2VOYW1lO1xuICByZXNvdXJjZU5hbWU6IHN0cmluZztcbn1cblxuLyoqIFBhcnNlIENsb3VkRm9ybWF0aW9uIHJlc291cmNlIHR5cGUgc3RyaW5nIGludG8gcGFydHMgKi9cbmV4cG9ydCBmdW5jdGlvbiBwYXJzZUNmblR5cGUoXG4gIGNmblR5cGU6IENmblNwZWMuUmVzb3VyY2VUeXBlXG4pOiBDZm5SZXNvdXJjZVR5cGVQYXJ0cyB7XG4gIHJldHVybiBjZm5UeXBlLm1hdGNoKFxuICAgIC8oPzxwcm92aWRlcj5cXHcrKTo6KD88c2VydmljZU5hbWU+XFx3Kyk6Oig/PHJlc291cmNlTmFtZT5cXHcrKS9cbiAgKSEuZ3JvdXBzIGFzIGFueTtcbn1cblxuLyoqIERlZmluaXRhaW9uIG9mIGFzc2V0IG1hcHBpbmdzIHRvIENsb3VkRm9ybWF0aW9uIHR5cGUgKi9cbmludGVyZmFjZSBDZm5UeXBlQXNzZXRzIHtcbiAgcmVhZG9ubHkgc2VydmljZU5hbWU/OiBBd3NBc3NldC5TZXJ2aWNlO1xuICByZWFkb25seSByZXNvdXJjZU5hbWU/OiBBd3NBc3NldC5SZXNvdXJjZTtcbiAgcmVhZG9ubHkgZ2VuZXJhbEljb24/OiBBd3NBc3NldC5HZW5lcmFsSWNvbjtcbn1cblxuLy8gVlBDIGlzIG5lc3RlZCBpbiBFQzIgd2l0aGluIENsb3VkRm9ybWF0aW9uIGJ1dCBzZXBhcmF0ZSBpbiBvdGhlciBzeXN0ZW1zXG5jb25zdCBWcGNBc3NldENvbXBhcmFibGVzOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+ID0gT2JqZWN0LmZyb21FbnRyaWVzKFxuICBPYmplY3Qua2V5cyhBd3NBc3NldC5SZXNvdXJjZXMpLnJlZHVjZSgoZW50cmllcywga2V5KSA9PiB7XG4gICAgaWYgKGtleS5zdGFydHNXaXRoKFwidnBjX1wiKSkge1xuICAgICAgZW50cmllcy5wdXNoKFtub3JtYWxpemVDb21wYXJpc29uU3RyaW5nKGtleS5yZXBsYWNlKFwidnBjX1wiLCBcIlwiKSksIGtleV0pO1xuICAgIH1cbiAgICByZXR1cm4gZW50cmllcztcbiAgfSwgW10gYXMgW3N0cmluZywgc3RyaW5nXVtdKVxuKTtcblxuLyoqIEZpbmQgYXNzZXQgdHlwZSBkZXRhaWxzIGZvciBnaXZlbiBDbG91ZEZvcm1hdGlvbiB0eXBlICovXG5leHBvcnQgZnVuY3Rpb24gZmluZENmblR5cGVBc3NldHMoXG4gIGNmblR5cGU6IENmblNwZWMuUmVzb3VyY2VUeXBlXG4pOiBDZm5UeXBlQXNzZXRzIHtcbiAgY29uc3QgY2ZuID0gcGFyc2VDZm5UeXBlKGNmblR5cGUpO1xuXG4gIGxldCBzZXJ2aWNlTmFtZTogQXdzQXNzZXQuU2VydmljZSB8IHVuZGVmaW5lZDtcbiAgbGV0IHJlc291cmNlTmFtZTogQXdzQXNzZXQuUmVzb3VyY2UgfCB1bmRlZmluZWQ7XG5cbiAgLy8gaGFuZGxlIGVkZ2UgY2FzZXMgKGVnOiBFQzI6OlZQQ3h4eCBpcyB2cGM6eHh4IGluIGFzc2V0cylcbiAgaWYgKGNmbi5zZXJ2aWNlTmFtZSA9PT0gXCJFQzJcIikge1xuICAgIGlmIChjZm4ucmVzb3VyY2VOYW1lLnN0YXJ0c1dpdGgoXCJWUENcIikpIHtcbiAgICAgIHNlcnZpY2VOYW1lID0gXCJ2cGNcIjtcbiAgICAgIGNmbi5yZXNvdXJjZU5hbWUgPSBjZm4ucmVzb3VyY2VOYW1lLnJlcGxhY2UoL15WUEMvLCBcIlwiKTtcbiAgICB9IGVsc2UgaWYgKFxuICAgICAgbm9ybWFsaXplQ29tcGFyaXNvblN0cmluZyhjZm4ucmVzb3VyY2VOYW1lKSBpbiBWcGNBc3NldENvbXBhcmFibGVzXG4gICAgKSB7XG4gICAgICBzZXJ2aWNlTmFtZSA9IFwidnBjXCI7XG4gICAgICBjZm4ucmVzb3VyY2VOYW1lID0gbm9ybWFsaXplQ29tcGFyaXNvblN0cmluZyhjZm4ucmVzb3VyY2VOYW1lKTtcbiAgICB9XG4gIH1cblxuICBpZiAoc2VydmljZU5hbWUgPT0gbnVsbCkge1xuICAgIHRyeSB7XG4gICAgICBzZXJ2aWNlTmFtZSA9IHJlc29sdmVTZXJ2aWNlTmFtZShjZm4uc2VydmljZU5hbWUpO1xuICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgIGNvbnNvbGUud2FybigoZSBhcyBFcnJvcikubWVzc2FnZSwgY2ZuVHlwZSk7XG4gICAgfVxuICB9XG5cbiAgaWYgKHJlc291cmNlTmFtZSA9PSBudWxsKSB7XG4gICAgLy8gVGhlcmUgYXJlIGxvdHMgb2YgbG93LWxldmVsIGNmbiByZXNvdXJjZSBkZWZpbml0aW9ucyB3aXRob3V0IG1hcHBpbmdzIHRvIG90aGVyIHN5c3RlbXMsXG4gICAgLy8gZm9yIHRoaXMgcmVhc29uIHdlIGp1c3QgaWdub3JlIHVucmVzb2x2ZWQgcmVzb3VyY2VzIHdpdGhvdXQgc3BhbW1pbmcgdGhlIGNvbnNvbGUgb3JcbiAgICAvLyBidWJibGluZyB0aGUgZXJyb3IgYXMgdGhpcyBpcyBleHBlY3RlZCBpbiBsYXJnZSBwZXJjZW50IG9mIGNhc2VzLlxuICAgIHRyeSB7XG4gICAgICByZXNvdXJjZU5hbWUgPSByZXNvbHZlUmVzb3VyY2VOYW1lKGNmbi5yZXNvdXJjZU5hbWUsIHNlcnZpY2VOYW1lKTtcbiAgICB9IGNhdGNoIHt9XG4gIH1cblxuICBjb25zdCBnZW5lcmFsSWNvbiA9IHJlc29sdmVHZW5lcmFsSWNvbihjZm4ucmVzb3VyY2VOYW1lKTtcblxuICByZXR1cm4ge1xuICAgIHNlcnZpY2VOYW1lLFxuICAgIHJlc291cmNlTmFtZSxcbiAgICBnZW5lcmFsSWNvbixcbiAgfTtcbn1cblxuLyoqIFJlc29sdmUgZ2VuZXJhbCBpY29uIGZvciBhIGdpdmVuIHZhbHVlLiAgKi9cbmZ1bmN0aW9uIHJlc29sdmVHZW5lcmFsSWNvbih2YWx1ZTogc3RyaW5nKTogQXdzQXNzZXQuR2VuZXJhbEljb24gfCB1bmRlZmluZWQge1xuICB2YWx1ZSA9IG5vcm1hbGl6ZUNvbXBhcmlzb25TdHJpbmcodmFsdWUpO1xuXG4gIGlmICh2YWx1ZS5tYXRjaCgvKGNvbmZpZ3VyYXRpb258Y29uZmlnfHNldHRpbmcpJC9pKSkge1xuICAgIHJldHVybiBcImNvbmZpZ1wiO1xuICB9XG4gIGlmICh2YWx1ZS5tYXRjaCgvXihjbGllbnQpPyhjZXJ0aWZpY2F0ZSkkL2kpKSB7XG4gICAgcmV0dXJuIFwic3NsX3BhZGxvY2tcIjtcbiAgfVxuICBpZiAodmFsdWUubWF0Y2goLyh1c2VyKSQvaSkpIHtcbiAgICByZXR1cm4gXCJ1c2VyXCI7XG4gIH1cbiAgaWYgKHZhbHVlLm1hdGNoKC8ocG9saWN5fHBlcm1pc3Npb25zPykkL2kpKSB7XG4gICAgcmV0dXJuIFwicG9saWN5XCI7XG4gIH1cbiAgaWYgKHZhbHVlLm1hdGNoKC8oa2V5KSQvaSkpIHtcbiAgICByZXR1cm4gXCJrZXlcIjtcbiAgfVxuICBpZiAodmFsdWUubWF0Y2goLyhyb2xlKSQvaSkpIHtcbiAgICByZXR1cm4gXCJyb2xlXCI7XG4gIH1cbiAgaWYgKHZhbHVlLm1hdGNoKC8oc2VydmVyKSQvaSkpIHtcbiAgICByZXR1cm4gXCJ0cmFkaXRpb25hbF9zZXJ2ZXJcIjtcbiAgfVxuICBpZiAodmFsdWUubWF0Y2goLyhkYXRhYmFzZSkkL2kpKSB7XG4gICAgcmV0dXJuIFwiZ2VuZXJpY19kYXRhYmFzZVwiO1xuICB9XG4gIGlmICh2YWx1ZS5tYXRjaCgvKGxvZylzPyQvaSkpIHtcbiAgICByZXR1cm4gXCJsb2dzXCI7XG4gIH1cbiAgaWYgKHZhbHVlLm1hdGNoKC8oYWxhcm18YWxlcnQpcz8kL2kpKSB7XG4gICAgcmV0dXJuIFwiYWxhcm1cIjtcbiAgfVxuICBpZiAodmFsdWUubWF0Y2goLyhldmVudClzPyQvaSkpIHtcbiAgICByZXR1cm4gXCJldmVudFwiO1xuICB9XG4gIGlmICh2YWx1ZS5tYXRjaCgvKHJ1bGUpcz8kL2kpKSB7XG4gICAgcmV0dXJuIFwicnVsZVwiO1xuICB9XG4gIGlmICh2YWx1ZS5tYXRjaCgvKGJyYW5jaCkkL2kpKSB7XG4gICAgcmV0dXJuIFwiZ2l0X3JlcG9zaXRvcnlcIjtcbiAgfVxuICBpZiAodmFsdWUubWF0Y2goL14oYXBwfGFwcGxpY2F0aW9uKSQvaSkpIHtcbiAgICByZXR1cm4gXCJnZW5lcmljX2FwcGxpY2F0aW9uXCI7XG4gIH1cbiAgaWYgKHZhbHVlLm1hdGNoKC9eKGRvY3VtZW50YXRpb24pL2kpKSB7XG4gICAgcmV0dXJuIFwiZG9jdW1lbnRcIjtcbiAgfVxuICBpZiAodmFsdWUubWF0Y2goL14obW9kZWwpJC9pKSkge1xuICAgIHJldHVybiBcImRvY3VtZW50XCI7XG4gIH1cbiAgaWYgKHZhbHVlLm1hdGNoKC8odGVtcGxhdGUpL2kpKSB7XG4gICAgcmV0dXJuIFwidGVtcGxhdGVcIjtcbiAgfVxuICBpZiAodmFsdWUubWF0Y2goLyhhdHRhY2htZW50KSQvaSkpIHtcbiAgICByZXR1cm4gXCJhdHRhY2htZW50XCI7XG4gIH1cblxuICByZXR1cm4gdW5kZWZpbmVkO1xufVxuIl19