from typing import Optional
from urllib import parse

from .base_client import DatabaseClient
from .database_credentials import DatabaseCredentials

import pyodbc


class PyodbcClient(DatabaseClient):
    def __init__(self, credentials: DatabaseCredentials, database: str, driver: Optional[str] = None, **kwargs):
        super().__init__(credentials, database, **kwargs)
        self.driver = driver or self._get_best_driver()
        self._connection_string = self._build_connection_string()

    def _build_connection_string(self) -> str:
        """Build the pyodbc connection string."""
        return (
            f"DRIVER={{{self.driver}}};"
            f"SERVER={self.credentials.host};"
            f"DATABASE={self.database};"
            f"UID={self.credentials.username};"
            f"PWD={self.credentials.password};"
            f"TrustServerCertificate=yes;"
        )

    def get_connection(self):
        return pyodbc.connect(self._connection_string)

    def execute_query(
        self, query: str, return_results: bool = False, parameters: Optional[list] = None
    ) -> list[pyodbc.Row] | None:
        """
        Execute a SQL query.

        Args:
            query: SQL query to execute
            return_results: Whether to return query results
            parameters: Optional list of parameters for parameterized queries

        Returns:
            List of rows if return_results is True, None otherwise
        """
        with self.get_connection() as conn:
            cursor = conn.cursor()

            if parameters:
                cursor.execute(query, parameters)
            else:
                cursor.execute(query)

            if return_results:
                return cursor.fetchall()

            conn.commit()
