from abc import ABC, abstractmethod
from typing import Optional, Any
import re

from .database_credentials import DatabaseCredentials

import pyodbc


class DatabaseClient(ABC):
    """Base interface for database clients."""

    def __init__(self, credentials: DatabaseCredentials, database: str, **kwargs):
        self.credentials = credentials
        self.database = database
        self.kwargs = kwargs

    @staticmethod
    def extract_version_number(driver_name: str) -> int:
        """Extract version number from SQL Server driver name."""
        match = re.search(r"Driver (\d+)", driver_name)
        if match:
            return int(match.group(1))

        numbers = re.findall(r"\d+", driver_name)
        if numbers:
            return int(numbers[0])

        return 0

    def _get_best_driver(self) -> str:
        """Get the best available SQL Server driver."""
        available_drivers = [d for d in pyodbc.drivers() if "SQL Server" in d]
        if not available_drivers:
            raise RuntimeError("No SQL Server drivers found")

        return sorted(available_drivers, key=self.extract_version_number, reverse=True)[0]

    @abstractmethod
    def get_connection(self) -> Any:
        """Get a database connection."""
        pass

    @abstractmethod
    def execute_query(
        self, query: str, parameters: Optional[list[Any]] = None, return_results: bool = False
    ) -> Optional[list[Any]]:
        """Execute a SQL query."""
        pass

    def execute_stored_procedure(
        self, procedure_name: str, arguments: Optional[dict[str, Any]] = None, return_results: bool = False
    ) -> Optional[list[Any]]:
        """Execute a stored procedure. Default implementation for simple cases."""

        if arguments is None:
            arguments = {}

        params = ", ".join([f"@{key} = '{value}'" for key, value in arguments.items()])
        query = f"EXEC {procedure_name} {params}"

        return self.execute_query(query=query, return_results=return_results)
