from pathlib import Path

from lxml import etree as et

from .connection import Connection
from .parse_operation import ParseOperation
from .tag import Tag
from ..helpers.parse import get_xml_from_archive, save_into_archive


class Workbook(object):
    """Represents a Tableau workbook and provides methods for extraction, manipulation, and saving.

    Allows the extraction of connection information, finding XML elements matching a given XPath,
    opening and saving Tableau workbooks, and applying a series of ParseOperations to modify the workbook content.

    Args:
        file (Path): The path to the Tableau workbook file.

    Raises:
        FileNotFoundError: If the specified workbook file does not exist.
        ValueError: If the file is not a Tableau workbook with a valid extension (.twb or .twbx).
    """

    def __init__(self, file: Path) -> None:
        self._filename = file

        if not self._filename.exists():
            raise FileNotFoundError(f"File {self._filename} does not exist.")

        if self._filename.suffix not in (".twb", ".twbx"):
            raise ValueError(f"File {self._filename} is not a Tableau workbook.")

        self._workbook_tree = self.open(self._filename)
        self._workbook_root = self._workbook_tree.getroot()

    def __repr__(self) -> str:
        return f"Workbook(file={self._filename})"

    @property
    def suffix(self):
        return self._filename.suffix

    @property
    def tree(self):
        return self._workbook_tree

    @property
    def root(self):
        return self._workbook_root

    @staticmethod
    def open(filename: Path) -> et.ElementTree:
        if filename.suffix == ".twbx":
            return get_xml_from_archive(filename)

        return et.parse(filename)

    def extract_views(self) -> set[str]:
        return {
            item.attrib["name"]
            for item in self._workbook_root.xpath(
                "//window[(@class='dashboard' or @class='worksheet') and not(@hidden='true')]"
            )
        }

    def extract_connections(self) -> list[Connection]:
        unique_connections = {}

        connections = [
            Connection(i) for i in self._workbook_root.xpath("//connection[@class='sqlserver' or @class='postgres']")
        ]

        for connection in connections:
            key = (connection.server, connection.username)
            if key not in unique_connections:
                unique_connections[key] = connection

        return list(unique_connections.values())

    def find_matches(self, xpath: str) -> list[Tag]:
        return [
            Tag(i)
            for i in self._workbook_root.xpath(
                xpath,
                namespaces={"re": "http://exslt.org/regular-expressions"},
            )
        ]

    def save(self, save_to: Path) -> None:
        if self._filename.suffix == ".twbx":
            save_into_archive(self._workbook_tree, self._filename, save_to)
        else:
            self._workbook_tree.write(save_to, encoding="utf-8", xml_declaration=True)

    def parse(self, operations: list[ParseOperation]) -> None:
        for operation in operations:
            for tag in self.find_matches(operation.xpath):
                match operation.operation:
                    case "replace":
                        tag.update_attribute(operation.attribute, operation.value)
                    case "substitute":
                        tag.update_attribute_partial(operation.attribute, operation.value, operation.to_replace)
                    case "delete":
                        tag.delete()
