import re
from lxml import etree as et


class Tag(object):
    """Represents an XML tag and provides methods for manipulation.

    Allows for the modification and deletion of the XML tag and its attributes.

    Args:
        xml (et.Element): The XML element representing the tag.
    """

    def __init__(self, xml: et.Element) -> None:
        self._xml = xml

    def __str__(self) -> str:
        return et.tostring(self._xml, encoding="unicode")

    @property
    def xml(self) -> et.Element:
        return self._xml

    def tostring(self) -> str:
        attrs = "".join(f' {k}="{v}"' for k, v in self._xml.attrib.items())
        text = self._xml.text or ""
        return f"<{self._xml.tag}{attrs}>{text}</{self._xml.tag}>"

    def delete(self) -> None:
        self._xml.getparent().remove(self._xml)

    def update_attribute(self, attribute: str, value: str) -> None:
        if attribute.lower() == "text":
            self._xml.text = value
            return

        if attribute.lower() not in self._xml.attrib.keys():
            raise KeyError(f"Attribute {attribute.lower()} does not exist.")

        if self._xml.attrib.get("datatype") == "string" and re.match(r'"(.+)"', value) is None:
            value = '"%s"' % value

        self._xml.attrib[attribute.lower()] = value

    def update_attribute_partial(self, attribute: str, value: str, to_replace: str) -> None:
        value = re.sub(
            to_replace,
            value,
            self._xml.attrib.get(attribute) or self._xml.text,
        )
        self.update_attribute(attribute, value)
