import os
from pathlib import Path

import tableauserverclient as tsc


class TableauSite(object):
    """Represents a Tableau site and provides methods for interacting with Tableau Server.

    This class facilitates authentication, retrieval of projects and workbooks,
    publishing and refreshing workbooks on a Tableau Server site.

    Args:
        server_name (str): The URL of the Tableau Server.
        site (str): The name of the Tableau site.
    """

    def __init__(self, server_name: str, site: str) -> None:
        self._site = site

        self._server = tsc.Server(server_name, use_server_version=True)
        self._authenticate(server_name)

    def __repr__(self) -> str:
        return f"TableauSite(server={self._server.server_address}, site={self._site})"

    def _authenticate(self, server: str):
        if server == "https://portal.valuemetrics.nl":
            password = os.environ["PORTAL_PASSWORD_PROD"]
        else:
            password = os.environ["PORTAL_PASSWORD_DEV"]

        authenticator = tsc.TableauAuth(
            username=os.environ["PORTAL_USER"],
            password=password,
            site_id=self._site,
        )

        self._server.auth.sign_in(authenticator)

    def get_project(self, project_name: str) -> tsc.ProjectItem:
        projects = self._server.projects.filter(name=project_name)

        if not projects:
            raise LookupError(f"Project named {project_name} not found.")

        return projects[0]

    def get_workbook(self, product_name: str, workbook_name: str) -> tsc.WorkbookItem:
        workbooks = self._server.workbooks.filter(project_name=product_name, name=workbook_name)

        if not workbooks:
            raise LookupError(f"Workbook named {workbook_name} not found.")

        if len(workbooks) > 1:
            raise ValueError(f"Multiple workbooks named {workbook_name} found.")

        return workbooks[0]

    def publish_workbook(
        self,
        workbook: tsc.WorkbookItem,
        file: Path,
        connections: list[tsc.ConnectionItem] = [],
    ) -> None:
        job = self._server.workbooks.publish(
            workbook_item=workbook,
            file=file,
            mode="Overwrite",
            connections=connections,
            skip_connection_check=False,
            as_job=True,
        )

        self._server.jobs.wait_for_job(job.id)

    def refresh_workbook(self, workbook: tsc.WorkbookItem) -> None:
        job = self._server.workbooks.refresh(workbook.id)
        self._server.jobs.wait_for_job(job.id)
