from dataclasses import dataclass
from string import Template
from typing import Optional
import html

from lxml import etree as et


@dataclass
class ParseOperation:
    """Represents an XML parsing operation.

    Attributes:
        operation (str): The type of operation to perform. Must be one of: replace, substitute, delete.
        xpath (str): The xpath expression to use for the operation.
        value (str): The value to use for the operation.
        attribute (str): The attribute to use for the operation (required for replace and subsitute operations).
        to_replace (str): The value to replace when operation is substitute (required for substitute operation).
    """

    operation: str
    xpath: str
    value: str
    attribute: Optional[str] = None
    to_replace: Optional[str] = None

    def __post_init__(self):
        if self.operation not in ("replace", "substitute", "delete"):
            raise ValueError("operation must be one of: replace, substitute, delete")

        if self.operation == "substitute" and not self.to_replace:
            raise ValueError("to_replace must be set when type is substitute")

        if self.operation in ("replace", "substitute") and not self.attribute:
            raise ValueError("attribute must be set when type is replace or substitute")

        if self.operation == "delete":
            self.xpath = html.unescape(Template(self.xpath).substitute(VAR=self.value))

        if not self.is_valid_xpath(self.xpath):
            raise ValueError(f"'{self.xpath}' is not a valid xpath expression")

    @staticmethod
    def is_valid_xpath(xpath: str) -> bool:
        try:
            et.XPath(xpath)
            return True
        except et.XPathSyntaxError:
            return False

    @classmethod
    def from_dict(cls, data: dict):
        kwargs = {key: data[key] for key in cls.__annotations__.keys() if key in data}
        return cls(**kwargs)
