from lxml import etree as et


class Connection(object):
    """Represents a database connection extracted from a Tableau workbook.

    Encapsulates the details of a database connection, including server information,
    database name, username, and a list of tables associated with the connection.

    Args:
        xml (Element): The XML element representing the connection.
    """

    def __init__(self, xml: et.Element) -> None:
        self._xml = xml

        self._server = self._xml.get("server")
        self._database = self._xml.get("dbname")
        self._username = self._xml.get("username")

        self._tables = self._extract_tables(self._xml)

    def __repr__(self) -> str:
        return f"Connection(server={self._server}, database={self._database}, username={self._username})"

    @property
    def server(self) -> str:
        return self._server

    @property
    def database(self) -> str:
        return self._database

    @property
    def username(self) -> str:
        return self._username

    @property
    def tables(self) -> list[str]:
        return self._tables

    @staticmethod
    def _extract_tables(root) -> list[str]:
        tables = []

        for table in root.xpath(".//relation[starts-with(@connection, 'sqlserver')]"):
            if all(i in table.attrib.keys() for i in ("table", "connection")):
                tables.append(table)

        return tables
