import os
import tempfile
import zipfile
from pathlib import Path
from lxml import etree as et


def get_xml_from_archive(filename: Path):
    with zipfile.ZipFile(filename, allowZip64=True) as zf:
        twb_file = find_file_in_zip(zf)
        if not twb_file:
            raise FileNotFoundError(f"Could not find a .twb file in {filename}.")

        with zf.open(twb_file) as f:
            try:
                return et.parse(f)
            except et.ParseError:
                raise ValueError(f"File {filename} is not a valid Tableau workbook.")


def replace_xml_chars(xpath: str) -> str:
    char_mapping = {"&quot;": '"'}

    for old, new in char_mapping.items():
        xpath = xpath.replace(old, new)

    return xpath


def build_archive_file(archive_contents, zip_file):
    for root_dir, _, files in os.walk(archive_contents):
        relative_dir = os.path.relpath(root_dir, archive_contents)
        for f in files:
            temp_file_full_path = os.path.join(archive_contents, relative_dir, f)
            zipname = os.path.join(relative_dir, f)
            zip_file.write(temp_file_full_path, arcname=zipname)


def save_into_archive(xml_tree, filename, new_filename=None):
    if new_filename is None:
        new_filename = filename

    with tempfile.TemporaryDirectory() as temp_path:
        with zipfile.ZipFile(filename, allowZip64=True) as zf:
            twb_file = find_file_in_zip(zf)
            if not twb_file:
                raise FileNotFoundError(f"Could not find a .twb file in {filename}.")
            zf.extractall(temp_path)

        source_twb_path = Path(temp_path, twb_file)
        source_twb_path.unlink()
        xml_tree.write(
            Path(temp_path, new_filename).with_suffix(".twb"),
            encoding="utf-8",
            xml_declaration=True,
        )

        with zipfile.ZipFile(new_filename, "w", compression=zipfile.ZIP_DEFLATED, allowZip64=True) as new_archive:
            build_archive_file(temp_path, new_archive)


def find_file_in_zip(zip_file: zipfile.ZipFile) -> str | None:
    return next(file for file in zip_file.namelist() if Path(file).suffix == ".twb")
