import pathlib

from rich import print
from rich.table import Table
from rich.markup import escape
from typing_extensions import Annotated
import copy

import yte
import typer
from xmldiff import main, actions

from vm_tableau_publisher.core.models.workbook import Workbook
from vm_tableau_publisher.core.models.parse_operation import ParseOperation


cli = typer.Typer()


@cli.command()
def parse(
    source_file: Annotated[pathlib.Path, typer.Argument(exists=True)],
    operations_file: Annotated[pathlib.Path, typer.Argument(exists=True)],
    output_file: Annotated[pathlib.Path, typer.Argument()] = None,
    overwrite: Annotated[bool, typer.Option("--overwrite", "-o")] = False,
    verbose: Annotated[bool, typer.Option("--verbose", "-v")] = False,
):
    """
    Parse a Tableau file (TWB/TWBX) using operations from a JSON file.

    Examples:
        # Parse and save to target
        vm-tableau-toolkit workbook parse dashboard.twb operations.json result.twb

        # Parse and overwrite source
        vm-tableau-toolkit workbook parse -o dashboard.twb operations.json

        # Parse and overwrite target
        vm-tableau-toolkit workbook parse -o dashboard.twb operations.json result.twb
    """
    if source_file.suffix.lower() not in (".twb", ".twbx"):
        raise ValueError("Source file must be a .twb or .twbx file")

    if operations_file.suffix.lower() not in (".yaml", ".yml"):
        raise ValueError("Operations file must be a .yaml or .yml file")

    if not output_file and not overwrite:
        raise ValueError("Must specify either --output or --overwrite")

    if not overwrite and output_file.exists():
        raise ValueError(f"Output file {output_file} exists. Use --overwrite to overwrite.")

    workbook = Workbook(source_file)
    source_workbook = copy.deepcopy(workbook)
    print(f"[green] \u2713 Loaded workbook from {source_file} [/green]")

    with operations_file.open("r") as f:
        result = yte.process_yaml(f)

    operations = [ParseOperation.from_dict(operation) for operation in result["operations"]]

    print(f"[green] \u2713 Loaded {len(operations)} parse operations from {operations_file} [/green]")
    if verbose:
        print(operations)

    workbook.parse(operations)

    if verbose:
        table = Table()
        table.add_column("Element", style="bold cyan")
        table.add_column("Attribute", style="magenta")
        table.add_column("Old Value", style="white")
        table.add_column("New Value", style="white")

        diffs = main.diff_trees(
            source_workbook.root,
            workbook.root,
            diff_options={"fast_match": True},
        )
        print(diffs)
        print(f"[green] \u2713 Found {len(diffs)} difference(s) between source and parsed workbook [/green]")

        for diff in diffs:
            if isinstance(diff, actions.UpdateTextIn):
                for tag in source_workbook.find_matches(diff.node):
                    table.add_row(
                        tag.tostring(),
                        "#text",
                        escape(tag.xml.text),
                        escape(diff.text),
                    )

            elif isinstance(diff, actions.UpdateAttrib):
                for tag in source_workbook.find_matches(diff.node):
                    table.add_row(
                        tag.tostring(),
                        diff.name,
                        escape(tag.xml.attrib.get(diff.name)),
                        escape(diff.value),
                    )

        print(table)

    if not output_file:
        output_file = source_file

    workbook.save(output_file)
    print(f"[green] \u2713 Saved parsed workbook to {output_file.resolve()} [/green]")
