import tempfile
import pathlib

import prefect

from vm_mssql_client.base_client import DatabaseClient

from vm_tableau_publisher.core.models.workbook import Workbook
from vm_tableau_publisher.core.models.tableau_site import TableauSite
from vm_tableau_publisher.core.models.parse_operation import ParseOperation
from vm_tableau_publisher.core.models.connection import Connection

from models.publication import Publication
from helpers import validate_customer_name, validate_datatypes


@prefect.task(retries=1, retry_delay_seconds=5)
def get_parse_operations(publication: Publication, database_client: DatabaseClient) -> list[ParseOperation]:
    arguments = {
        "product_tableau_id": publication.product.id,
        "klant": publication.customer.name,
        "stage": publication.config.stage,
    }

    if publication.product.id in (5, 6, 11, 12):
        if publication.settings.snapshot_id:
            arguments["lwi_setting_snapshotid"] = publication.settings.snapshot_id

        if publication.settings.version:
            arguments["lwi_setting_versionid"] = publication.settings.version

        if publication.settings.taxateur:
            arguments["lwi_setting_taxateur"] = publication.settings.taxateur

    result = database_client.execute_stored_procedure(name="p_get_publisher_parse_steps", arguments=arguments)

    return [ParseOperation.from_dict(i) for i in result]


@prefect.task(retries=1, retry_delay_seconds=30)
def publish_workbook(workbook: Workbook, file: pathlib.Path, site: TableauSite, connections: list[Connection]) -> None:
    site.publish_workbook(
        workbook=workbook,
        file=file,
        connections=connections,
    )


@prefect.task
def parse_workbook(workbook: Workbook, operations: list[ParseOperation]) -> None:
    workbook.parse(operations)


@prefect.task(retries=1, retry_delay_seconds=30)
def refresh_workbook(site: TableauSite, workbook: Workbook) -> None:
    site.refresh_workbook(workbook)


@prefect.task(retries=1, retry_delay_seconds=30)
def get_workbook(tableau_site: TableauSite, workbook_name: str, project_name: str) -> Workbook:
    return tableau_site.get_workbook(workbook_name, project_name)


@prefect.task(retries=1, retry_delay_seconds=30)
def get_project(tableau_site: TableauSite, project_name: str) -> None:
    return tableau_site.get_project(project_name)


@prefect.task
def validate_workbook_parameters(publication: Publication, workbook: Workbook) -> None:
    validate_customer_name(workbook, publication.customer.name)
    validate_datatypes(workbook)
