import os
from typing import Optional
import functools
import concurrent.futures
from pathlib import Path


import yaml
from dotenv import load_dotenv
from fastapi import APIRouter, File, Form, UploadFile, HTTPException, Depends
from fastapi.responses import FileResponse
from pydantic import BaseModel

from vm_mssql_client.base_client import DatabaseClient

from vm_tableau_publisher.core.models.workbook import Workbook

from models.brevo_client import BrevoClient, render_template
from models.publication import Payload, Publication
from flows import process_publication
from helpers import get_database_client


load_dotenv()

router = APIRouter()

product_mapping = yaml.safe_load(open("artifacts/mappings/product_mapping.yml"))
workbook_views = yaml.safe_load(open("artifacts/mappings/workbook_views.yml"))


class WorkbookInput(BaseModel):
    file: str
    product: str
    stage: str
    product_tableau: str


@router.get("/all")
def get_workbooks(
    database_client: DatabaseClient = Depends(functools.partial(get_database_client, database="vm_publisher")),
):
    files_db = database_client.execute_stored_procedure("p_get_tableau_workbooks", return_results=True)

    mapped_files = {}

    for file in files_db:
        if file[0] not in mapped_files.keys():
            mapped_files[file[0]] = {
                "Product_id": file[1],
                "File_path": file[4],
                "Product": file[2],
                "ProductTableau": file[3],
                "Stage": file[5],
                "Uploaded_on": file[7],
                "Uploaded_by": file[6],
                "views": [{"view": file[8], "show": file[8] in workbook_views[file[1]]}],
            }
        else:
            mapped_files[file[0]]["views"].append({"view": file[8], "show": file[8] in workbook_views[file[1]]})

    return {"files": [file for file in mapped_files.values()]}


@router.get("/download")
def download_workbook(product: str, workbook: str):
    path = Path(os.environ["FILE_DIR"], product_mapping[product.lower()], workbook)

    if not path.exists():
        return

    return FileResponse(path=path, filename=workbook)


@router.put("/delete")
def delete_workbook(
    file: WorkbookInput,
    database_client: DatabaseClient = Depends(functools.partial(get_database_client, database="vm_publisher")),
):
    product = file.product
    product_tableau = file.product_tableau
    file = file.file

    path = Path(os.environ["FILE_DIR"], product_mapping[product.lower()], file)

    try:
        path.unlink()
    except FileNotFoundError:
        raise HTTPException(status_code=404, detail="File not found")

    database_client.execute_stored_procedure(
        procedure_name="p_delete_tableau_workbook",
        arguments={
            "File_path": file,
            "Product": product,
            "Product_Tableau": product_tableau,
        },
        return_results=False,
    )


@router.post("/upload")
def upload_workbook(
    product: str = Form(...),
    productTableau: Optional[str] = Form(""),
    stage: str = Form(...),
    user: str = Form(...),
    file: UploadFile = File(...),
    database_client: DatabaseClient = Depends(functools.partial(get_database_client, database="vm_publisher")),
):
    if not file.filename.endswith(("twb", "twbx")):
        return {"Only .twb or .twbx files are allowed!"}

    path = Path(os.environ["FILE_DIR"], product_mapping[product.lower()], file.filename)

    if os.path.exists(path):
        database_client.execute_stored_procedure(
            procedure_name="p_delete_tableau_workbook",
            arguments={
                "File_path": file.filename,
                "Product": product,
                "Product_Tableau": productTableau,
            },
            return_results=False,
        )

    run_id = database_client.execute_stored_procedure(
        procedure_name="p_insert_workbook_metadata",
        arguments={
            "Product": product,
            "ProductTableau": productTableau,
            "File_path": file.filename,
            "Stage": stage,
            "Uploaded_by": user,
        },
        return_results=True,
    ).fetchone()[0]

    try:
        with path.open("wb") as f:
            f.write(file.file.read())

        workbook = Workbook(path)
        for view in workbook.extract_views():
            view = view.replace("'", "''")
            database_client.execute_query(
                f"""
                INSERT INTO tableau_workbook_views VALUES (
                    '{run_id}', '{view}', '1'
                )
                """
            )

    except Exception as e:
        print(e)


@router.post("/publish")
def publish_workbooks(
    publications: Payload,
    database_client: DatabaseClient = Depends(functools.partial(get_database_client, database="vm_publisher")),
):
    product = publications.payload[0]["product"]
    user_email = publications.payload[0]["config"]["publisher"]

    results = execute_publications(publications=publications.payload, database_client=database_client)
    success_count = sum(1 for result in results.values() if result.get("status") == "success")

    send_notification_email(
        results=results,
        product_name=product["name"],
        total_count=len(publications.payload),
        success_count=success_count,
        user_email=user_email,
    )

    if success_count < len(publications.payload):
        raise HTTPException(
            status_code=500,
            detail=f"Published {success_count}/{len(publications.payload)} workbooks successfully",
        )

    return {
        "status": "success",
        "total": len(publications.payload),
        "successful": success_count,
        "results": results,
    }


def execute_publications(publications: list, database_client: DatabaseClient) -> dict[str, dict]:
    """Execute publications concurrently using ProcessPoolExecutor."""
    results = {}

    with concurrent.futures.ProcessPoolExecutor(max_workers=2) as executor:
        futures = {}
        for publication in publications:
            future = executor.submit(process_publication, publication)
            futures[publication["customer"]["name"]] = {"publication": publication, "future": future}

        for customer_name, future_info in futures.items():
            try:
                task_result = future_info["future"].result()
                results[customer_name] = task_result
                insert_publication_log(publication=future_info["publication"], database_client=database_client)
            except Exception as e:
                results[customer_name] = {
                    "status": "error",
                    "error": str(e),
                }

    return results


def insert_publication_log(publication: Publication, database_client: DatabaseClient):
    database_client.execute_stored_procedure(
        procedure_name="p_insert_publication_log",
        arguments={
            "Site": publication.customer.site,
            "Product": publication.product.name,
            "Filename": publication.workbook.source_file.name,
            "Published_by": publication.config.publisher,
            "Klant": publication.customer.name,
            "Workbook_name": publication.workbook.name.replace("'", "''"),
            "Stage": publication.config.stage,
            "Snapshot": publication.settings.snapshot_id,
            "Version": publication.settings.version,
            "TableauServer": publication.config.server_url,
            "Url": publication.workbook.url,
        },
        return_results=False,
    )


def send_notification_email(
    results: dict[str, dict],
    product_name: str,
    total_count: int,
    success_count: int,
    publisher_email: str,
) -> None:
    if success_count == total_count:
        subject = f"Successfully published {product_name} for {total_count} customers"
    else:
        failed_count = total_count - success_count
        subject = f"Published {product_name}: {success_count} succeeded, {failed_count} failed"

    recipients = [{"email": publisher_email}]
    html = render_template(template_name="mail_template.html", publications=results)

    BrevoClient().send_email(subject=subject, recipients=recipients, html=html)
