import functools
from collections import defaultdict

from fastapi import APIRouter, Depends

from vm_mssql_client.base_client import DatabaseClient

from helpers import parse_product_mapping, get_database_client


router = APIRouter()


@router.get("/health")
def read_health(
    database_client: DatabaseClient = Depends(
        functools.partial(get_database_client, backend="sqlalchemy", database="vm_publisher")
    ),
):
    conn = database_client.engine.connect()
    conn.close()


@router.get("/workbook_name_templates")
def get_workbook_name_templates(
    database_client: DatabaseClient = Depends(functools.partial(get_database_client, database="vm_publisher")),
):
    response = database_client.execute_query("SELECT * FROM tableau_workbook_name_templates", return_results=True)

    workbook_name_templates = {}

    for i in response:
        if i[1] in ("klantomgeving", "amwoco") and i[2] == "development":
            continue

        if i[0] not in workbook_name_templates:
            workbook_name_templates[i[0]] = defaultdict(dict)

        workbook_name_templates[i[0]][i[1]][i[2]] = i[3]

    return workbook_name_templates


@router.get("/tableau_sites")
def get_available_sites(
    database_client: DatabaseClient = Depends(functools.partial(get_database_client, database="vm_admin")),
):
    response = database_client.execute_query(
        """
        SELECT DISTINCT url_namespace
        FROM [VM_Admin].[dbo].[tableau_sites]
        WHERE url_namespace != ''
        """,
        return_results=True,
    )

    return [x[0] for x in response]


@router.get("/products")
def get_available_products(
    database_client: DatabaseClient = Depends(functools.partial(get_database_client, database="vm_publisher")),
):
    response = database_client.execute_query(
        """
        SELECT * FROM v_product_mapping
        """,
        return_results=True,
    )

    product_mapping = parse_product_mapping(response)

    return [product for product in product_mapping if product["available"] == 1]


@router.get("/publications")
def recent_publications(
    database_client: DatabaseClient = Depends(functools.partial(get_database_client, database="vm_publisher")),
):
    response = database_client.execute_query(
        "SELECT * FROM tableau_publications ORDER BY Published_on DESC", return_results=True
    )
    return {"publications": [[x for x in row] for row in response]}


@router.get("/customers")
def get_subscribed_customers(
    product: int,
    database_client: DatabaseClient = Depends(functools.partial(get_database_client, database="vm_publisher")),
):
    response = database_client.execute_stored_procedure(
        procedure_name="p_get_clients_publisher",
        arguments={
            "product_id": product,
        },
        return_results=True,
    )
    return {"customers": [[x for x in row] for row in response]}
