import os
import re
from pathlib import Path
from typing import Optional
from datetime import date
import pydantic
import yaml
from dotenv import load_dotenv

from vm_mssql_client.base_client import DatabaseClient


load_dotenv()


class Product(pydantic.BaseModel):
    id: int
    name: str
    project_name: Optional[str] = None
    abbreviation: Optional[str] = None

    @pydantic.model_validator(mode="after")
    def get_abbreviation(cls, values):
        with Path("mappings/product_abbreviations.yaml").open() as f:
            abbreviations = yaml.safe_load(f)

        values.abbreviation = abbreviations[values.id]

        return values


class Customer(pydantic.BaseModel):
    name: str
    site: str


class Payload(pydantic.BaseModel):
    payload: list


class Config(pydantic.BaseModel):
    environment: str
    stage: str
    server: str
    publisher: str
    server_url: Optional[str] = None

    @pydantic.model_validator(mode="after")
    def convert_server_name(cls, values):
        server_mapping = {
            "production": "https://portal.valuemetrics.nl",
            "development": "https://tableau-develop.vmetrics-online.nl",
        }

        values.server_url = server_mapping[values.server.lower()]
        return values


class Settings(pydantic.BaseModel):
    version: Optional[int | str] = None
    snapshot: Optional[str] = None
    snapshot_id: Optional[int] = None
    taxateur: Optional[str] = None

    @pydantic.model_validator(mode="after")
    def convert_snapshot(cls, values):
        if "snapshot_id" not in values or not values.snapshot_id:
            return values

        database = DatabaseClient("WW_DWH")
        result = database.execute_query(
            f"SELECT Snapshot_omschrijving FROM m_snapshot WHERE SnapshotID = {values.snapshot_id}"
        )
        if not result:
            raise ValueError(f"Snapshot {values.snapshot_id} not found in [WW_DWH].[dbo].[m_snapshot]!")

        values.snapshot = result[0]["snapshot_omschrijving"]
        return values


class Workbook(pydantic.BaseModel):
    name: str
    stage: str
    manual_view_selection: bool
    views: list[str]
    source_file: str | Path
    url: Optional[str] = None
    description: Optional[str] = None


class Publication(pydantic.BaseModel):
    customer: Customer
    product: Product
    settings: Settings
    workbook: Workbook
    config: Config

    @pydantic.model_validator(mode="after")
    def convert_workbook_name(cls, values):
        if values.workbook.name:
            return values

        product_tableau = int(values.product.id)
        environment = values.config.environment.lower()
        stage = values.config.stage.lower()
        workbook_stage = values.workbook.stage.lower()

        if stage == "development (localhost)":
            stage = "development"

        if product_tableau == 6 and environment == "klantomgeving":
            values.settings.snapshot = re.findall(r"\d{4}", values.settings.snapshot)[0]

        workbook_name_mapping = yaml.safe_load(
            Path("../packages/vm_tableau_publisher/src/mappings/workbook_name_mapping.yml").open()
        )

        if values.config.server.lower() == "development":
            workbook_name_mapping = workbook_name_mapping["development"]
            workbook_name_template = workbook_name_mapping[workbook_stage][stage]
        else:
            workbook_name_mapping = workbook_name_mapping["production"]
            workbook_name_template = workbook_name_mapping[product_tableau][environment][stage]

        _values = values.dict()
        _values["date"] = date.today().strftime("%y%m%d")

        values.workbook.name = workbook_name_template.format(**_values)

        return values

    @pydantic.model_validator(mode="before")
    def convert_view_selection(cls, values):
        if values["workbook"]["manual_view_selection"]:
            return values

        database = DatabaseClient("VM_Publisher")
        result = database.execute_stored_procedure("p_get_tableau_workbooks")
        all_views = set([file["view"] for file in result if file["file_path"] == values["workbook"]["source_file"]])

        with Path("mappings/workbook_views.yml").open() as f:
            views_to_keep = yaml.safe_load(f)[values["product"]["id"]]

        views_to_hide = list(set(all_views) - set(views_to_keep))

        values["workbook"]["views"] = views_to_hide

        return values

    @pydantic.model_validator(mode="after")
    def convert_source_file(cls, values):
        product_mapping = {
            "benchmark marktwaarde won": "bmmw won",
            "benchmark marktwaarde bog": "bmmw bog",
            "lwi": "lwi",
            "wwi": "wwi",
            "toolbox rendement": "tbrm",
            "toolbox marktwaarde": "tbmw",
            "toolbox": "tbmw",
            "benchmark rendement": "bmrm",
            "platform": "platform",
            "benchmark marktwaarde bog/mog/zog/ppl": "bmmw bog",
        }

        path = Path(
            os.getenv("FILE_DIR"),
            product_mapping[values.product.name.lower()],
            values.workbook.source_file,
        )

        if not path.exists():
            raise ValueError(f"File {path} does not exist!")

        values.workbook.source_file = path

        return values

    @pydantic.model_validator(mode="after")
    def convert_customer_site(cls, values):
        environment = values.config.environment.lower()

        if values.settings.taxateur and environment == "klantomgeving":
            with Path("mappings/taxateur_mapping.yml").open() as f:
                taxateur_mapping = yaml.safe_load(f)
            values.customer.site = taxateur_mapping[values.settings.taxateur.lower()]
        elif environment == "amwoco":
            values.customer.site = "Amwoco"
        elif environment == "valuemetrics intern":
            values.customer.site = "VM_intern"

        return values

    @pydantic.model_validator(mode="after")
    def get_project_name(cls, values):
        with Path("mappings/project_mapping.yml").open() as f:
            project_mapping = yaml.safe_load(f)

        environment = values.config.environment.lower()
        product_name = values.product.name.lower()

        values.product.project_name = project_mapping[environment][product_name]

        return values
