import os
import yaml
import pathlib
import re

from prefect.blocks.system import Secret

from vm_mssql_client.base_client import DatabaseClient
from vm_mssql_client import create_client
from vm_mssql_client.database_credentials import DatabaseCredentials


def get_database_client(database: str, backend: str = "pyodbc") -> DatabaseClient:
    database_credentials_block = Secret.load(os.environ["DB_CREDENTIALS_BLOCK_SLUG"]).get()
    return create_client(
        backend=backend,
        database=database,
        credentials=DatabaseCredentials(**database_credentials_block),
    )


def get_password(server: str, username: str) -> str:
    path = pathlib.Path("artifacts/mappings/connection_credentials.yaml")

    if not path.exists():
        raise FileNotFoundError(f"Connection credentials file not found at {path.absolute()}")

    credentials = yaml.safe_load(path.open())

    try:
        return os.environ[credentials[server][username]]
    except KeyError:
        raise LookupError(f"Credentials for {username} on {server} not found in {path.absolute()}")


def validate_customer_name(workbook, customer_name: str) -> None:
    customer_captions = [
        "** Kies klant",
        "Kies klant",
        "** Laad data voor klant **",
        "** Laad Data Voor Klant **",
    ]

    try:
        for caption in customer_captions:
            assert all(
                i.attrib.get("value").lower() == '"%s"' % customer_name.lower()
                for i in workbook.tree.xpath(f"//column[@caption='{caption}']")
            )
            assert all(
                i.attrib.get("formula").lower() == "%s" % customer_name.lower()
                for i in workbook.tree.xpath(f"//column[@caption='{caption}']/calculation")
            )
            assert all(
                i.attrib.get("value").lower() == '"%s"' % customer_name.lower()
                for i in workbook.tree.xpath(f"//column[@caption='{caption}']/members/member")
            )
    except AssertionError:
        raise ValueError(f"Not all customer parameters in workbook match the customer name '{customer_name}'")


def validate_datatypes(workbook) -> None:
    matches = [x.attrib["value"] for x in workbook.tree.xpath("//*[@datatype='string' and @value]")]

    for match in matches:
        if not re.match(r'"?(.+)"', match):
            raise ValueError(f"Value {match} is not enclosed in quotes!")


def parse_product_mapping(response: list) -> list:
    product_mapping = []

    for x in response:
        if x[0] in [i["product_id"] for i in product_mapping]:
            index = next(
                (index for (index, d) in enumerate(product_mapping) if d["product_id"] == x[0]),
                None,
            )
            product_mapping[index]["product_tableau"].append({"id": x[1], "name": x[2]})
        else:
            product_tableau = [{"id": x[1], "name": x[2]}] if x[2] else []
            product_mapping.append(
                {
                    "name": x[3],
                    "product_id": x[0],
                    "id": x[1],
                    "available": x[4],
                    "product_tableau": product_tableau,
                }
            )

    return product_mapping
