import os
import tempfile
import urllib
import pathlib

import httpx
import prefect
from prefect.blocks.system import Secret

from vm_mssql_client import create_client
from vm_mssql_client.database_credentials import DatabaseCredentials

from vm_tableau_publisher.core.models.workbook import Workbook
from vm_tableau_publisher.core.models.tableau_site import TableauSite
from vm_tableau_publisher.core.helpers.publish import (
    create_data_connection,
    create_tableau_workbook,
    format_workbook_url,
)

from models.publication import Publication
from tasks import get_parse_operations, publish_workbook, parse_workbook, refresh_workbook, get_workbook, get_project
from helpers import get_password


def trigger_slack_workflow(flow, flow_run, state) -> None:
    webhook_url = prefect.variables.Variable.get("alerts_prefect_webhook_url")
    server_url = urllib.parse.urlparse(str(prefect.settings.PREFECT_API_URL))

    response = httpx.post(
        webhook_url,
        json={
            "run_name": flow_run.name,
            "prefect_url": f"{server_url.scheme}://{server_url.netloc}/flow-runs/flow-run/{flow_run.id}",
            "dataflow": flow.name,
            "state": state.name,
        },
    )
    response.raise_for_status()


@prefect.flow(
    name="vm-tableau-publisher", flow_run_name="{publication.workbook.name}", on_failure=[trigger_slack_workflow]
)
def process_publication(publication: Publication):
    logger = prefect.get_run_logger()

    database_credentials_block = Secret.load(os.environ["DB_CREDENTIALS_BLOCK_SLUG"]).get()
    database_client = create_client(
        backend="pyodbc",
        database="VM_Publisher",
        credentials=DatabaseCredentials(**database_credentials_block),
    )

    parse_operations = get_parse_operations(publication, database_client)
    logger.info(f"Retrieved {len(parse_operations)} parse operations: {parse_operations}")

    workbook = Workbook(file=publication.workbook.source_file)
    parse_workbook(workbook=workbook, operations=parse_operations)

    extracted_connections = workbook.extract_connections()
    logger.info(f"Extracted {len(extracted_connections)} connections: {extracted_connections}")

    connections = [
        create_data_connection(
            server=connection.server,
            username=connection.username,
            password=get_password(connection.server, connection.username),
        )
        for connection in extracted_connections
    ]

    tableau_site = TableauSite(publication.config.server_url, publication.customer.site)
    logger.info(f"Signed in to {tableau_site}")

    project = get_project(tableau_site, publication.product.project_name)

    tableau_workbook = create_tableau_workbook(
        product_id=publication.product.id,
        project_id=project.id,
        workbook_name=publication.workbook.name,
        views_to_hide=publication.workbook.views,
    )

    if project.id in (11, 12):
        tableau_workbook.show_tabs = False

    with tempfile.NamedTemporaryFile(suffix=workbook.suffix) as file:
        workbook.save(file)
        publish_workbook(
            workbook=tableau_workbook, file=pathlib.Path(file.name), site=tableau_site, connections=connections
        )

    logger.info(f"Published workbook {workbook} to {tableau_site}")

    published_workbook = get_workbook(
        tableau_site,
        publication.product.project_name,
        publication.workbook.name,
    )

    publication.workbook.url = format_workbook_url(
        publication.config.server_url,
        published_workbook.webpage_url,
        publication.customer.site,
    )
    logger.info(f"Published workbook '{publication.workbook.name}' to {publication.workbook.url}")

    if publication.product.id in (1, 3, 4, 11, 12):
        refresh_workbook(workbook=publish_workbook, site=tableau_site)
        logger.info(f"Refreshed workbook {publication.workbook.name}")
