import time
import logging
import random

import httpx

from .client import Client
from .models.client_manager_config import ClientManagerConfig

logger = logging.getLogger(__name__)


class ClientValidator:
    def __init__(self, config: ClientManagerConfig):
        self.config = config

    def validate(self, client: Client) -> bool:
        """
        Validate the client configuration.

        Returns:
            bool: Boolean indicating success.
        """

        for attempt in range(self.config.validation_retries + 1):
            try:
                start_time = time.time()
                response = client.get(url=self.config.validation_url)
                response_time = time.time() - start_time

                if response.status_code == 200:
                    logger.info(f"Client is valid. Response time: {response_time:.2f} seconds")
                    return True

            except Exception as e:
                logger.info(f"Client validation attempt {attempt + 1} failed: {e}")
                if attempt < self.config.validation_retries:
                    self._rotate_user_agent(client)
                    time.sleep(self.config.validation_retry_delay)
                    continue

        logger.warning(f"Client failed validation after {self.config.validation_retries + 1} attempts")
        return False

    def _rotate_user_agent(self, client: httpx.Client) -> None:
        """
        Rotate the user agent for the client to avoid detection.
        """
        if self.config.default_user_agents:
            new_user_agent = random.choice(self.config.default_user_agents)
            client.headers["User-Agent"] = new_user_agent
            logger.info(f"Rotated 'User-Agent' to: {new_user_agent}")
