import threading
import logging

from .client import Client
from .exceptions import ClientPoolEmptyError
from .models.client_manager_config import ClientManagerConfig
from .models.client_status import ClientStatus

logger = logging.getLogger(__name__)


class ClientPool:
    def __init__(self, config: ClientManagerConfig) -> None:
        self.config = config
        self._lock = threading.Lock()
        self._active_clients: list[Client] = []
        self._current_index: int = 0

    def add_client(self, client: Client) -> None:
        with self._lock:
            self._active_clients.append(client)

            logger.info(f"Client added to active clients. Total active: {len(self._active_clients)}")

    def get_next_client(self) -> Client:
        with self._lock:
            if not self._active_clients:
                raise ClientPoolEmptyError

            client = self._active_clients[self._current_index]

            self._current_index = (self._current_index + 1) % len(self._active_clients)

            return client

    def remove_client(self, client: Client) -> None:
        with self._lock:
            if client in self._active_clients:
                self._active_clients.remove(client)

            client.status = ClientStatus.INACTIVE

            self._current_index = self._current_index % len(self._active_clients)

            logger.info(f"Client removed. Total active: {len(self._active_clients)}")

    def get_all_clients(self) -> list[Client]:
        with self._lock:
            return self._active_clients.copy()
