from typing import Optional
import logging
import random

from .client import Client
from .client_health_monitor import ClientHealthMonitor
from .client_pool import ClientPool
from .client_validator import ClientValidator
from .exceptions import ClientManagerError
from .models.client_manager_config import ClientManagerConfig

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


class ClientManager:
    def __init__(self, clients: list[Client], config: Optional[ClientManagerConfig] = None) -> None:
        self.config = config or ClientManagerConfig()
        self.pool = ClientPool(self.config)

        self.validator = ClientValidator(self.config)
        self._initialize_clients(clients)

        if self.config.validate_clients:
            self.monitor = ClientHealthMonitor(config=self.config, pool=self.pool, validator=self.validator)
            self.monitor.start_monitoring()

    def _initialize_clients(self, clients: list[Client]) -> None:
        n_valid_clients = 0

        for client in clients:
            if self.config.default_user_agents:
                client.headers.update({"user-agent": random.choice(self.config.default_user_agents)})

            try:
                if self.config.validate_clients:
                    is_valid = self.validator.validate(client=client)

                    if not is_valid:
                        logger.error(f"Client {client} is not valid, skipping.")
                        continue

                self.pool.add_client(client=client)
                n_valid_clients += 1

            except Exception as e:
                logger.error(f"Error validating client: {e}")

        if n_valid_clients < self.config.min_active_clients:
            raise ClientManagerError(
                f"Not enough valid clients ({n_valid_clients}) to meet minimum requirement "
                f"({self.config.min_active_clients})"
            )

    def get_client(self) -> Client:
        return self.pool.get_next_client()

    def shutdown(self) -> None:
        """
        Shutdown the client manager, stopping all monitoring and cleaning up resources.
        """
        logger.info("Shutting down client manager")
        self.monitor.stop_monitoring()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        self.shutdown()
