import threading
import logging
import time

from .client_validator import ClientValidator
from .client_pool import ClientPool
from .client import Client
from .models.client_manager_config import ClientManagerConfig

logger = logging.getLogger(__name__)


class ClientHealthMonitor:
    def __init__(self, config: ClientManagerConfig, pool: ClientPool, validator: ClientValidator) -> None:
        self.config = config
        self.pool = pool
        self.validator = validator
        self._monitor_thread = self.start_monitoring()

    def start_monitoring(self) -> None:
        monitor_thread = threading.Thread(target=self._monitor_loop, daemon=True)

        monitor_thread.start()
        logger.info("Health monitoring started")

        return monitor_thread

    def stop_monitoring(self) -> None:
        if self._monitor_thread and self._monitor_thread.is_alive():
            self._monitor_thread.join(timeout=1)

        logger.info("Health monitoring stopped")

    def _monitor_loop(self) -> None:
        while True:
            try:
                self._check_all_clients()
                time.sleep(self.config.health_check_interval)
            except Exception as e:
                logger.error(f"Error during health monitoring: {e}")

    def _check_all_clients(self) -> None:
        clients = self.pool.get_all_clients()

        logger.debug(f"Starting health check for {len(clients)} clients")

        for client in clients:
            try:
                is_healthy = self._check_client_health(client)

                if not is_healthy:
                    logger.warning(f"Client {client} is unhealthy, removing from pool")
                    self.pool.remove_client(client)

            except Exception as e:
                logger.error(f"Error checking client health: {e}")
                self.pool.remove_client(client)

    def _check_client_health(self, client: Client) -> bool:
        return self.validator.validate(client=client)
