import logging
from typing import Optional, Any

import httpx

from .models.client_status import ClientStatus
from .models.client_config import ClientConfig

logger = logging.getLogger(__name__)

RETRYABLE_STATUS_CODES: list[int] = [200, 408, 429, 500, 502, 503, 504]


class Client(httpx.Client):
    def __init__(self, config: Optional[ClientConfig] = None, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.config = config or ClientConfig()

        self.status: ClientStatus = ClientStatus.ACTIVE
        self.last_health_check: int = 0
        self.consecutive_failures: int = 0
        self.total_requests: int = 0
        self.successful_requests: int = 0

    def request(
        self,
        method: str,
        url: str,
        *,
        content: Optional[bytes] = None,
        data: Optional[dict[str, Any]] = None,
        files: Optional[dict[str, Any]] = None,
        json: Optional[Any] = None,
        params: Optional[dict[str, Any]] = None,
        headers: Optional[dict[str, str]] = None,
        cookies: Optional[dict[str, str]] = None,
        auth: Optional[Any] = None,
        follow_redirects: Optional[bool] = None,
        timeout: Optional[float] = None,
        extensions: Optional[dict[str, Any]] = None,
        **kwargs,
    ) -> httpx.Response:
        for attempt in range(self.config.max_retries + 1):
            try:
                response = super().request(
                    method=method,
                    url=url,
                    content=content,
                    data=data,
                    files=files,
                    json=json,
                    params=params,
                    headers=headers,
                    cookies=cookies,
                    auth=auth,
                    follow_redirects=follow_redirects,
                    timeout=timeout,
                    extensions=extensions,
                    **kwargs,
                )

                response.raise_for_status()

                self.successful_requests += 1
                self.consecutive_failures = 0

                return response

            except httpx.HTTPStatusError as e:
                logger.error(f"Status code error on attempt {attempt + 1}: {e}")
                self.consecutive_failures += 1

                if e.response.status_code in RETRYABLE_STATUS_CODES and attempt < self.config.max_retries:
                    continue

                raise

            except httpx.RequestError as e:
                logger.warning(f"Request error on attempt {attempt + 1}: {e}")
                self.consecutive_failures += 1

                if attempt < self.config.max_retries:
                    continue

                raise

            finally:
                self.total_requests += 1

    @property
    def is_active(self) -> bool:
        return self.status == ClientStatus.ACTIVE

    @property
    def success_rate(self) -> float:
        if self.total_requests == 0:
            return 0.0

        return self.successful_requests / self.total_requests
